/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.entitystream.identiza.entity.resolve.metadata.IIndex;

public class CollectionRemote implements ICollection, Serializable{

    private MonsterClient client=null;

    private String dbName;
    private String collName;
    private int nodeNum;

    private ConcurrentHashMap<Long, MonsterClient> threads = new ConcurrentHashMap();

    public MonsterClient getClient() {
	long threadid = Thread.currentThread().getId() % 50;
	MonsterClient client=null;
	try {
	if (threads.containsKey(threadid)) {
	    client= getClientInternal(threads.get(threadid));
	} else {
	    MonsterClient clientinn= getClientInternal(this.client.clone());
	    threads.put(threadid, clientinn);

	    client= clientinn;
	}
	client.setRemoteView(true);
	} catch (Exception e) {
	    System.out.println("Remote Server is Offline, results will be affected");
	}
	return client;

    }

    private MonsterClient getClientInternal(MonsterClient client) {
	try {
	    if (!client.isConnected()) {

		if (!client.reconnect())
		    throw new Exception("replica is down, command is rejected");

	    }
	    client.database=dbName;
	    client.collection=collName;
	    return client;
	} catch (Exception e) {
	    System.out.println("Replica failed to come online, command is rejected " +e.toString());
	    return null;
	}
    }

    public CollectionRemote(MonsterClient connection, String dbName , Document collDoc, boolean initialised)  {
	System.out.println("Opening connection to: "+connection.getNodeNum() + " for collection " + collDoc.getString("Name"));
	nodeNum=connection.getNodeNum();
	
	if (!collDoc.getBoolean("remoteCall", false)) {
	    connection.createDatabase(dbName); //create or use
	    
	    System.out.println("Sending Collection description:"+collDoc.toString());
	    
	    connection.createCollection(collDoc.append("remoteCall", true));
	} 

	this.dbName=dbName;
	this.collName=collDoc.getString("Name");
	this.client=connection;
	this.client.setRemoteView(true);

	MonsterClient client = getClient();
	if (client!=null) {
	    client.useDatabase(dbName);
	    client.useCollection(collName);
	}
    }

    @Override
    public void createIndex(Document fields, Document options) {

	MonsterClient client = getClient();
	if (client!=null) client.createIndex(fields, options);

    }

    @Override
    public void createIndex(Document fields) {
	MonsterClient client = getClient();
	if (client!=null) client.createIndex(fields);

    }

    @Override
    public void createUniqueIndex(Document fields) {
	MonsterClient client = getClient();
	if (client!=null) client.createIndex(fields);

    }

    @Override
    public DBCursor find(Document filter) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.find(filter);
	else return new DBCursor();
    }

    @Override
    public DBCursor find() {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.find();
	else return new DBCursor();
    }

    @Override
    public DBCursor findFuzzy(Document filter) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findFuzzy(filter);
	else return new DBCursor();
    }

    @Override
    public DBCursor findFuzzy(String textQuery) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findFuzzy(textQuery);
	else return new DBCursor();
    }


    @Override
    public Document save(Document doc) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.save(doc);
	else return null;
    }

    @Override
    public Document insertOne(Document doc) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.insertOne(doc);
	else return null;
    }

    @Override
    public long count(Document query) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.count(query);
	else return 0;
    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement, Document options) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findOneAndReplace(filter, replacement, options);
	else return null;
    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findOneAndReplace(filter, replacement);
	else return null;
    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findOneAndUpdate(filter, amendments);
	else return null;
    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments, Document updateOpts) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findOneAndUpdate(filter, amendments, updateOpts);
	else return null;
    }

    @Override
    public Document updateOne(Document filter, Document amendments) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateOne(filter, amendments);
	else return null;
    }

    @Override
    public Document updateOne(Document filter, Document amendments, Document options) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateOne(filter, amendments, options);
	else return null;
    }

    @Override
    public int updateMany(Document filter, Document amendments, Document options) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateMany(filter, amendments, options);
	else return 0;
    }

    @Override
    public int updateMany(Document filter, Document amendments) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateMany(filter, amendments, new Document());
	else return 0;
    }

    @Override
    public int insertMany(List<Document> listRecs) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.insertMany(listRecs);
	else return 0;
    }

    @Override
    public Document findOneAndDelete(Document filter) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findOneAndDelete(filter);
	else return null;
    }

    @Override
    public Document deleteOne(Document filter) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.deleteOne(filter);
	else return null;
    }

    @Override
    public int deleteMany(Document filter) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.deleteMany(filter);
	else return 0;
    }

    @Override
    public Stream<Document> aggregate(List<Document> pipeline, Document options) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregate(pipeline,options);
	else return Stream.empty();
    }

    @Override
    public Stream<Document> aggregate(List<Document> pipeline) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregate(pipeline);
	else return Stream.empty();
    }

    @Override
    public Stream<Document> aggregate(List<Document> in, List<Document> pipeline) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregate(in, pipeline);
	else return Stream.empty();
    }

    @Override
    public Document aggregateMetadata(List<Document> in, List<Document> pipeline) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregateMetadata(in, pipeline);
	else return new Document();
    }

    @Override
    public Document aggregateMetadata(List<Document> pipeline) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregateMetadata(pipeline);
	else return new Document();
    }
    
    @Override
    public DBCursor listIndexes() {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.listIndexes();
	else return new DBCursor();
    }



    @Override
    public void rebuildIndex(String name) {
	MonsterClient client = getClient();
	if (client!=null) client.rebuildIndex(name);

    }

    @Override
    public void dropIndex(String name) {

	MonsterClient client = getClient();
	if (client!=null) client.dropIndex(name);
    }

    @Override
    public int saveMany(List<Document> records) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.saveMany(records);
	else return 0;
    }

    @Override
    public void disconnect() throws Exception {

	MonsterClient client = getClient();
	if (client!=null) client.disconnect();
    }

    @Override
    public void setExplaining(boolean explain) {

	MonsterClient client = getClient();
	if (client!=null) client.setExplaining(explain);
    }

    @Override
    public DBCursor find(Document query, Document explain) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.find(query, explain);
	else return new DBCursor();
    }

    @Override
    public synchronized DBCursor executeCommand(String command, User user, Session session) {

	MonsterClient client = getClient();
	if (client!=null) {

	    DBCursor ret = client.executeCommand(command, user,session);

	    return ret;
	}
	else return new DBCursor();
    }

    @Override
    public void drop() {
	MonsterClient client = getClient();
	if (client!=null) client.drop();

    }

    @Override
    public <TResult> Iterable<TResult> distinct(String fieldName, Class<TResult> resultClass) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.distinct(fieldName, resultClass);
	else return null;
    }


    @Override
    public Document getDocument(Object pointer) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getDocument(pointer);
	else return null;
    }

    @Override
    public Document getIndexPrefixSubMap(String indexName, String key, boolean b) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getIndexPrefixSubMap(indexName, key, b);
	else return null;

    }

    @Override
    public Document keyCount(String indexName) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.keyCount(indexName);
	else return null;
    }

    @Override
    public Document conceptCount(String indexName) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.conceptCount(indexName);
	else return null;
    }

    @Override
    public Document getStandardised(String indexName, String id) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getStandardised(indexName, id);
	else return null;
    }

    @Override
    public void saveTable(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveTable(def);
    }

    @Override
    public void setAutoMatch(boolean b) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.setAutoMatch(b);
    }

    @Override
    public void deleteTable(String def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteTable(def);
    }

    @Override
    public void saveConceptGroup(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveConceptGroup(def);
    }

    @Override
    public void deleteConceptGroup(String def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteConceptGroup(def);
    }

    @Override
    public void saveConcept(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveConcept(def);
    }

    @Override
    public void deleteConcept(String purposeName, String purposeColumn) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteConcept(purposeName,purposeColumn);
    }

    @Override
    public void saveConceptMapping(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveConceptMapping(def);
    }

    @Override
    public void deleteConceptMapping(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteConceptMapping(def);
    }



    @Override
    public void saveMatchRule(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveMatchRule(def);
    }

    @Override
    public void saveMatchRules(List<Document> defs) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveMatchRules(defs);
    }
    
    @Override
    public void deleteMatchRule(long order) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteMatchRule(order);
    }

    @Override
    public void saveFuzzyIndex(Document def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveFuzzyIndex(def);
    }

    @Override
    public void deleteFuzzyIndex(String def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteFuzzyIndex(def);
    }

    @Override
    public Document getDefinition() {
	return null;
    }

    @Override
    public void removeFuzzy() {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.removeFuzzy();

    }

    @Override
    public Document getTable(String def) {
	return null;
    }

    @Override
    public void updateDefinition(Document def) {


    }

    @Override
    public void addTrigger(String name) {
	MonsterClient client = getClient();
	if (client!=null) client.addTrigger(name);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findRelationships(Document filter) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findRelationships(filter);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findTasks(Document filter) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findTasks(filter);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#addRelationship(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveRelationship(Document relationship) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.saveRelationship(relationship);
	return null;

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#resolveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public void resolveTask(Document taskData) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.resolveTask(taskData);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#aggregateTasks(java.util.ArrayList)
     */
    @Override
    public DBCursor aggregateTasks(ArrayList<Document> andlist) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.aggregateTasks(andlist);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateTask(java.lang.String, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateTask(String id, Document doc) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateTask(id, doc);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#relsDistinct(java.lang.String, java.lang.Class)
     */
    @Override
    public <TResult> Iterable<TResult> distinctRelationship(String fieldName, Class<TResult> resultClass) {

	MonsterClient client = getClient();
	if (client!=null) 
	    return client.distinct(fieldName, resultClass);
	else return null;

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateRelationship(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateRelationship(Document document, Document setUpdate, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.updateRelationship(document, setUpdate, options);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#peekQueue()
     */
    @Override
    public DBCursor peekQueue() {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.peekQueue();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteTasks(Document document) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteTasks(document);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#deleteRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteRelationships(Document document) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.deleteTasks(document);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findStream(com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> findStream(Document filter) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.findStream(filter);
	else return Stream.empty();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getReplicaConnections()
     */
    @Override
    public Map<Integer, ICollection> getReplicaConnections() {
	try {
	    throw new Exception("Replicas cannot be seen from a Remote Collection");
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDatabase()
     */
    @Override
    public Database getDatabase() {
	try {
	    throw new Exception("Database cannot be seen from a Remote Collection");
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getTrigger()
     */
    @Override
    public String getTrigger() {
	try {
	    throw new Exception("Triggers cannot be seen from a Remote Collection");
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return null;
    }
    
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#split(java.util.List, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Map<Document, List<Document>> split(List<Document> list, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.split(list, in, options);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.List)
     */
    @Override
    public Document fuzzyMatch(Document matchNode, List<IIndex> matchIndexes) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.fuzzyMatch(matchNode, matchIndexes);
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#match(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> match(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.match(asDocument, in, options).collect(Collectors.toList()).stream();
	else
	    return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#analyse(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> analyse(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.analyse(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#limit(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> limit(long amount, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.limit(amount, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#lookup(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> lookup(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.lookup(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#join(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> join(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.join(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#minus(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> minus(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.minus(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#skip(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> skip(long amount, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.skip(amount, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#group(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> group(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.group(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#out(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> out(String collectionName, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.out(collectionName, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#sort(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> sort(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.sort(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#bucket(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> bucket(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.bucket(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzySearch(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> fuzzySearch(String string, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.fuzzySearch(string, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> fuzzyMatch(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.fuzzyMatch(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#unwind(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> unwind(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.unwind(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#spinOut(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> spinOut(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.spinOut(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRelated(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> getRelated(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getRelated(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#rematch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> rematch(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.rematch(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierBuild(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierBuild(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.classifierBuild(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierPredict(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierPredict(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.classifierPredict(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#arrf(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> arrf(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.arrf(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierTree(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierTree(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.classifierTree(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#coerce(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> coerce(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.coerce(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#compare(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> compare(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.compare(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#count(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> count(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.count(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#first(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> first(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.first(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#last(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> last(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.last(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#between(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> between(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.between(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#cluster(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> cluster(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.cluster(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#project(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> project(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.project(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#evaluate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> evaluate(Document asDocument, Stream<Document> in, Document options) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.evaluate(asDocument, in, options).collect(Collectors.toList()).stream();
	return null;
    }

  
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#traverseTop(com.entitystream.monster.db.Document, java.lang.String)
     */
    @Override
    public DBCursor traverseTop(Document from, String relType) {
	// TODO Auto-generated method stub
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#writeRel(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> writeRel(Document options, Stream<Document> in, Document goptions) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.writeRel(options, in, goptions).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#script(java.util.Map, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> script(Document scriptStatements, Stream<Document> in, Document stageoptions) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.script(scriptStatements, in, stageoptions).collect(Collectors.toList()).stream();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getMatchTypes()
     */
    @Override
    public DBCursor getMatchTypes() {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getMatchTypes();
	else return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveFuzzyIndexes(java.util.List)
     */
    @Override
    public void saveFuzzyIndexes(List<Document> def) {
	MonsterClient client = getClient();
	if (client!=null) 
	    client.saveFuzzyIndexes(def);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlows()
     */
    @Override
    public DBCursor getFlows() {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getFlows();
	else return new DBCursor();
    }
    @Override
    public DBCursor getBooks() {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getBooks();
	else return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlow(java.lang.String)
     */
    @Override
    public DBCursor getFlow(String name) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getFlow(name);
	else return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#validate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> validate(String type, Stream<Document> in, Document globalOptions) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.validate(type, in, globalOptions);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#task(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> task(Document options, Stream<Document> in, Document globalOptions) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.task(options, in, globalOptions);
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveTask(Document task) {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.saveTask(task);
	return null;

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDeltas()
     */
    @Override
    public Stream<Document> getDeltas() {
	MonsterClient client = getClient();
	if (client!=null) 
	    return client.getDeltas();
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateDelta(com.entitystream.monster.db.Document)
     */
    @Override
    public void updateDelta(Document doc) {
	// TODO Auto-generated method stub
	
    }
    
   

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getNodes()
     */
    @Override
    public Stream<Document> getNodes() {
	// TODO Auto-generated method stub
	return null;
    }

}
