/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import com.entitystream.monster.thrift.*;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;

import org.apache.commons.io.FileUtils;
import org.apache.thrift.TException;
import org.apache.thrift.server.TThreadPoolServer;

public class ConnectionHandler implements Connection.Iface {
    private Map<Long, Session> sessions = new HashMap<Long, Session>();

    private Container client;

    private Thread ticker;

    private Users authFile;

    private TThreadPoolServer server;

    private String dbPath;

    private String[] replicaSet;

    private User user;

    private Map<String, Document> roles;

    private int nodenum;

    private ClassLoader classLoader;


    public ConnectionHandler(String authFile, String dbPath, String[] replicaSet, int nodeNum) {

	this.authFile=Users.build(authFile);
	if(this.authFile!=null)
	    this.roles=this.authFile.getRoles();
	else 
	    this.roles=new HashMap<String, Document>();
	this.setDbPath(dbPath);
	this.replicaSet=replicaSet;
	if (this.replicaSet == null)
	    this.replicaSet=new String[] {};
	this.client = new Container(dbPath, this.replicaSet,nodeNum);
	this.nodenum=nodeNum;



	ticker = new Thread(new Runnable() {
	    @Override
	    public void run() {

		while (sessions.size()!=-1) {
		    try {

			ArrayList<Long> removeit = new ArrayList<Long>();
			for (Long cur : sessions.keySet()) {

			    if (sessions.get(cur).lastTouch !=-1 && sessions.get(cur).lastTouch + Session.MAXAGE < System.currentTimeMillis()) {
				//some cursors may be active -- ie being pulled from
				boolean anyActive=false;
				for (AtomicBoolean ab: sessions.get(cur).cursorActive.values())
				    if (ab.get()) {
					anyActive=true;
					break;
				    }

				if (!anyActive) {
				    sessions.get(cur).disconnect();
				    removeit.add(cur);
				}
				//sessions.remove(cur); --concurrent issue
			    }

			}
			if (removeit.size()>0) {
			    for (long cur: removeit)
				sessions.remove(cur);
			    System.out.println("Session pool is now " + sessions.size());
			}
			Thread.sleep(10000);
		    }
		    catch (Exception e) {
			e.printStackTrace();
		    }
		}
		System.out.println("Connection handler terminated");

	    }
	});
	ticker.start();
	System.out.println("Server "+nodeNum+" is UP!");
	if (replicaSet!=null)
	    System.out.println("Replicas are " + Arrays.toString(replicaSet));

	//install local libs

	loadLibs();
    }

    @Override
    public long command(long connection, String database, String collection, String command, boolean remoteView) throws TException {
	long cursorID=UUID.randomUUID().getLeastSignificantBits();

	//System.out.println("Receiving command="+command + ", remoteView? " + remoteView);
	try {
	    if (sessions.containsKey(connection)) {
		Session session = sessions.get(connection);
		session.isRemote(remoteView);

		if (command!=null) {
		    if (database !=null) {
			if (collection !=null) {
			    //session command
			    ICollection coll = client.getDatabase(database).getCollection(collection);

			    if (remoteView && coll instanceof Collection)
				coll=((Collection)coll).getLocalCollection();
			    else if (!remoteView && coll instanceof CollectionLocal)
				coll=((CollectionLocal)coll).getParent();

			    if (remoteView && !(coll instanceof CollectionLocal))
				throw new Exception("Recursive Call from Remote View to Collection (Global)");

			    if (!remoteView && !(coll instanceof Collection))
				throw new Exception("Illegal Local Collection Call");

			    return session.execute(coll, command);

			} else {
			    //db command
			    Database db = client.getDatabase(database);
			    return session.execute(db, command);

			}
		    } else {
			//non db command
			if (command.equalsIgnoreCase("{\"MethodName\":\"shutdown\"}")) {
			    client.shutdown();
			    System.exit(0);
			}
			else
			    return session.execute(client, command);
		    }
		}
	    } else {
		InvalidOperation io = new InvalidOperation();
		io.why = "Session Invalid";
		throw io;
	    }
	} catch(Exception e) {
	    Logger.getGlobal().severe(e.toString());
	    e.printStackTrace();
	    return -1;
	}
	return cursorID;
    }


    public long addCursor(long connectionID, DBCursor cursor, String command) {
	Session session =sessions.get(connectionID);
	if (session != null) {
	    return session.addCursor(cursor, command);
	}
	return -1;
    }

    @Override
    public long connect(String username, String password) throws TException {
	if (   (authFile==null && username==null) 
		|| (authFile==null && username.equalsIgnoreCase("guest") && password.equalsIgnoreCase("guest")) 
		|| ((authFile !=null) && ((user=authFile.ok(username, password))!=null))) {

	    //just return a random UUID
	    long sessionid = UUID.randomUUID().getLeastSignificantBits();
	    List<Document> uroles = new ArrayList<Document>();
	    if (user!=null)
		uroles=user.pullRoles(this.roles);
	    Session session = new Session(sessionid, user, uroles);
	    sessions.put(sessionid, session);
	    System.out.println("Session pool is now " + sessions.size());
	    return sessionid;
	} else {
	    if (authFile!=null)
		System.out.println("AuthFile is " + authFile.toString());
	    System.out.println("Failed to authenticate user " + username);
	    return -1;
	}
    }

    @Override
    public ByteBuffer cursorNext(long connection, long cursor, int count) throws TException {
	int counter=0;
	//System.out.println("Getting Cursor " +cursor+ " for connection "+ connection);
	StringBuffer sb = new StringBuffer();
	try {
	    sb.append("[");
	    Session sess = sessions.get(connection);
	    if (sess!=null) {
		DBCursor dbcursor = sess.getCursor(cursor);
		if (dbcursor!=null && !dbcursor.isStream()) {
		    while (dbcursor.hasNext() && (count==-1 || counter<count)) {
			Document doc = dbcursor.next();
			if (counter>0)
			    sb.append(",");
			sb.append(doc.toJson());
			counter++;
		    }

		    if (!dbcursor.hasNext())
			sessions.get(connection).removeCursor(cursor);

		} else if (dbcursor!=null && dbcursor.isStream()) {
		    Document readAhead= dbcursor.streamGet();
		    if (readAhead!=null) {
			sb.append(readAhead.toJson());
			counter++;
		    }
		    else
			sessions.get(connection).removeCursor(cursor);
		}
	    } else {
		InvalidOperation io = new InvalidOperation();
		io.why = "Session Invalid";
		throw io;
	    }
	} catch (StopIterationException ignore) {

	} catch (Exception e) {

	    InvalidOperation io = new InvalidOperation();
	    io.why = e.toString();
	    throw io;
	}
	sb.append("]");
	if (counter!=0)
	    return ByteBuffer.wrap(sb.toString().getBytes());
	else {
	    return ByteBuffer.allocate(0);
	}

    }

    @Override
    public void closeCursor(long connection, long cursor) throws TException {
	sessions.get(connection).removeCursor(cursor);
    }

    @Override
    public void closeConnection(long connection) throws TException {
	sessions.get(connection).disconnect();
	sessions.remove(connection);

    }

    public void setServer(TThreadPoolServer server2) {
	this.server=server2;


    }

    public String getDbPath() {
	return dbPath;
    }

    public void setDbPath(String dbPath) {
	//System.out.println("Server data is stored in "+dbPath);
	this.dbPath = dbPath;
    }

    public int getNodeNum() {
	return nodenum;
    }

    /**
     * @param file
     */
    public void installLib(File file) {
	//install it into the lib directory
	try {
	    File lib = new File(dbPath+File.separator+"libs"+File.separator+file.getName());
	    FileUtils.copyFile(file, lib);
	    loadLibs();
	} catch (Exception e1) {
	    e1.printStackTrace();
	} 
	//forward to other nodes


    }

    public void loadLibs() {
	try {
	    File libs = new File(dbPath+File.separator+"libs"+File.separator);
	    File[] flibs=libs.listFiles();
	    if (flibs!=null) {
		URL[] ulibs=new URL[flibs.length];
		for (int p=0; p<flibs.length;p++)
		    ulibs[p]=flibs[p].toURL();
		classLoader = URLClassLoader.newInstance(ulibs, Thread.currentThread().getContextClassLoader());
	    } else classLoader=Thread.currentThread().getContextClassLoader();
	} catch (Exception ex) {
	    throw new RuntimeException("Cannot load libraries from custom jar files. Reason: " + ex.getMessage());
	}
    }

}
