/**
 *
	MonsterDB - Collection Based Database with fuzzy matching
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.File;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ClassUtils;
import org.reflections.Reflections;

import com.entitystream.identiza.metadata.IdentizaSettings;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.RuleSet;
import com.entitystream.identiza.wordlist.RuleSetMap;
import weka.classifiers.AbstractClassifier;
import weka.core.Drawable;

public class Container {
	private static Map<String,Database> databases = new HashMap();
	private static String dataPath;
	private static String rulePath=IdentizaSettings.getApplicationRootPath("rules");
	private static String[] replicaSet;
	private static boolean debug;
	private static int nodeNum;
	private static boolean remoteView=false;

	public Container( String dbPath, String[] replicaSet1, int node ) {
		IdentizaSettings.setWorkingDir(dbPath);
		RuleFactory.setOffline();
		rulePath=IdentizaSettings.getApplicationRootPath("rules");
		new File(rulePath).mkdirs();
		RuleFactory.setRuleSetMap(RuleSetMap.readMap(rulePath));
		dataPath=IdentizaSettings.getApplicationRootPath("db");
		new File(dataPath).mkdirs();
		replicaSet=replicaSet1;
		nodeNum=node;
	}

	public static Database getDatabase(String name) { 
		if (databases.containsKey(name))
			return databases.get(name);
		else {
			Database newone = new Database(name, false,replicaSet,nodeNum);
			databases.put(name, newone);
			newone.setExplain(debug);
			return newone;
		}
	}


	public static Database createDatabase(String name) { 
		if (!databases.containsKey(name)) {
			Database newone = new Database(name, false,replicaSet,nodeNum);
			databases.put(name, newone);
			newone.setExplain(debug);
			return newone;
		} else return databases.get(name); 
	}


	public static void dropDatabase(String name) { 
		if (databases.containsKey(name)) {
			databases.get(name).drop();
			databases.remove(name);
		} else {
			(new Database(name, false,replicaSet,nodeNum)).drop();
		}
		
	}

	public static Database getDatabaseRO(String name) { 
		if (databases.containsKey(name))
			return databases.get(name);
		else {
			Database newone = new Database(name, true,replicaSet,nodeNum);
			newone.setExplain(debug);
			databases.put(name, newone);
			return newone;
		}
	}

	public BasicDBList listClassifiers() {
	    BasicDBList list = new BasicDBList();
	    
	    Reflections reflections = new Reflections("weka.classifiers");
	    Set<Class<? extends AbstractClassifier>> classes = reflections.getSubTypesOf(AbstractClassifier.class);
	    classes.retainAll(reflections.getSubTypesOf(Drawable.class));
	    for (Class<? extends AbstractClassifier> implClass : classes) {
	        list.add(implClass.getName().replaceAll("weka.classifiers.", ""));
	    }
	    
	    return list;
	}

	public BasicDBList listDatabaseNames() {
		
		File fpath = new File(dataPath);
		if (!fpath.exists())
			fpath.mkdirs();
		BasicDBList list = new BasicDBList();
		
		File[] lp = fpath.listFiles();
		if (lp!=null)
		for (File filename : lp) {
			if (!filename.isDirectory() && !filename.getName().startsWith("."))
				list.add(filename.getName());
		}
		return list;
	}

	public static void setDebug(boolean ddebug) {
		debug=ddebug;

	}

	public void disconnect(String dbn) throws Exception {
		System.out.println("Closing " + dbn);
		Database db = databases.get(dbn);
		if (db!=null) {
			db.close();
			databases.remove(dbn);
		}
		System.out.println("Closed " + dbn);
	}

	public DBCursor executeCommand(String command, User user, Session session) {

		return Container.executeCommand(this, command, user, session);
	}


	public static DBCursor executeCommand(Object on, String command, User user, Session session) {
		try {
			Document cmdDoc = Document.parse(command);
			if (cmdDoc.getString("MethodName")==null || cmdDoc.getString("MethodName").length()==0)
				return null;
			List<String> nparts=null;
			if (cmdDoc.containsKey("nextParts")) {
				nparts=new ArrayList<String>();
				
				for (String o : cmdDoc.getString("nextParts").split(",(?=([^\"]*\"[^\"]*\")*[^\"]*$)"))
					if (o.length()>0)
							nparts.add(o.replaceFirst("^.", ""));
				
				cmdDoc.remove("nextParts");
			}
			
			Method methodToUse =null;
			Object[] args=null;
			for (Method m : on.getClass().getMethods()) {
				if (m.getName().equals(cmdDoc.getString("MethodName"))){
					int found=0;
					args=new Object[m.getParameterCount()];
					int pos=0;
					for (Parameter p : m.getParameters()) {
						pos++;
						for (String key: cmdDoc.keyString()) {
							if (key.equalsIgnoreCase(""+pos) && (ClassUtils.isAssignable( p.getType(), cmdDoc.get(key).getClass(), true)
									|| p.getType().isAssignableFrom(cmdDoc.get(key).getClass()))) {
								args[found]= cmdDoc.get(key);
								found++;
							}
						}

					}
					if (found==m.getParameterCount() && found == cmdDoc.size()-1) {
						methodToUse = m;
						break;
					}
				}
			}
			if(methodToUse != null) {


				try {
				        Object ret=methodToUse.invoke(on, args);
					
					if (ret!=null) {
						DBCursor dbc = null;
						if (ret instanceof DBCursor)
							dbc= (DBCursor)ret;
						else
						    dbc=new  DBCursor(ret);
						
						if (dbc!=null) {
							if (nparts!=null && nparts.size()>0) {
								
								for (String part : nparts) {
									if (part.length()>0) {
									   dbc=followOn(dbc, part, user,session);
									}
								}
							}
							
						}
						return dbc;
					}
					
				} catch (Exception e) {
					e.printStackTrace();
				}

			} else {
				if (args==null)
					throw new MethodNotFoundException (on.getClass().getSimpleName()+"."+(cmdDoc.getString("MethodName")+"()"));
				else throw new MethodNotFoundException (on.getClass().getSimpleName()+"."+(cmdDoc.getString("MethodName")+"("+ cmdDoc.toJson() + ")"));
			}
		} catch (Exception e) {
			e.printStackTrace();
			Logger.getLogger("").severe(e.toString());
			return new DBCursor(new Document("Error", e.toString()));
		}
		return null;
	}

	
	private static DBCursor followOn(DBCursor dbc, String argString, User user, Session session) {
		
			String command=argString;
			if (argString.indexOf("(")!=-1) {
				command=argString.substring(0,argString.indexOf("("));
				argString=argString.substring(argString.indexOf("(")+1);

				if (argString.indexOf(")")!=-1) {
						argString=argString.substring(0, argString.indexOf(")"));
				}
			}
			List args=null;
			if (argString.length()==0) {

			}
			else if (argString.trim().startsWith("{") || argString.trim().startsWith("[")) {

				Object argso = Document.parseListOrDoc(argString);

				if (argso instanceof String) {
					System.err.println("Error: " + (String)argso);
					return null;
				}
				args = (List)argso;
			} else {
				//might be a single quoted string or a list of them
				String[] arganaught = argString.split(",");
				if (arganaught.length>0) {
					args=new ArrayList<String>();
					for (String a : arganaught){
						if (a.trim().startsWith("\"")|| a.trim().startsWith("'"))
							args.add(a.replaceAll("\\\"|'", "").trim());
						else {
							//number or boelan
							try{
								args.add(Double.parseDouble(a.trim()));
							} catch (NumberFormatException n){
								try{
									args.add(Boolean.parseBoolean(a.trim()));
								} catch (Exception e){
									args.add(a);
								}
							}

						}
					}
				}
			}


			if (args!=null && args.size()==0)
				args=null;


			//convert to a doc

			int count=0;
			Document argDc=null;
			if (args!=null){
				argDc = new Document();
				for (Object argitem : args) {
					count++;
					argDc.put(""+count, argitem);
				}
				argDc.append("MethodName", command);
			} else {
				argDc=new Document("MethodName", command);
			}
			return executeCommand(dbc, argDc.toJson(), user,session);
		
	}
	
	public void shutdown() throws Exception {
		for (String dbn :databases.keySet()) {
			disconnect(dbn);
		}

	}
	
	public int getNodeNum() {
		return nodeNum;
	}

	public void setDBPath(String dbPath) {
		dataPath=dbPath;
		
	}
	
	public static void setRuleSet(RuleSetMap map) {
		RuleFactory.setOffline();
		RuleFactory.setRuleSetMap(map);
		RuleSetMap.writeMap(rulePath, map);
	}
	
	public static void setRuleSet(Document doc) {
		RuleFactory.setOffline();
		RuleSetMap map = RuleSetMap.toMap(Document.parse(doc.toJson()));
		RuleFactory.setRuleSetMap(map);
		RuleSetMap.writeMap(rulePath, map);
	}
	
	public static String getRuleSetPath() {
		return rulePath;
	}
	
	public static void installRules() {
		Container.installRules("MatchCompanyName,MatchPersonName,MatchCountry");
	}
	
	public static void installRules(String list2) {
		String[] list = list2.split(",");
		ArrayList<String> cleanList = new ArrayList<String>();
		for (String fi : list) {
			cleanList.add(fi.trim());
		}
		RuleFactory.invalidate();
		for (String purpose : cleanList){
			ArrayList<Document> elemlist = (ArrayList<Document>)RuleFactory.executeApiCall("rule/"+purpose);
			if (elemlist!=null) {
				System.out.println("Installing " + elemlist.size() + " " + purpose + " rules");
			   RuleFactory.build(purpose, elemlist);
			}
		}
		RuleSetMap rsm = RuleFactory.getRuleSetMap();
		System.out.println("Writing to " + rulePath);
		RuleSetMap.writeMap(rulePath, rsm);
		
		for(String client : replicaSet) {
			MonsterClient clientC =new MonsterClient(client);
			clientC.setRuleSet(rsm);
			try {
				clientC.disconnect();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		setRuleSet(rsm);
		
	}
	
	public static void setRuleSetPath(String path) {
		
		RuleFactory.setOffline();
		
		RuleSetMap rsm = RuleSetMap.readMap(path);
		//send to other nodes
		for(String client : replicaSet) {
			MonsterClient clientC =new MonsterClient(client);
			clientC.setRuleSet(rsm);
			try {
				clientC.disconnect();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		setRuleSet(rsm);
	}
	
	public static void saveRuleSet(String path) {
		RuleSetMap rsm =RuleFactory.getRuleSetMap();
		RuleSetMap.writeMap(path, rsm);
	}

	
	public void saveQualityRule(Document def) {
		RuleSet rules = RuleFactory.getRuleSetMap().get(def.getString("rulePurpose"));
		RuleSet.updateRuleSet(rules, Collections.singletonList(def).iterator());
		saveRuleSet(rulePath);
	}

	public DBCursor findQualityRules(Document def) {
		String p=def.getString("rulePurpose");
		Set<String> purps = new HashSet<String>();
		if (p!=null)
			purps = Collections.singleton(p);
		else
			purps = RuleFactory.getRuleSetMap().keySet();
		List<Document> docs = new ArrayList<Document>();
		for (String purp : purps) {
		    RuleSet rules = RuleFactory.getRuleSetMap().get(purp);
		    docs.addAll(rules.toDocuments(purp, false));
		    docs.addAll(rules.toDocuments(purp, true));
		}
		return new DBCursor(ICollection.filter(def, docs.stream(), new Document()).collect(Collectors.toList()));
	}

	public void setRemoteView(boolean remoteViw) {
		remoteView=remoteViw;
		
	}

}
