/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DBCursor implements Iterable<Document>, Iterator<Document>{

    private List<Document> cursor;
    private Stream<Document> stream;
    private int position = 0;
    private Iterator<Document> intIterator;
    LinkedBlockingQueue<Document> blockering;
    Thread background;
    private AtomicBoolean inProgress;

    DBCursor(List<Document> inlist){
	this.cursor=inlist;
    }

    public DBCursor() {
	cursor=new ArrayList<Document>();
    }

    public DBCursor(Object ret) {
	if (ret == null) {
	    cursor=new ArrayList<Document>();
	} else if (ret instanceof Document) {
	    cursor=new ArrayList<Document>();
	    cursor.add((Document)ret);
	} else if (ret instanceof String) {
	    cursor=new ArrayList<Document>();
	    cursor.add(new Document("String",ret));
	} else if (ret instanceof Number) {
	    cursor=new ArrayList<Document>();
	    cursor.add(new Document("Number",ret));
	} else if (ret instanceof List) {
	    cursor=new ArrayList<Document>();
	    for (Object o : (List)ret)
		if (o instanceof Document )
		    cursor.add((Document)o);
		else {
		    if (o instanceof Map)
			cursor.add(new Document((Map)o));
		    else
			cursor.add(new Document("value", o));
		}
	} else if (ret instanceof Set) {
	    cursor=new ArrayList<Document>();
	    for (Object o : (Set)ret)
		if (o instanceof Document )
		    cursor.add((Document)o);
		else {
		    if (o instanceof Map)
			cursor.add(new Document((Map)o));
		    else
			cursor.add(new Document("value", o));
		}
	} else if (ret instanceof AggregateIterable) {
	    cursor=new ArrayList<Document>();
	    Iterator<Document> it = ((AggregateIterable)ret).iterator();
	    if (it!=null)
		while (it.hasNext()) {
		    Object o = it.next();
		    if (o instanceof Document)
			cursor.add((Document)o);
		    else
			cursor.add(new Document("value", o));
		}

	} else if (ret instanceof Iterable) {
	    cursor=new ArrayList<Document>();
	    Iterable it = (Iterable)ret;
	    if (it!=null)
		for (Object o : it)
		    if (o instanceof Document)
			cursor.add((Document)o);
		    else
			cursor.add(new Document("value", o));
	} else if (ret instanceof Stream) {
	    stream=((Stream)ret);
	}
    }


    public DBCursor extend(DBCursor more) {
	if (cursor!=null)
	    cursor.addAll(more.list());

	return this;
    }


    public void add(Document d) {
	if (cursor!=null)
	    cursor.add(d);
    }

    public Collection<? extends Document> list() {
	if (cursor!=null)
	    return cursor;
	else return null;
    }
    @Override
    public Iterator<Document> iterator() {
	if (cursor!=null)
	    return cursor.iterator();
	else if (stream!=null) {
	    if (intIterator==null)
		intIterator=stream.iterator();
	    return intIterator;
	}
	else return null;
    }

    public DBCursor limit(long limit) {
	if (cursor!=null)
	    return new DBCursor(cursor.stream().limit(limit).collect(Collectors.toList()));
	else if (stream!=null)
	    return new DBCursor(stream.limit(limit));
	else return null;
    }

    public DBCursor skip(long skip) {
	if (cursor!=null)
	    return new DBCursor(cursor.stream().skip(skip).collect(Collectors.toList()));
	else if (stream!=null)
	    return new DBCursor(stream.skip(skip));
	else return null;
    }

    public Document first() {
	if (cursor.size()>0)
	    return cursor.get(0);
	else if (stream!=null)
	    if (iterator().hasNext())
		iterator().next();
	return null;
    }

    public Document last() {
	if (cursor.size()>0)
	    return cursor.get(cursor.size()-1);
	else if (stream!=null)
	    iterator().next();
	return null;
    }

    public int count() {
	if (cursor!=null)
	    return cursor.size();
	else return 0;
    }
    public Stream<Document> stream() {
	if (stream==null && cursor!=null)
	    return cursor.stream();
	else return stream;
    }

    @Override
    public boolean hasNext() {
	if (cursor!=null) {
	    if (position < cursor.size())
		return true;
	    else
		return false;
	} else if (stream !=null)
	    return iterator().hasNext();
	else return false;
    }

    @Override
    public Document next() {
	if (cursor!=null) {
	    if (this.hasNext())
	    {
		toggleInProgress(true);
		Object next = cursor.get(position++);
		if (next instanceof Document) {
		    toggleInProgress(false);
		    return (Document)next;
		}
		
	    }
	} 
	toggleInProgress(false);
	return null;

    }


    public Iterable<Document> sort(Document object) {
	return (Iterable<Document>) cursor
		.stream()
		.sorted(new Comparator() {

		    @Override
		    public int compare(Object doc1, Object doc2) {
			int c=0;
			if (object!=null)
			  for (Object key: object.keySet()) {
			    Object v1=((Document)doc1).getProjection((String)key);
			    Object v2=((Document)doc2).getProjection((String)key);
			    int dir=object.getInteger((String)key, 1);
			    if (v1 instanceof String && v2 instanceof String) {
				c = v1.toString().compareTo(v2.toString())*dir;
			    } else if (v1 instanceof Number && v2 instanceof Number) {
				c = ((Double)((Number)v1).doubleValue()).compareTo(((Number)v2).doubleValue())*dir;
			    } else if (v1 instanceof Date && v2 instanceof Date) {
				c = ((Date)v1).compareTo((Date)v2)*dir;
			    } else if (v1 instanceof Boolean && v2 instanceof Boolean) {
				c = ((Boolean)v1).compareTo((Boolean)v2)*dir;
			    } else if (v1 instanceof List && v2 instanceof List) {
				for (Object it1: (List)v1) {
				    for (Object it2: (List)v2) {
					int _c = 0;
					if (it1 instanceof String && it2 instanceof String)
					    _c = v1.toString().compareTo(v2.toString())*dir;
					else if (it1 instanceof Number && it2 instanceof Number) 
					    _c = ((Double)((Number)it1).doubleValue()).compareTo(((Number)it2).doubleValue())*dir;
					else if (it1 instanceof Date && it2 instanceof Date)
					    _c = ((Date)it1).compareTo((Date)it2)*dir;
					else if (it1 instanceof Boolean && it2 instanceof Boolean) 
					    _c = ((Boolean)it1).compareTo((Boolean)it2)*dir;
					c=_c;
					break;
				    }	
				    break;
				}

			    } else c=dir;
			    if (c!=0)
				return c;
			}
			return c;
		    }

		})
		.collect(Collectors.toList());

    }
    public Iterable<Document> projection(Document projection) {
	return (Iterable<Document>) cursor
		.stream().parallel().map(doc -> {
		    Document doco = new Document();
		    for (Object k : projection.keySet())
			doco.append((String)k,doc.getProjection((String)k));
		    return doco;
		}).collect(Collectors.toList());
    }

    public DBCursor allowDiskUse(boolean b) {
	if (cursor instanceof AggregateIterable)
	    ((AggregateIterable)cursor).allowDiskUse(b);
	return this;
    }

    /**
     * @return
     */
    public boolean isStream() {
	return stream!=null;
    }

    /**
     * 
     */
    AtomicBoolean stop=new AtomicBoolean(false);
    private String statement;
    public Document streamGet() {
	
	if (blockering==null) {
	    blockering = new LinkedBlockingQueue<Document>(100);    
	    background=new Thread(new Runnable() {
		@Override
		public void run() {
		    stream().forEach(doc -> {
			try {
			    while (!blockering.offer(doc, 100, TimeUnit.MILLISECONDS));
			} catch (Exception e) {
			}
		    });
		    stop.set(true);
		}
		
	    });
	    background.start();;
	    
	}
	
	try {
	    Document d = null;
	    while (d==null && (!stop.get() || blockering.peek()!=null)) {
		toggleInProgress(true);
		d=blockering.poll(1000, TimeUnit.MILLISECONDS);
	    }
	    toggleInProgress(false);
	    
	    return d;
	} catch (InterruptedException e) {
	   return null;
	}

    }

    public void destroy() {
	if (blockering!=null)
	    blockering.clear();
	if (background!=null)
	    background.interrupt();
	blockering=null;
	background=null;
	stream=null;
	cursor=null;
	
    }

   

    public void toggleInProgress(boolean b) {
	if (inProgress==null)
	    inProgress=new AtomicBoolean(b);
	else
	    inProgress.set(b);
    }
    
    public void setInProgress(AtomicBoolean ab) {
	this.inProgress = ab;
    }

    /**
     * @param statement
     */
    public void setStatement(String statement) {
	this.statement=statement;
	
    }

}
