/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintWriter;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

import org.apache.commons.io.FileUtils;

import org.mapdb.DB;
import org.mapdb.DBMaker.Maker;

import com.entitystream.identiza.metadata.IdentizaSettings;
import com.entitystream.identiza.wordlist.RuleFactory;
import com.entitystream.identiza.wordlist.RuleSetMap;

public class Database implements Serializable {
    private String dataPath;
    private transient DB data;
    private Map<String, Document> collections;
    private Map<String, ICollection> openCollections = new HashMap<String, ICollection>();
    private String name;
    private boolean ro = false;
    private boolean explain = false;

    private List<String> replicaSet;
    private int nodeNum;
    private boolean remoteView = false;
    private Invocable invocable;

    Database(String name, boolean allowRO, String[] replicaSet, int node) {
	this.replicaSet = new ArrayList<String>();
	if (replicaSet != null)
	    for (String rp : replicaSet)
		this.replicaSet.add(rp);

	this.nodeNum = node;

	this.dataPath = IdentizaSettings.getApplicationRootPath("db");

	File fpath = new File(dataPath);
	if (!fpath.exists())
	    fpath.mkdirs();
	this.name = name;
	this.ro = allowRO;
	DB _data = getData(ro);
	if (_data != null) {
	    try {
		collections = (Map<String, Document>) _data.hashMap("Collections").createOrOpen();

	    } catch (Exception e) {
		throw new NoDatabaseException("Read Only databases must exist before connected to");
	    }
	}
	// load the triggers if one can be found
	setupJS();

    }

    private void setupJS() {
	File jsPath = new File(dataPath + File.separator + name + ".js");
	;
	if (jsPath.exists()) {
	    try {
		ScriptEngineManager manager = new ScriptEngineManager();
		ScriptEngine engine = manager.getEngineByName("JavaScript");
		// read script file
		engine.eval(Files.newBufferedReader(Paths.get(jsPath.getAbsolutePath()), StandardCharsets.UTF_8));
		invocable = (Invocable) engine;
		try {
		    invocable.invokeFunction("init", this);
		} catch (NoSuchMethodException e) {
		    System.out.println("Warning: No init function defined in " + jsPath.getAbsolutePath());
		}
	    } catch (Exception e) {

		System.out.println("Scripting engine failed to start..." + jsPath.getAbsolutePath());
		System.out.println(e.toString());
	    }
	}
    }

    private DB getData() {
	return this.getData(false);
    }

    private DB getData(boolean allowRO) {
	if (data == null) {
	    Maker _data = org.mapdb.DBMaker.fileDB(dataPath + File.separator + name).fileMmapEnable()
		    .checksumHeaderBypass().closeOnJvmShutdown();
	    try {

		if (allowRO)
		    _data = _data.readOnly();
		data = _data.make();
	    } catch (Exception e) {
		Logger.getAnonymousLogger().severe("Database can not be found: " + e.toString());
		// e.printStackTrace();
	    }

	}

	return data;
    }

    public java.util.Collection getTempStore(String name) {
	return getData().hashSet(name).createOrOpen();
    }

    public ICollection getRelCollection(String name) throws NoDatabaseException {
	ICollection relCollection = null;
	name = cleanName(name);
	if (!collections.containsKey(name)) {
	    Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		    .append("Path", File.separator + this.name + "_Collections" + File.separator)
		    .append("Internal", true).append("ReplicaSet", replicaSet);
	    relCollection = createCollection(collDoc);
	    if (relCollection != null) {
		relCollection.createIndex(new Document("fromCol", 1));
		relCollection.createIndex(new Document("toCol", 1));
	    }
	} else
	    relCollection = getCollection(name);

	return relCollection;
    }

    public ICollection getTaskCollection(String name) throws NoDatabaseException {
	name = cleanName(name);
	ICollection taskCollection = null;
	if (!collections.containsKey(name)) {
	    Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		    .append("Path", File.separator + this.name + "_Collections" + File.separator)
		    .append("Internal", true).append("ReplicaSet", replicaSet);
	    taskCollection = createCollection(collDoc);
	    if (taskCollection != null) {
		taskCollection.createIndex(new Document("Table", 1));
		taskCollection.createIndex(new Document("TaskID", 1), new Document("unique", true));
	    }
	}

	else
	    taskCollection = getCollection(name);

	return taskCollection;
    }

    public ICollection getCollection(String name) throws NoDatabaseException {
	try {
	    name = cleanName(name);
	    ICollection collObject = openCollections.get(name);
	    if (collObject == null) {
		Document collDoc = collections.get(name);
		if (collDoc == null)
		    collObject = createCollection(name);
		else {
		    collObject = new Collection(this, collDoc);
		    openCollections.put(name, collObject);
		}
	    }

	    if (collObject != null) {

		return collObject;
	    } else
		return null;
	} catch (Exception e) {
	    e.printStackTrace();
	    throw new NoDatabaseException("Database connection is invalid");
	}
    }

    public ICollection createCollection(String name) throws NoDatabaseException {
	name = cleanName(name);
	if (collections.containsKey(name))
	    return getCollection(name);

	Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		.append("Path", File.separator + this.name + "_Collections" + File.separator)
		.append("ReplicaSet", replicaSet);

	ICollection c = null;
	if (!openCollections.containsKey(name))
	    c = new Collection(this, collDoc);
	else
	    c = openCollections.get(name);

	if (!ro)
	    collections.put(name, collDoc);
	else
	    throw new NoDatabaseException("Read only Databases cannot be used to create a collection");
	openCollections.put(name, c);
	return c;
    }

    public ICollection createCollection(String name, Document ranges) throws NoDatabaseException {
	// create or open
	name = cleanName(name);
	if (collections.containsKey(name))
	    return getCollection(name);
	Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		.append("Path", File.separator + this.name + "_Collections" + File.separator)
		.append("ReplicaSet", replicaSet);
	if (ranges != null)
	    collDoc.append("Ranges", ranges);

	ICollection c = null;
	if (!openCollections.containsKey(name))
	    c = new Collection(this, collDoc);
	else
	    c = openCollections.get(name);

	if (!ro)
	    collections.put(name, collDoc);
	else
	    throw new NoDatabaseException("Read only Databases cannot be used to create a collection");
	openCollections.put(name, c);
	return c;
    }

    private String cleanName(String name) {
	return name.replaceAll("\\\"|\\\\|\\'", "");
    }

    public ICollection createCollection(String name, String hashKey) throws NoDatabaseException {
	// create or open
	name = cleanName(name);
	if (collections.containsKey(name))
	    return getCollection(name);
	Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		.append("Path", File.separator + this.name + "_Collections" + File.separator)
		.append("ReplicaSet", replicaSet);
	if (hashKey != null)
	    collDoc.append("HashKey", hashKey);

	ICollection c = null;
	if (!openCollections.containsKey(name))
	    c = new Collection(this, collDoc);
	else
	    c = openCollections.get(name);

	if (!ro)
	    collections.put(name, collDoc);
	else
	    throw new NoDatabaseException("Read only Databases cannot be used to create a collection");
	openCollections.put(name, c);
	return c;
    }

    public ICollection createCollection(Document collDoc) throws NoDatabaseException {
	String name = cleanName(collDoc.getString("Name"));
	if (collections.containsKey(name))
	    return getCollection(name);
	ICollection c = null;

	boolean internal = collDoc.getBoolean("Internal", false);
	System.out.println("Opening Collection: " + name);
	if (!openCollections.containsKey(name))
	    c = new Collection(this, collDoc);
	else
	    c = openCollections.get(name);

	if (!ro && !internal)
	    collections.put(name, collDoc);
	else if (ro)
	    throw new NoDatabaseException("Read only Databases cannot be used to create a collection");
	openCollections.put(name, c);
	return c;
    }

    public ICollection createFuzzyCollection(String name) {

	return createFuzzyCollection(name, new Document(), null, null);

    }

    public ICollection createFuzzyCollection(String name, Document definition) {

	return createFuzzyCollection(name, definition, null, null);

    }

    public ICollection createFuzzyCollection(String name, String definitionFile) {
	Document definition = Document.fromFile(definitionFile);

	if (definition != null)
	    return createFuzzyCollection(name, definition, null, null);
	else
	    return null;
    }

    public ICollection createFuzzyCollection(String name, String definitionFile, Document ranges) {
	// create or open

	Document definition = Document.fromFile(definitionFile);

	if (definition != null)
	    return createFuzzyCollection(name, definition, ranges, null);
	else
	    return null;
    }

    public ICollection createFuzzyCollection(String name, String definitionFile, String hashKey) {
	// create or open

	Document definition = Document.fromFile(definitionFile);

	if (definition != null)
	    return createFuzzyCollection(name, definition, null, hashKey);
	else
	    return null;
    }

    public ICollection createFuzzyCollection(String name, Document definition, Document ranges, String hashKey) {
	// create or open
	name = cleanName(name);
	Document collDoc = new Document("Name", name).append("DBPath", dataPath)
		.append("Path", File.separator + this.name + "_Collections" + File.separator)
		.append("Definition", definition)

		.append("ReplicaSet", replicaSet);
	if (ranges != null)
	    collDoc.append("Ranges", ranges);
	if (hashKey != null)
	    collDoc.append("HashKey", hashKey);
	ICollection c = null;
	if (!openCollections.containsKey(name))
	    c = new Collection(this, collDoc);
	else
	    c = openCollections.get(name);
	c.setExplaining(isExplain());
	if (!ro)
	    collections.put(name, collDoc);
	openCollections.put(name, c);
	return c;
    }

    public void dropCollection(String name) {
	if (!ro) {
	    System.out.println("Dropping collection:" + name);

	    if (remoteView) {
		// do the work remotely first
		for (String client : replicaSet) {
		    System.out.println(
			    "Opening remote DB connection to " + client + ", because  remoteView=" + remoteView);
		    MonsterClient clientC = new MonsterClient(client);
		    clientC.setRemoteView(false);
		    clientC.useDatabase(this.name);
		    clientC.useCollection(name);
		    clientC.dropCollection();
		    try {
			clientC.disconnect();
		    } catch (Exception e) {
			e.printStackTrace();
		    }
		}
	    }

	    ICollection c = openCollections.get(name);
	    if (c != null) {
		try {
		    c.disconnect();
		} catch (Exception e) {
		    e.printStackTrace();
		}
		openCollections.remove(name);
	    }

	    Document collDoc = collections.get(name);
	    if (collDoc != null) {
		collections.remove(name);
		CollectionLocal.drop(collDoc);
	    }

	}

    }

    public void close() throws Exception {
	for (String c : openCollections.keySet()) {
	    openCollections.get(c).disconnect();
	}
	openCollections.clear();
	getData().close();

    }

    public synchronized void storeCollection(String collectionName, ICollection collection) {

	collections.put(collectionName, collection.getDefinition());
	openCollections.put(collectionName, collection);

    }

    public BasicDBList listCollectionNames() {
	BasicDBList list = new BasicDBList();
	for (String collName : collections.keySet())
	    if (!collections.get(collName).getBoolean("Internal", false))
		list.add(new Document("Name", collName).append("isFuzzy",
			collections.get(collName).containsKey("Definition")));
	return list;
    }

    public boolean isReadOnly() {
	return ro;

    }

    public void drop() {
	try {
	    File dir = new File(dataPath + File.separator + name + "_Collections");
	    File master = new File(dataPath + File.separator + name);

	    data.close();

	    if (dir.exists())
		FileUtils.forceDelete(dir);
	    if (master.exists())
		FileUtils.forceDelete(master);
	    System.out.println("Database dropped: " + name);
	} catch (Exception e) {

	    // e.printStackTrace();
	}

    }

    public boolean isExplain() {
	return explain;
    }

    public void setExplain(boolean explain) {
	this.explain = explain;
    }

    public DBCursor executeCommand(String command, User user, Session session) {

	return Container.executeCommand(this, command, user, session);
    }

    public String getName() {
	return name;
    }

    public String getDataPath() {
	return dataPath;
    }

    public List<String> getReplicaSet() {
	return replicaSet;
    }

    public int getNodeNum() {
	// TODO Auto-generated method stub
	return nodeNum;
    }

    public Document describeCollection(String name) {
	return collections.get(name);

    }

    public void setRemoteView(boolean remoteView) {
	this.remoteView = remoteView;

    }

    public void setScriptCode(String code) {
	if (remoteView)
	    for (String client : replicaSet) {
		System.out.println("Opening remote DB connection to " + client + ", because  remoteView=" + remoteView);
		MonsterClient clientC = new MonsterClient(client);
		clientC.setRemoteView(false);
		clientC.useDatabase(this.name);
		clientC.setScriptCode(code);
		try {
		    clientC.disconnect();
		} catch (Exception e) {
		    e.printStackTrace();
		}
	    }
	// actually do something
	try {
	    File jsPath = new File(dataPath + File.separator + name + ".js");
	    PrintWriter pw = new PrintWriter(new FileOutputStream(jsPath));
	    pw.print(code);
	    pw.flush();
	    pw.close();
	} catch (Exception e) {
	    e.printStackTrace();
	}
	setupJS();
    }

    public void setScriptPath(String path) {
	// do the work remotely first
	StringBuilder code = new StringBuilder();
	try {
	    BufferedReader br = new BufferedReader(new FileReader(path));
	    String ln = "";
	    while ((ln = br.readLine()) != null) {
		code.append(ln);
	    }
	} catch (Exception e) {
	    e.printStackTrace();
	}

	setScriptCode(code.toString());

    }

    /**
     * 
     */
    public Invocable getInvoker() {
	return invocable;
    }

}
