/**
 *
	MonsterDB - Collection Based Database with fuzzy matching
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import org.reflections.Reflections;

import com.entitystream.identiza.entity.resolve.metadata.IIndex;


public interface ICollection {

	void createIndex(Document fields, Document options);

	void createIndex(Document document);

	DBCursor find(Document filter);

	DBCursor find();

	DBCursor findFuzzy(Document filter);

	Document save(Document doc);

	Document insertOne(Document doc);

	long count(Document query);

	Document findOneAndReplace(Document filter, Document replacement, Document options);

	Document findOneAndReplace(Document filter, Document replacement);

	Document findOneAndUpdate(Document filter, Document amendments);

	Document findOneAndUpdate(Document filter, Document amendments, Document updateOpts);

	Document updateOne(Document filter, Document amendments);

	Document updateOne(Document filter, Document amendments, Document options);

	int updateMany(Document filter, Document amendments, Document options);

	int insertMany(List<Document> listRecs);

	Document findOneAndDelete(Document filter);

	Document deleteOne(Document filter);

	int deleteMany(Document filter);

	Stream<Document> aggregate(List<Document> pipeline, Document options);

	Stream<Document> aggregate(List<Document> pipeline);
	
	Stream<Document> aggregate(List<Document> in, List<Document> pipeline);

	DBCursor listIndexes();

	
	void rebuildIndex(String name);
	
	void dropIndex(String name);

	int saveMany(List<Document> records);

	void disconnect() throws Exception;

	void setExplaining(boolean explain);

	DBCursor find(Document document, Document explain);

	DBCursor executeCommand(String command, User user, Session session);

	void drop();

	<TResult> Iterable<TResult> distinct(String fieldName, Class<TResult> resultClass);

	
	Document getIndexPrefixSubMap(String indexName, String key, boolean b);

	Document getDocument(Object pointer);

	Document keyCount(String indexName);

	Document getStandardised(String indexName, String id);

	Document conceptCount(String indexName);


	public void removeFuzzy();
	public void saveTable(Document def);
	public void deleteTable(String def);
	public void saveConceptGroup(Document def);
	public void deleteConceptGroup(String def);
	public void saveConcept(Document def);
	public void deleteConcept(String purposeName, String purposeColumn);
	public void saveConceptMapping(Document def);
	public void deleteConceptMapping(Document def);
	
	
	
	public void saveMatchRule(Document def);
	public void deleteMatchRule(long order);
	public void saveFuzzyIndex(Document def);
	public void deleteFuzzyIndex(String def);

	Document getDefinition();

	Document getTable(String def);

	void createUniqueIndex(Document fields);

	void updateDefinition(Document def);

	
	DBCursor findFuzzy(String textQuery);

	
	DBCursor findRelationships(Document filter);
	

	DBCursor findTasks(Document filter);

	
	Document saveRelationship(Document relationship);

	void resolveTask(Document taskData);

	
	public DBCursor aggregateTasks(ArrayList<Document> andlist);

	
	Document updateTask(String id, Document doc);

	<TResult> Iterable<TResult> distinctRelationship(String string, Class<TResult> resultClass);

	Document updateRelationship(Document document, Document setUpdate, Document options);

	
	DBCursor peekQueue();
	
	void setAutoMatch(boolean a);
	
	void deleteTasks(Document document);
	void deleteRelationships(Document document);

	/**
	 * @param name
	 */
	void addTrigger(String name);

	/**
	 * @param filter
	 * @param amendments
	 * @return
	 */
	int updateMany(Document filter, Document amendments);

	/**
	 * @param filter
	 * @return
	 */
	Stream<Document> findStream(Document filter);

	/**
	 * @return
	 */
	Map<Integer, ICollection> getReplicaConnections();

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream match(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream analyse(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param long1
	 * @param in
	 * @param options
	 * @return
	 */
	Stream limit(long long1, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream lookup(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream join(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream minus(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param long1
	 * @param in
	 * @param options
	 * @return
	 */
	Stream skip(long long1, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream group(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param string
	 * @param in
	 * @param options
	 * @return
	 */
	Stream out(String string, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream sort(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream bucket(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param string
	 * @param in
	 * @param options
	 * @return
	 */
	Stream fuzzySearch(String string, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream fuzzyMatch(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream unwind(Document asDocument, Stream<Document> in, Document options);

	
	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream spinOut(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream getRelated(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream rematch(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream classifierBuild(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream classifierPredict(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream arrf(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream classifierTree(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream coerce(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream compare(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream count(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream first(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream last(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream between(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream cluster(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream project(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param asDocument
	 * @param in
	 * @param options
	 * @return
	 */
	Stream evaluate(Document asDocument, Stream<Document> in, Document options);

	/**
	 * @param list
	 * @param in
	 * @param options
	 * @return
	 */
	Map<Document, List<Document>> split(List<Document> list, Stream<Document> in, Document options);

	/**
	 * @return
	 */
	Database getDatabase();
	
	
	 public static  boolean filter(Document doc, Document filter) {
		if (doc==null)
		    return false;
		boolean same=true;
		if (filter!=null) {
		for (Object key: filter.keySet()) {
		    Object fvalue=filter.get(key);

		    if (((String)key).startsWith("$")){

			if (((String)key).equalsIgnoreCase("$or")) {
			    //iterate over the list
			    boolean innersame=false;
			    for (Document innerFilter : (List<Document>)filter.get(key)) {
				innersame  = filter(doc, innerFilter) || innersame;
			    }
			    if (!innersame) {
				same=false;
				break;
			    }
			}
			if (((String)key).equalsIgnoreCase("$and")) {
			    boolean innersame=true;
			    for (Document innerFilter : (List<Document>)filter.get(key)) {
				innersame  = filter(doc, innerFilter) && innersame;
			    }
			    if (!innersame) {
				same=false;
				break;
			    }
			}
		    } else {
			Object value = doc.getProjection((String) key);
			if (value!=null){
			    if (value instanceof Number && fvalue instanceof Number) {
				if (((Number)value).doubleValue()!=((Number)fvalue).doubleValue()){
				    same=false;
				    break;
				}

			    } else if (value instanceof Date && fvalue instanceof Date) {
				if (!((Date)value).equals((Date)fvalue)){
				    same=false;
				    break;
				}
			    } else if (value instanceof String) {
				if (fvalue instanceof String) {
				    if (!((String)value).matches((String)fvalue) && !((String)value).equals(((String)fvalue))){
					same=false;
					break;
				    }
				} else if (fvalue instanceof Pattern) {

				    if (!((Pattern) fvalue).matcher((String)value).matches()) {
					same=false;
					break;
				    }
				} else {
				    if (fvalue instanceof Document) {
					Object b=Document.translate$(fvalue, new Document("value", value));
					if (b instanceof Boolean)
					   same=(boolean) b;
					else 
					    same=Boolean.parseBoolean((String)b);
					if (!same)
					    break;
				    }
				    else if (fvalue instanceof List) {

					//check each one....
					if (!((List) fvalue).contains(value)) {

					    same=false;
					    break;
					}
				    }
				    else if (!fvalue.equals(value)) {
					same=false;
					break;
				    }
				}
			    } else if (value instanceof List) {
				if (fvalue instanceof Pattern) {
				    boolean foundone=false;
				    for (Object lvalue : (List)value) {
					if ((lvalue instanceof String) && ((Pattern) fvalue).matcher((String)lvalue).matches()) {
					    foundone=true;
					    break;
					}
				    }
				    if (!foundone) {
					same=false;
					break;
				    }
				}
				else {
				    if (fvalue instanceof Document) {
					same=(boolean) Document.translate$(fvalue, new Document("value", value));
					if (!same)
					    break;
				    } else
					if (!((List) value).contains(fvalue)) {

					    same=false;
					    break;
					}
				}
			    }
			    else
				if (fvalue instanceof Document) {
				    same=(boolean) Document.translate$(fvalue, new Document("value", value));
				    if (!same)
					break;
				}
				else if (!fvalue.equals(value)) {
				    same=false;
				    break;
				}
			} else {
			    same=false;
			    break;
			}
		    }
		} 
		} else return false;

		return same;

	    }

	 public static Stream<Document> filter(Document filter, Stream<Document> in, Document options) {
		if (in != null){
		    return in
			    .parallel()
			    .filter(doc -> ICollection.filter(doc, filter));

		} else return null;
	    }

	/**
	 * @return
	 */
	String getTrigger();

	/**
	 * @param matchNode
	 * @param matchIndexes
	 * @return
	 */
	Document fuzzyMatch(Document matchNode, List<IIndex> matchIndexes);

/**
	 * @param from
	 * @param relType
	 * @return
	 */
	DBCursor traverseTop(Document from, String relType);

	/**
	 * @param options
	 * @param in
	 * @param goptions
	 * @return
	 */
	Stream<Document> writeRel(Document options, Stream<Document> in, Document goptions);

	/**
	 * @param in
	 * @param pipeline
	 * @return
	 */
	Document aggregateMetadata(List<Document> in, List<Document> pipeline);

	/**
	 * @param pipeline
	 * @return
	 */
	Document aggregateMetadata(List<Document> pipeline);

	/**
	 * @param scriptStatements
	 * @param in
	 * @param stageoptions
	 * @return
	 */
	Stream<Document> script(Document scriptStatements, Stream<Document> in, Document stageoptions);

	/**
	 * @param defs
	 */
	void saveMatchRules(List<Document> defs);

	DBCursor getMatchTypes();

	/**
	 * @param def
	 */
	void saveFuzzyIndexes(List<Document> def);

	/**
	 * @return
	 */
	DBCursor getFlows();

	/**
	 * @return
	 */
	DBCursor getBooks();

	/**
	 * @param name
	 * @return
	 */
	DBCursor getFlow(String name);

	/**
	 * @param type
	 * @param in
	 * @param stageoptions
	 * @return
	 */
	Stream<Document> validate(String type, Stream<Document> in, Document stageoptions);
	Stream<Document> task(Document options, Stream<Document> in, Document globalOptions);

	/**
	 * @param task
	 * @return
	 */
	Document saveTask(Document task);

	/**
	 * @return
	 */
	Stream<Document> getDeltas();

	/**
	 * @param doc
	 */
	void updateDelta(Document doc);

	
	/**
	 * @return
	 */
	Stream<Document> getNodes();

	
	    

}