/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

import javax.script.Invocable;
import javax.script.ScriptException;

import org.mapdb.DB;
import org.mapdb.DBMaker.Maker;
import org.mapdb.IndexTreeList;
import org.mapdb.serializer.SerializerCompressionWrapper;

import com.entitystream.identiza.db.WorkTypes;
import com.entitystream.identiza.entity.resolve.match.Matchable;
import com.entitystream.identiza.entity.resolve.match.Indexable;

import com.entitystream.identiza.entity.resolve.metadata.ISchemaMeta;

/**
 * @author roberthaynes
 *
 */
public class MatchCollection extends Indexable implements Matchable{
    private static final String FROMCOL = "fromCol";
    private static final String TOCOL = "toCol";
    private static final String TABLENAME = "Table";

    private int sortCount;
    private ICollection nodeCollection;
    private ICollection relCollection;
    private ICollection taskCollection;
    private int eidCount;
    private DB _catalog;
    private IndexTreeList data;
    private Thread bgthread;
    private boolean killed;
    private int mergeCount;
    private int taskCount;
    private int linkCount;
    private String name;
    private String trigger;
    private Invocable invoker;
  

    public MatchCollection(ICollection nodeCollection, ICollection relCollection, ICollection taskCollection, ISchemaMeta _schDoc) {
	super.initialize(_schDoc);
	this.nodeCollection = nodeCollection;
	this.relCollection = relCollection;
	this.taskCollection = taskCollection;
	data=(IndexTreeList) getData().indexTreeList("Queue", SerializerCompressionWrapper.JAVA).createOrOpen();

	trigger=nodeCollection.getTrigger();
	invoker=nodeCollection.getDatabase().getInvoker();



	if (bgthread==null || killed) { 
	    killed=false; 


	    bgthread=new Thread(new Runnable() {

		@Override public void run() {
		    try { 
			System.out.println("Matching starts on " + name);
			while (!killed) { 

			    while (data.size()>0) {
				try { 
				    data.getLock().writeLock().lock();
				    Document matchNode= (Document) data.remove(0);
				    Document matchDoc=null;
				    if (matchNode.containsKey("Table")){
				      // System.out.println("rematching"  + matchNode.toString());
				       matchDoc = nodeCollection.fuzzyMatch(matchNode, matchIndexes);
					}
				    invoke("SAVE",matchNode);

				    mergeSort(matchDoc );
				    data.getLock().writeLock().unlock();
				} catch (Exception e) {
				    e.printStackTrace();
				}	
			    }
			    Thread.sleep(100);
			}
		    }
		    catch (Exception e) 
		    { 
			e.printStackTrace(); 
		    }
		    try {
			System.out.println("Matching stopping on " + name);
			_catalog.close();
			_catalog=null;
			relCollection.disconnect();
			taskCollection.disconnect();
			System.out.println("Matching stopped on " + name);
		    } catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		    }



		}});
	    bgthread.start();
	}


    }

    public void offer(Document matchNode) {
	data.add(matchNode);
    }

    public DBCursor peekAll() {
	return new DBCursor(data.iterator());
    }

    private DB getData() {
	if (_catalog==null) {

	    Document definition = nodeCollection.getDefinition();
	    name=definition.getString("Name");
	    File fpath = new File(definition.getString("DBPath")+definition.getString("Path"));
	    if (!fpath.exists())
		fpath.mkdirs();
	    String path=fpath.getAbsolutePath()+File.separator+definition.getString("Name")+"_Queue";
	    Maker _data = org.mapdb.DBMaker.fileDB(path).fileMmapEnable().checksumHeaderBypass().closeOnJvmShutdown();
	    try {
		if (_data!=null)
		    _catalog = _data.make();
	    } catch (Exception e) {
		System.out.println("Match Collections Queue can't be created on a non existent database, " + e.toString());
		//e.printStackTrace();
	    }
	}
	return _catalog;
    }


    @Override

    public boolean mergeSort(Document matches) {

	
	try {

	    if (matches!=null) {
		String id1=matches.getString("_id");
		sortCount++;
		for ( Object oMatchedTo : matches.getList("Matches")) {
		    Document matchedTo = (Document)oMatchedTo;

		    String id2=matchedTo.getString("_id");
		    if (!id1.equalsIgnoreCase(id2)) {
			//now make sure they are not related
			Document where = new Document();
			ArrayList<Document> orlist = new ArrayList<Document>();
			orlist.add(new Document().append(FROMCOL,id1).append(TOCOL, id2));
			orlist.add(new Document().append(TOCOL,id2).append(FROMCOL,id1));
			where.append("$or", orlist);
			DBCursor list = relCollection.find(where);
			if (list!=null && list.hasNext())
			    continue;

			//if we are here then go for it - merge, link or EID stamp them

			String action=matchedTo.getString("action");
			int iAction=WorkTypes.IGNORE;
			if (action.equalsIgnoreCase("LINK")){  
			    iAction=WorkTypes.LINK;
			}
			else if (action.equalsIgnoreCase("MERGE")){
			    iAction=WorkTypes.MERGE;
			}
			else if (action.equalsIgnoreCase("EID")){
			    iAction=WorkTypes.EID;
			}

			Document rule = matchedTo.getAsDocument("rule");
			double score = matchedTo.getDouble("score");
			if (score<=rule.getDouble("highScore") && score>=rule.getDouble("lowScore"))
			    //override to task - but leave actiontext
			    iAction=WorkTypes.TASK;

			Document mr = new Document();
			mr.append("action", action);
			mr.append("actionText", matchedTo.getString("actionText"));
			mr.append("rule", rule);
			mr.append("score", score);
			mr.append("_id1", id1);
			mr.append("_id2", id2);

			new MatchItem(mr, iAction).compute();
		    }

		}


	    }
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return true;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.Matchable#eid(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override

    public boolean eid(Document mr) {

	if (mr!=null) {
	    try {

		eidCount++;

		String leftID=mr.getString("_id1");
		String rightID=mr.getString("_id2");

		Document left=nodeCollection.getDocument(leftID);
		Document right=nodeCollection.getDocument(rightID);

		
		if (left.containsKey(TABLENAME) && right.containsKey(TABLENAME)) {

		    long leftEID=left.getLong("EID");
		    long rightEID=right.getLong("EID");
		    if (leftEID!=-1 && rightEID!=-1) {

			long EID=Math.min(leftEID, rightEID);
			left.append("EID", EID);
			right.append("EID", EID);

			Document internal=null;
			Document external=null;
			if (schDoc.getTable((left.getString(TABLENAME))).isInternal() && !schDoc.getTable((right.getString(TABLENAME))).isInternal())
			{
			    internal = left;
			    external = right;

			} else if (schDoc.getTable((right.getString(TABLENAME))).isInternal() && !schDoc.getTable((left.getString(TABLENAME))).isInternal())
			{
			    internal = right;
			    external = left;
			} 

			String sentIT="";
			if (internal!=null && external != null) {
			    internal = applyInheritanceFrom(internal,external);
			    nodeCollection.save(internal);
			    sentIT=internal.getString("_id");
			}

			if (left.getLong("EID")!=leftEID && !sentIT.equalsIgnoreCase(left.getString("_id"))) {
			    nodeCollection.updateMany(new Document("EID", leftEID), new Document("$set", new Document("EID", EID)), new Document("upsert", false));
			}
			if (right.getLong("EID")!=rightEID && !sentIT.equalsIgnoreCase(right.getString("_id"))) {
			    nodeCollection.updateMany(new Document("EID", rightEID), new Document("$set", new Document("EID", EID)), new Document("upsert", false));
			    
			}
			invoke("EID", new Document("left", left).append("right",right).append("match", mr));
			


			return true;
		    }
		}
		return false;
	    }catch(Exception e) {
		e.printStackTrace();
	    }

	}
	return false;
    }

    @Override
    public void invoke(String action, Document data) {
	if (this.invoker!=null && this.trigger!=null) {
	    try {
		invoker.invokeFunction(trigger, action, data.toJson());
	    } catch (NoSuchMethodException | ScriptException e) {
		e.printStackTrace();
	    }
	}
	
    }

    @Override
    public void resolveTask(Document mr)  {
	String id = mr.getString("_id");
	mr.remove("_id");

	//removed 
	//{"TaskID":"71d4f9e8-f775-449c-99db-a9fabdf1da3e_7f1956e4-7e8e-49bd-b66b-01245825a168","Resolved":true,"ResolvedBy":"robert@haynes-web.com","lastUpdated":"Oct 16, 2019 3:07:45 PM","Type":"Rejected","_action":-1}
	if (mr.getInteger("_action")==-1) { //accept or reject
	    // Document is as we need it. Accept or rejected.
	    taskCollection.findOneAndUpdate(new Document("_id", id), new Document("$set", mr));
	} else {
	    //do some work on the nodes..
	    //{"Task":"Link of 71d4f9e8-f775-449c-99db-a9fabdf1da3e and 66f278a1-45b1-41ee-8d5e-957095b82ddb",
	    //"Created":"Oct 16, 2019 12:41:51 PM","Due":"Oct 16, 2019 12:41:51 PM",
	    //"Resolved":"false",
	    //"Tablename":"[t1, t1]","Table":"Tasks",
	    //"Score":100,
	    //"Rule":{"canMatchSameSystem":true,"systemMatchType":0,"lowScore":90,"highScore":100,"actionText":""},
	    //"Type":"EID",
	    //"Nodes":["71d4f9e8-f775-449c-99db-a9fabdf1da3e","66f278a1-45b1-41ee-8d5e-957095b82ddb"],
	    //"_action":7}
	    Document activity = new Document();
	    activity.append("action", WorkTypes.toString(mr.getInteger("_action")));
	    activity.append("actionText", mr.getAsDocument("Rule").getString("actionText"));
	    activity.append("rule", mr.getAsDocument("Rule"));
	    activity.append("score", mr.getDouble("Score"));
	    activity.append("_id1", mr.getList("Nodes").get(0));
	    activity.append("_id2", mr.getList("Nodes").get(1));

	    new MatchItem(activity, mr.getInteger("_action")).compute();

	}

    }

    @Override
    public boolean addTask(Document mr)  {
	if (mr !=null) {
	    taskCount++;
	    try {

		String leftID=mr.getString("_id1");
		String rightID=mr.getString("_id2");

		Document left=nodeCollection.getDocument(leftID);
		Document right=nodeCollection.getDocument(rightID);

		if (left.containsKey(TABLENAME) && right.containsKey(TABLENAME)) {

		    ArrayList<String> tablenames = new ArrayList<String>();

		    tablenames.add(left.getString(TABLENAME));
		    tablenames.add(right.getString(TABLENAME));
		    Collections.sort(tablenames);



		    if (!leftID.equalsIgnoreCase(rightID)){
			String ID = leftID + "_" + rightID;
			if (leftID.compareTo(rightID)<0) { 
			    ID = rightID + "_" + leftID;
			}


			Document doc = taskCollection.getDocument(ID);//
			if (doc==null) {
			    Document values = new Document();
			    Date dueDate = new Date();
			    values.append("Task", mr.getString("actionText")+ " of " + leftID + " and " + rightID);
			    values.append("Created", new Date());
			    values.append("Due", dueDate);
			    values.append("Resolved", "false");
			    values.append("Tablename", tablenames.toString());
			    values.append("Table", "Tasks");
			    values.append("Score", mr.getDouble("score"));
			    values.append("Rule", mr.get("rule"));
			    values.append("Type", mr.getString("action"));
			    values.append("_id",ID);
			    ArrayList<String> nodes = new ArrayList<String>();
			    nodes.add(leftID);
			    nodes.add(rightID);
			    values.append("Nodes", nodes);

			    taskCollection.findOneAndReplace(new Document("_id", ID),  values, new Document("upsert", true));
			}
			return true;
		    }
		} 
		return false;
	    } catch (Exception e) {
		e.printStackTrace();
	    }
	}
	return false;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.Matchable#link(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override

    public boolean link(Document mr) {
	if (mr!=null) {
	    linkCount++;

	    try {


		String leftID=mr.getString("_id1");
		String rightID=mr.getString("_id2");

		Document left=nodeCollection.getDocument(leftID);
		Document right=nodeCollection.getDocument(rightID);

		if (left.containsKey(TABLENAME) && right.containsKey(TABLENAME)) {

		    Map<String, Object> values = new HashMap<String, Object>();
		    values.put("TaskResolved", "true");
		    values.put("TaskResolvedBy", "System");
		    values.put("TaskScore", mr.getDouble("score"));
		    values.put("TaskDue", new Date());
		    values.put("TaskRule", mr.get("rule"));
		    values.put("TaskCreatedBy", "System");


		    String relType=mr.getString("actionText");


		    String fromTable=left.getString(TABLENAME);
		    String toTable=right.getString(TABLENAME);





		    Document existing=relCollection.getDocument(leftID+"_"+rightID);
		    if (existing==null)
			existing=relCollection.getDocument(rightID+"_"+leftID);


		    if (existing==null || !existing.getString("relType").equalsIgnoreCase(relType)) {
			//add the rel
			Document rel = new Document("relType", relType);
			rel.append("task", values);
			rel.append("history", new Document("status", "ACTIVE").append("lastUpdate", new Date()));
			rel.append(FROMCOL, leftID);
			rel.append(TOCOL, rightID);
			rel.append("fromTable", fromTable);
			rel.append("toTable", toTable);
			rel.append("_id", leftID+"_"+rightID);
			relCollection.save(rel);
		    }
		    invoke("LINK", new Document("left", left).append("right",right).append("match", mr));
			
		    return true;
		}
		return false;
	    } catch(Exception e) {
		e.printStackTrace();
	    }
	}
	return false;
    }



    /* (non-Javadoc)
     * @see com.entitystream.monster.db.Matchable#merge(com.entitystream.identiza.entity.resolve.match.MatchRecordInterface)
     */
    @Override
    public boolean merge(Document mr){
	if (mr!=null) {
	    try{
		mergeCount++;

		String leftID=mr.getString("_id1");
		String rightID=mr.getString("_id2");

		Document rec1=nodeCollection.getDocument(leftID);
		Document rec2=nodeCollection.getDocument(rightID);

		if (rec1.containsKey(TABLENAME) && rec2.containsKey(TABLENAME)) {

		    String fromTable=rec1.getString(TABLENAME);
		    String toTable=rec2.getString(TABLENAME);

		    Document survivedRec = null;

		    Document goldenRec = calcGoldenRec(rec1, rec2, fromTable);


		    Document deletedRec = null;
		    if (rec1.getString("_id").equalsIgnoreCase(goldenRec.getString("_id"))){
			deletedRec = rec2;
			survivedRec = rec1;
		    }
		    else {
			deletedRec = rec1;
			survivedRec = rec2;
		    }
		    String deletedID=deletedRec.getString("_id");
		    String survivedID=survivedRec.getString("_id");

		    System.out.println("Removing record " + deletedID + ", surviving with " + survivedID);
		    System.out.println("Deleting Node ("+deletedRec.toJson()+") survived by ("+survivedRec.toJson()+") ");


		    String deletedKey = deletedRec.getString(schDoc.getTable(deletedRec.getString("Table")).getKeyField());
		    //which nodes and tasks is this deleted record linked to?
		    taskCollection.deleteMany(new Document("_id", Pattern.compile("^"+deletedID+"_.*")));
		    taskCollection.deleteMany(new Document("_id", Pattern.compile("^*._"+deletedID+"$")));
		    nodeCollection.deleteOne(new Document("_id", deletedID));
		    relCollection.deleteMany(new Document(FROMCOL, deletedID));
		    relCollection.deleteMany(new Document(TOCOL, deletedID));

		    //todo: add the relationships back to the new record

		    nodeCollection.save(goldenRec);

		    invoke("MERGE", new Document("left", rec1).append("right",rec2).append("match", mr).append("result", goldenRec));
			
		    
		    return true;
		}

	    } catch (Exception e) {
		e.printStackTrace();
	    }
	}
	return false;
    }

    /**
     * 
     */
    public void disconnect() {
	   killed=true;

    }

    /**
     * @param name
     */
    public void addTrigger(String name) {
	this.trigger=name;
	this.invoker=nodeCollection.getDatabase().getInvoker();
    }




}
