/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.PrintWriter;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import org.apache.commons.collections4.queue.CircularFifoQueue;
import org.jline.reader.LineReader;
import org.jline.reader.LineReaderBuilder;
import org.jline.terminal.Terminal;
import org.jline.terminal.Terminal.Signal;
import org.jline.terminal.Terminal.SignalHandler;
import org.jline.terminal.TerminalBuilder;


import picocli.CommandLine;
import picocli.CommandLine.Option;

public class MonsterCLI extends MonsterClient implements Runnable {
    private static boolean interrupted;
    @Option(names = {"-h", "--host"}, required = false, description = "Remote host IP or DNS Name")
    private  String host="localhost";
    @Option(names = {"-p", "--port"}, required = false, description = "Remote host port (default 27018)")
    private  int port=27018;
    @Option(names = {"-p2", "--httpport"}, required = false, description = "Remote http host port (default 27019)")
    private  int httpport=27019;
    @Option(names = {"-u", "--user"}, required = false, description = "The username to connect with")
    private  String username;
    @Option(names = {"-pw", "--password"}, required = false, description = "The usernames password")
    private  String password;
    @Option(names = {"-r", "--run"}, required = false, description = "Run the following command")
    private  String command;
    @Option(names = {"-d", "--db"}, required = false, description = "connect the DB")
    private  String db;
    @Option(names = {"-s", "--start"}, required = false, description = "Start a server")
    private  boolean deamon=false;
    @Option(names = {"-a", "--auth"}, required = false, description = "Authentication settings ")
    private  String auth;
    @Option(names = {"-c", "--collection"}, required = false, description = "Collection to load")
    private  String collection;
    @Option(names = {"-f", "--file"}, required = false, description = "File to load")
    private  String filename;
    @Option(names = {"-t", "--recordType"}, required = false, description = "Record Type to load (fuzzy tables only)")
    private  String tableName;
    @Option(names = {"-o", "--options"}, required = false, description = "Load options")
    private  String propertiestr;
    @Option(names = {"-db", "--dbPath"}, required = false, description = "Path for DB Files (Server only)")
    private  String dbPath;
    @Option(names = {"-x", "--script"}, required = false, description = "Run a grr script and exit")
    private  String script;
    @Option(names = {"-rs", "--replica"}, required = false, description = "Server replica (Server only)")
    private  String[] replicaSet;
    @Option(names = {"-n", "--node"}, required = false, description = "Server replica Node Number (hash distribution)")
    private  int node;
    @Option(names = {"-pbp", "--postBatchPipeline"}, required = false, description = "Post Batch Pipeline")
    private  String postBatchPipeline;
    @Option(names = {"-plp", "--postLoadPipeline"}, required = false, description = "Post Load Pipeline")
    private  String postLoadPipeline;
    @Option(names = { "-?", "--help"}, usageHelp = true, description = "Display this help and exit")
    private boolean help;
    @Option(names = { "-q", "--quiet"}, description = "Be less verbose")
    private boolean quiet;
    @Option(names = { "-v", "--version"}, description = "Report Version")
    private boolean vers;


    private LineReader lineReader;
    private Queue<String> history=new CircularFifoQueue<String>(10);

    private boolean explain=false;

    public MonsterCLI() throws NoDatabaseException {
	super();
    }

    public static void main(String... args) {
	CommandLine.run(new MonsterCLI(), System.err, args);
    }

    @Override
    public void run() {

	super.quiet=quiet;

	if (vers)
	    quiet=true;

	String subTitle="CLI";
	String typeit =", for details type \"show license\"";
	if (deamon) {
	    subTitle="Server";
	    typeit ="";
	}
	if (filename!=null) {
	    subTitle="Loader";
	    typeit ="";
	}

	if (!quiet){
	    System.out.println("                                _            _____  ____  ");
	    System.out.println("                               | |          |  __ \\|  _ \\ ");
	    System.out.println("      _ __ ___   ___  _ __  ___| |_ ___ _ __| |  | | |_) |");
	    System.out.println("     | '_ ` _ \\ / _ \\| '_ \\/ __| __/ _ \\ '__| |  | |  _ < ");
	    System.out.println("     | | | | | | (_) | | | \\__ \\ ||  __/ |  | |__| | |_) |");
	    System.out.println("     |_| |_| |_|\\___/|_| |_|___/\\__\\___|_|  |_____/|____/   "+ subTitle +" Build 119.21.0");
	    System.out.println("\n     (c) Robert James Haynes (EntityStream KFT), released under AGPL3 license"+typeit+"\n");

	}

	if (vers)
	    System.out.println("\nBuild 119.21.0");


	if (deamon){
	    if (replicaSet==null || replicaSet.length==0 || (replicaSet.length==1 && replicaSet[0].equalsIgnoreCase("null")))
		replicaSet=new String[] {};

	    DBServer.start(port, httpport, auth, dbPath, replicaSet,node);
	    return;
	} 

	if (filename!=null){
	    new MonsterLoadCLI(host, port, username, password, collection, db, filename, tableName, propertiestr, postBatchPipeline, postLoadPipeline, quiet, true);

	} else {


	    try {
		this.connect(host, port, username, password);
	    } catch (Exception e) {
		if (!quiet)
		    System.err.println("An error ("+e.toString()+") occurred, you can continue, but there is nothing useful to do.");
	    }
	    if (db!=null){
		this.useDatabase(db);

	    }
	    if (command!=null) {
		parseCMD(command, new PrintWriter(System.out, true), history, output, explain, this, quiet, lineReader,"");
	    } else if (script!=null) {
		parseScript(script);
	    } else {


		try {

		    final Terminal terminal = TerminalBuilder.builder()
			    .system(true)
			    .signalHandler(new SignalHandler(){


				@Override
				public void handle(Signal signal) {
				    if (signal.equals(Signal.INT))
				    {
					System.out.println("Interrupting Stream from Server...");
					interrupted=true;

				    }
				}
			    })
			    .build();

		    lineReader = LineReaderBuilder.builder()
			    .terminal(terminal)
			    .build();





		    while (true) {
			String shPrompt = "monsterDB> ";
			if (database!=null)
			    shPrompt=database+">";
			String cmd = lineReader.readLine(shPrompt);
			if (cmd.trim().length()>0){
			    if (!parseCMD(cmd, new PrintWriter(System.out, true), history, output, explain, this, quiet,lineReader,""))
				break;

			}
		    }

		} catch (Exception e1) {
		    System.err.println(e1.getClass().getSimpleName() + " caused monsterDB to stop.");
		}
		if (!quiet)
		    System.err.println("Grrr...");

	    }
	    try {

		this.disconnect();
	    } catch (Exception e) {
		System.err.println("Disconnect failed...");
	    }

	    if (filename==null)
		System.exit(0);
	}
    }

    private void parseScript(String script)  {
	try{
	    File file = new File(script);
	    if (file.exists()){
		BufferedReader br = new BufferedReader(new FileReader(file));
		String cmd="";
		int lc=0;
		while ((cmd=br.readLine())!=null){
		    lc++;
		    if (!parseCMD(cmd, new PrintWriter(System.out, true), history, output, explain, this, quiet,lineReader,"")){
			System.err.println("Grr command failed on line "+lc+": " + cmd);
			break;
		    }
		}
		br.close();
	    } else {
		System.err.println("Grr file not found: " + file.getAbsoluteFile().getPath());
	    }

	} catch (Exception e) {
	    System.err.println("An error ("+e.toString()+") occurred");
	}
    }





    public static boolean parseCMD(String cmd, PrintWriter out, Queue<String> history, OutputType output, boolean explain, MonsterClient client, boolean quiet, LineReader lineReader, String delimiter) {

	cmd=cmd.trim();
	String[] cmds = cmd.split(" ");
	if (!quiet)
	    out.println();
	if (cmds.length>0) {
	    if (cmds[0].equalsIgnoreCase("show")) {
		if (cmds.length>1 && cmds[1].equalsIgnoreCase("license")) {
		    out.println("    MonsterDB - Collection Based Database with fuzzy matching\n" + 
			    "    \n" + 
			    "    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary\n" + 
			    "\n" + 
			    "    This program is free software: you can redistribute it and/or modify\n" + 
			    "    it under the terms of the GNU Affero General Public License as\n" + 
			    "    published by the Free Software Foundation, either version 3 of the\n" + 
			    "    License, or (at your option) any later version.\n" + 
			    "\n" + 
			    "    This program is distributed in the hope that it will be useful,\n" + 
			    "    but WITHOUT ANY WARRANTY; without even the implied warranty of\n" + 
			    "    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" + 
			    "    GNU Affero General Public License for more details.\n" + 
			    "\n" + 
			    "    You should have received a copy of the GNU Affero General Public License\n" + 
			    "    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html");
		}
	    } else
		if (cmds[0].equalsIgnoreCase("recall")) {
		    if (cmds.length==1){
			Iterator<String> it = history.iterator();
			int count=0;
			while (it.hasNext())
			    out.println((++count)+": "+it.next());
		    } else {
			Iterator<String> it = history.iterator();
			int count=0;
			while (it.hasNext()){
			    cmd=it.next();
			    if ((""+(++count)).equalsIgnoreCase(cmds[1])){
				if (!quiet)
				    out.println(cmd);
				return parseCMD(cmd,out,history,output,explain,client,quiet,lineReader,"");

			    }
			}
		    }
		} else if (cmd.equalsIgnoreCase("/")){
		    Iterator<String> it = history.iterator();
		    while (it.hasNext())
			cmd=it.next();
		    if (!quiet) {
			out.println(cmd);
		    }
		    return parseCMD(cmd,out, history,output,explain,client,quiet,lineReader,"");
		} 
		else {

		    if (cmds[0].equalsIgnoreCase("use")) {
			client.useDatabase(cmds[1]);
		    } else if (cmds[0].equalsIgnoreCase("quit")) {
			return false;
		    } else if (cmds[0].equalsIgnoreCase("db")) {
			if (cmds.length==1){
			    if (!quiet) {
				out.println(client.currentDatabase());
			    }
			}
			else {
			    if (client.currentDatabase()!=null){
				Document argd  = new Document();
				if (cmds[1].equalsIgnoreCase("dropCollection")){
				    if (cmds.length>2)
					argd.append("1", cmd.substring("db dropcollection ".length())); 
				} else if (cmds[1].equalsIgnoreCase("createCollection")){
				    if (cmds.length>2)
					argd.append("1", cmds[2]); //name
				    if (cmds.length>3){
					String remain = cmd.split(" ", 4)[3];
					//remain is hashkey or rangedoc
					if (remain.startsWith("{"))
					    argd.append("2",Document.parse(remain));
					else
					    argd.append("2",remain);
				    }
				} else if (cmds[1].equalsIgnoreCase("createFuzzyCollection")){
				    if (cmds.length>2)
					argd.append("1", cmds[2]); //name
				    if (cmds.length>3)
					argd.append("2", cmds[3]); //def file
				    if (cmds.length>4)
					argd.append("3", cmds[4]); //rs path

				    if (cmds.length>5){
					String remain = cmd.split(" ", 6)[5];
					//remain is hashkey or rangedoc
					if (remain.startsWith("{"))
					    argd.append("4",Document.parse(remain));
					else
					    argd.append("4",remain);
				    }
				} else  if(cmds[1].equalsIgnoreCase("loadCollection")){
				    try{


					String sargs="-h::"+client.connection+"::-p::"+client.port+"::-c::"+cmds[2]+"::-d::" + client.database +"::";
					if (client.username!=null)
					    sargs+= "-u::"+client.username+"::-pw::" +client.password;
					List<String> args = new ArrayList<String>();
					for (String g : sargs.split("::"))
					    args.add(g);
					//ask for the rest
					String[] questions = new String[] {"Filename: ", "Post Batch Pipeline (Enter for None): ", "Post Load Pipeline (Enter for None): ", "Fuzzy Record Type (Enter for None): ","FileType: ","XML RootNode: ","Delimiter: "};
					String[] shortCodes = new String[]{"-f", "-pb", "-pl", "-r", "fileType", "rootNode", "Delimiter"};
					Boolean[] mandatory = new Boolean[]{true, false, false, false, false, false, false};
					Boolean[] isOptions = new Boolean[]{false, false, false, false, true, true, true};
					int pos=-1;
					Document options = new Document();
					for (String question : questions){
					    pos++;
					    String answer = null;
					    boolean first=true;
					    while ((mandatory[pos] && (answer==null || answer.length()==0)) || first){
						answer= lineReader.readLine(question);
						first=false;
					    }
					    if (answer!=null && answer.length()!=0){
						if (!isOptions[pos]){
						    args.add(shortCodes[pos]);
						    args.add(answer);
						} else {
						    options.append(shortCodes[pos], answer);
						}

					    }
					}
					if (options.size()>0){
					    args.add("-o");
					    args.add("'"+options.toJson()+"'");
					}

					if (!quiet){
					    StringBuilder sb = new StringBuilder();
					    for (String a: args)
						sb.append(" "+a);
					    System.out.println("Executing: monsterCLI " + sb.toString().trim());
					}

					MonsterLoadCLI.main(args.toArray(new String[args.size()]));
					return true;
				    } catch (Exception e){
					e.printStackTrace();
				    }
				}



				else{

				    if (cmds.length>2)
					for (int t=2;t<cmds.length;t++){
					    if (cmds[t].trim().startsWith("{"))
						argd.append(""+(t-1),Document.parse(cmds[t]));
					    else
						argd.append(""+(t-1), cmds[t]);
					}
				}
				if (argd.size()==0)
				    argd=null;
				return runCommand(client.currentDatabase(), null, cmds[1], argd, cmd,null,out,history,output,explain,client,quiet,delimiter);
			    }
			    else
				out.println("Use a database first, ie use dbname");
			}
		    } else if (cmds[0].equalsIgnoreCase("drop")) {
			if (cmds.length>1)
			    return runCommand(null, null, "dropDatabase", new Document("1",cmds[1]), cmd,null,out,history,output,explain,client,quiet,delimiter);
			else
			    out.println("Supply a database name");

		    } else if (cmds[0].equalsIgnoreCase("create")) {
			if (cmds.length>1)
			    return runCommand(null, null, "createDatabase", new Document("1",cmds[1]), cmd,null,out,history,output,explain,client,quiet,delimiter);
			else
			    out.println("Supply a database name");



		    } else if (cmds[0].equalsIgnoreCase("disconnect")) {
			//disconnect 
			if (cmds.length>1)
			    return runCommand(null, null, "disconnect", new Document("1",cmds[1]), cmd,null,out,history,output,explain,client,quiet,delimiter);
			else
			    out.println("Supply a database name");
		    } else if (cmds[0].equalsIgnoreCase("set")) {
			if (cmds.length>1) {
			    String what=cmds[1];
			    if (cmds.length>2) {
				String to=cmds[2];
				if (what.equalsIgnoreCase("output"))
				    client.setOutputType(OutputType.parse(to));
				else if (what.equalsIgnoreCase("quiet"))
				    quiet=Boolean.parseBoolean(to);
				else if (what.equalsIgnoreCase("explain")) {
				    explain=Boolean.parseBoolean(to);
				    return runCommand(null, null, "setDebug", new Document("1", explain), "setDebug " + explain,null,out,history,output,explain,client,quiet,delimiter);
				}
			    }
			}
		    } else if (cmds[0].equalsIgnoreCase("help")) {
			if (cmds.length == 1) {
			    out.println("help db\t\t\t\t\t\t-Database help");
			    out.println("help collection\t\t\t\t\t-Collection commands help");
			    out.println("help dq\t\t\t\t\t\t-Data Quality commands help");
			    out.println("help fuzzy\t\t\t\t\t-Fuzzy commands help");
			    out.println("set output csv|xml|json|pretty\t\t\t-Set output");
			    out.println("set quiet true|false\t\t\t\t-Set quiet mode");
			    out.println("set explain true|false\t\t\t\t-Set explain mode");
			    out.println("listDatabaseNames\t\t\t\t-List Databases");
			    out.println("connect [host] [port] [username] [password]\t-Connect/Reconnect session");
			    out.println("use dbname\t\t\t\t\t-Connect to a DB");
			    out.println("disconnect dbname\t\t\t\t-Disconnect from a DB");
			    out.println("drop dbname\t\t\t\t\t-Drop a DB");
			    out.println("create dbname\t\t\t\t\t-Create a new DB");
			    out.println("shutdown\t\t\t\t\t-Shutdown the server");
			    out.println("quit\t\t\t\t\t\t-Leave");
			}
			else if (cmds[1].equalsIgnoreCase("db")) {
			    out.println("db listCollectionNames\t\t\t\t\t\t\t\t-List Collections for current DB");
			    out.println("db close\t\t\t\t\t\t\t\t\t-Close current DB");
			    out.println("db createCollection name [{Ranges}] | [Hash]\t\t\t\t\t-Create new OPTIONALLY replicated collection");
			    out.println("db createFuzzyCollection name definition.json ruleSetDir [{Ranges}] | [Hash]\t-Create a fuzzy collection");
			    out.println("db createFuzzyCollection name [{Ranges}] | [Hash]\t\t\t\t-Create a fuzzy capable collection, with no fuzzy rules ");
			    out.println("db describeCollection name\t\t\t\t\t\t\t-Describe a collection");
			    out.println("db dropCollection name\t\t\t\t\t\t\t\t-Drop a collection");
			    out.println("db loadCollection name\t\t\t\t\t\t\t\t-Load a collection from a source (interactive)");

			} else if (cmds[1].equalsIgnoreCase("collection")) {
			    out.println("db.collectionName.deleteMany({filter: \"value\"})\t\t\t\t\t-Delete Documents matching filter document");
			    out.println("db.collectionName.find({filter: \"value\"})\t\t\t\t\t-Find Documents matching filter document");
			    out.println("db.collectionName.insertOne({....})\t\t\t\t\t\t-Create new Documents");
			    out.println("db.collectionName.save({....})\t\t\t\t\t\t\t-Create new Documents");
			    out.println("db.collectionName.createIndex({field: 1/0,...})\t\t\t\t-Create non-unique Index");
			    out.println("db.collectionName.createUniqueIndex({field: 1/0,...})\t\t\t\t-Create unique Index");

			}else if (cmds[1].equalsIgnoreCase("fuzzy")){
			    out.println("db.collectionName.saveTable({tableName: \"...\", keyField: \"...\", columns: [{ colName: \"...\"}, ...]})\t\t\t\t\t\t-Create or update a record type and specify the optional columns to use");
			    out.println("db.collectionName.deleteTable(name)\t\t\t\t\t\t\t\t\t\t\t\t\t\t-Delete a record type");
			    out.println("db.collectionName.saveConceptGroup({purposeName: \"...\", purposeType:  \"...\", .... })\t\t\t\t\t\t\t\t-Create or update a concept group (purpose) and its options");
			    out.println("db.collectionName.deleteConceptGroup(name)\t\t\t\t\t\t\t\t\t\t\t\t\t-Delete a Concept Group, and its concepts");
			    out.println("db.collectionName.saveConcept({ purposeName: \"...\", column:  \"...\", matchClass:  \"...\"})\t\t\t\t\t\t\t-Create or update a concept (Purpose Column) for a group (Purpose) and its options");
			    out.println("db.collectionName.deleteConcept(column)\t\t\t\t\t\t\t\t\t\t\t\t\t\t-Delete a Concept (Purpose Column)");
			    out.println("db.collectionName.saveConceptMapping({purposeName:  \"...\", purposeColumn:  \"...\",  tableName:  \"...\", tableColumn:  \"...\", columnOrder: 0})\t-Create or update a concept to table mapping");
			    out.println("db.collectionName.deleteConceptMapping({purposeName:  \"...\", purposeColumn:  \"...\",  tableName:  \"...\", tableColumn:  \"...\"})\t\t\t-Delete a concept to table mapping");
			    out.println("db.collectionName.saveMatchRule({order: 0 ..., rulePurpose: [ purposeName: \"\", ...]})\t\t\t\t\t\t\t\t-Create or update a match rule");
			    out.println("db.collectionName.deleteMatchRule(order)\t\t\t\t\t\t\t\t\t\t\t\t\t-Delete a match rule at position denoted by order");
			    out.println("db.collectionName.saveFuzzyIndex({indexName: \"...\", purposeName: \"...\", ...))\t\t\t\t\t\t\t\t\t-Create or update a fuzzy index");
			    out.println("db.collectionName.deleteFuzzyIndex(name)\t\t\t\t\t\t\t\t\t\t\t\t\t-Delete a fuzzy index");
			    out.println("db.collectionName.setAutoMatch(true|false)\t\t\t\t\t\t\t-Swicth on/off automatch (default off)");
			}else if (cmds[1].equalsIgnoreCase("dq")){
			    out.println("dq.setRuleSetPath(\"...path...\")\t\t\t\t\t\t\t-Update the DQ rules from the path specified");
			    out.println("dq.getRuleSetPath()\t\t\t\t\t\t\t\t-Show the permanent location of the rules (this node)");
			    out.println("dq.saveQualityRule({type:  \"...\" , parent:  \"...\", children:  [ \"\", \"\", ...]})\t-Create or update a quality rule");
			    out.println("dq.findQualityRules({...filter...})\t\t\t\t\t\t-List quality rules");
			    out.println("dq.installRules(list)\t\t\t\t\t\t-Install quality rules, list is comma sep list of rule types or empty");
			}
			return true;
		    } else {
			cmds=cmd.split("\\.",3);
			if (cmds.length>0) {
			    if (cmds[0].equalsIgnoreCase("db")) {
				if (client.currentDatabase()==null) {
				    out.println("Use a database first, ie use dbname");
				    return true;
				}
				if (cmds.length>1) {
				    client.useCollection(cmds[1]);
				    String nextPart=null;
				    if (cmds.length>2) {
					String argString =cmds[2];
					String command=argString;
					if (argString.indexOf("(")!=-1) {
					    command=argString.substring(0,argString.indexOf("("));
					    argString=argString.substring(argString.indexOf("(")+1);

					    if (argString.indexOf(")")!=-1) {


						nextPart=argString.substring(argString.indexOf(")")+1);
						argString=argString.substring(0, argString.indexOf(")"));


					    }
					}
					List args=null;
					if (argString.length()==0) {

					}
					else if (argString.trim().startsWith("{") || argString.trim().startsWith("[")) {

					    Object argso = Document.parseListOrDoc(argString);

					    if (argso instanceof String) {
						out.println("Error: " + (String)argso);
						return true;
					    }
					    args = (List)argso;
					} else {
					    //might be a single quoted string or a list of them
					    String[] arganaught = argString.split(",");
					    if (arganaught.length>0) {
						args=new ArrayList<String>();
						for (String a : arganaught){
						    if (a.trim().startsWith("\"")|| a.trim().startsWith("'"))
							args.add(a.replaceAll("\\\"|'", "").trim());
						    else {
							//number or boelan
							try{
							    args.add(Double.parseDouble(a.trim()));
							} catch (NumberFormatException n){
							    try{
								args.add(Boolean.parseBoolean(a.trim()));
							    } catch (Exception e){
								args.add(a);
							    }
							}

						    }
						}
					    }
					}


					if (args!=null && args.size()==0)
					    args=null;


					//convert to a doc

					int count=0;
					Document argDc=null;
					if (args!=null){
					    argDc = new Document();
					    for (Object argitem : args) {
						count++;
						argDc.put(""+count, argitem);
					    }
					}
					return runCommand(client.currentDatabase(), cmds[1], command, argDc, cmd, nextPart,out,history,output,explain,client,quiet,delimiter);
				    }
				}//just try to run it as it
				else return runCommand(null, null, cmds[0], null, cmd,null,out,history,output,explain,client,quiet,delimiter);
			    } else if (cmds[0].equalsIgnoreCase("dq")) {
				cmds=cmd.split("\\.",2);
				if (cmds.length>1) {
				    String command=cmds[1].substring(0,cmds[1].indexOf("("));
				    String argString =cmds[1].substring(cmds[1].indexOf("(")+1);
				    if (argString.endsWith(")"))
					argString=argString.substring(0, argString.length()-1);
				    List args=null;
				    if (argString.length()==0) {

				    }
				    else if (argString.trim().startsWith("{") || argString.trim().startsWith("[")) {
					Object argso=null;
					try {
					    argso = Document.parseListOrDocument("["+argString+"]");


					} catch (Exception e) {
					    out.println("Error: " + (String)argso);
					    return true;
					}
					args = (List)argso;
				    } else {
					//might be a single quoted string or a list of them
					String[] arganaught = argString.split(",");
					if (arganaught.length>0) {
					    args=new ArrayList<String>();
					    for (String a : arganaught)
						args.add(a.replaceAll("\\\"|'", ""));
					}
				    }


				    if (args!=null && args.size()==0)
					args=null;


				    //convert to a doc
				    int count=0;
				    Document argDc=null;
				    if (args!=null){
					argDc = new Document();
					for (Object argitem : args) {
					    count++;
					    argDc.put(""+count, argitem);
					}
				    }
				    return runCommand(null, null, command, argDc, cmd,null,out,history,output,explain,client,quiet,delimiter);

				}
				else return runCommand(null, null, cmds[0], null, cmd,null,out,history,output,explain,client,quiet,delimiter);
			    } else if (cmds[0].equalsIgnoreCase("aggregate") || cmds[0].equalsIgnoreCase("stream")) {
				if (client.currentDatabase()==null) {
				    System.err.println("Use a database first, ie use dbname");
				    return true;
				}
				if (cmds.length>1) {
				    client.useCollection(cmds[1]);
				    if (cmds.length>2) {
					List<Document> pipeline=decodeStreamCommand(cmds[2]);

					runCommand(client.currentDatabase(), cmds[1], "aggregate", new Document("1", pipeline), cmd, null, out,history,output,explain,client,quiet,delimiter);
				    }
				}

			    }




			    else return runCommand(null, null, cmds[0], null, cmd,null,out,history,output,explain,client,quiet,delimiter);

			}

		    } 
		}
	}

	return true;
    }

    private static String extractNextDocument(String argString) {
	boolean inquote=false;
	boolean indquote=false;
	int curlycount=0;
	int listcount=0;
	StringBuilder out=new StringBuilder();
	for (int i=0;i<argString.length(); i++) {
	    char c =argString.charAt(i);
	    if (c=='"')
		indquote=!indquote;
	    if (c=='\'')
		inquote=!inquote;
	    if (!indquote && !inquote) {
		if (c=='{')
		    curlycount++;
		if (c=='}')
		    curlycount--;
		if (c=='[')
		    listcount++;
		if (c==']')
		    listcount--;
	    }
	    out.append(c);
	    if (listcount==0 && curlycount==0)
		break;
	}
	return out.toString();
    }

  
    public static List<Document> decodeStreamCommand(String cmd) {
	String[] aggString =cmd.split("\\|");
	List<Document> pipeline=new ArrayList<Document>();
	for (String argString : aggString) {
	    argString=argString.replaceAll("&pipe;", "|");
	    if (argString.indexOf("(")!=-1) {
		String command = argString.substring(0,argString.indexOf("("));
		argString=argString.substring(argString.indexOf("(")+1);
		if (argString.indexOf(")")!=-1) {
		    argString=argString.substring(0, argString.lastIndexOf(")"));
		}
		//decode argString to Documents
		List args=null;
		if (argString.length()!=0 && (argString.trim().startsWith("{"))) {
		    argString=argString.trim();
		    String argStringFirst=extractNextDocument(argString);
		    Document argso = Document.parse(argStringFirst);
		    argString=argString.replace(argStringFirst, "").trim();
		    Document options=null;
		    if (argString.length()>0) {
			//options
			if (argString.startsWith(","))
			    argString=argString.substring(1).trim();
			options = Document.parse(argString);
		    }
		    Document stage=new Document("$"+command.trim(), argso);
		   
		    if (options!=null)
			stage.append("options", options);
		    pipeline.add(stage);
			
		} else {
		    //string
		    pipeline.add(new Document("$"+command.trim(), argString.trim()));
		}



	    }
	}
	return pipeline;
    }

    private static boolean runCommand(String db, String coll, String command, Document argDc, String origString, String nextParts, PrintWriter out, Queue<String> history, OutputType output, boolean explain, MonsterClient client, boolean quiet, String delimiter) {
	try {
	    interrupted=false;
	    OutputType output2=client.getOutputType();

	    if (origString.trim().length()>0)
		if (history!=null && !history.contains(origString))
		    history.add(origString);
	    Stream<Document> ret =null;

	    if (argDc!=null && argDc.containsKey("1")) {
		if (nextParts!=null)
		    argDc.append("nextParts", nextParts);
		ret = client.pushDocInnerStream(db, coll, command, argDc);
	    }
	    else
		ret = client.pushDocStream(db, coll, command, nextParts, argDc);

	    boolean hasAny=false;
	    if (ret !=null) {

		if(output2==OutputType.XML)
		    out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?><"+coll+"s>");
		AtomicBoolean first=new AtomicBoolean(true);
		ret.forEach(d -> {

		    if (interrupted)
			throw new StopIterationException("User requested cancel");
		    if (!first.getAndSet(false))
			out.println(delimiter);
		    if (output2==OutputType.Pretty)
			out.println(d.toPrettyJson());
		    else if (output2==OutputType.Json)
			out.println(d.toJson());
		    else if (output2==OutputType.XML)
			out.println(d.toXML(coll));
		    else if (output2==OutputType.CSV)
			out.println(d.toCSV());


		});

		if(output2==OutputType.XML)
		    out.println("</"+coll+"s>");




	    } else {
		System.err.println("Incorrect command syntax, session invalid, or database not used");
		return false;
	    }
	    if (false)
		throw new org.apache.thrift.transport.TTransportException();

	    if (!hasAny && !quiet)
		System.out.println("No results");
	} 
	catch (StopIterationException eof) {
	    //ignored
	    interrupted=false;
	}  
	catch (org.apache.thrift.transport.TTransportException t) {
	    out.println("Session is disconnected.");
	    return false;
	}
	catch (Exception t) {
	    out.println("Session is disconnected.");
	    return false;
	}
	return true;
    }



}
