/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.thrift.TException;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TSocket;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;

import com.entitystream.monster.thrift.Connection;
import com.entitystream.monster.thrift.Connection.Client;
import com.entitystream.monster.thrift.InvalidOperation;
import com.entitystream.identiza.entity.resolve.metadata.IIndex;
import com.entitystream.identiza.wordlist.RuleSetMap;

public class MonsterClient implements ICollection, IBasicDB, Serializable{

    private transient ConcurrentHashMap<Long, Connection.Client> clients= new ConcurrentHashMap<Long, Connection.Client>();
    private transient ConcurrentHashMap<Long, TTransport> transports = new ConcurrentHashMap<Long, TTransport>();
    private transient ConcurrentHashMap<Long, Long> connectionIDs = new ConcurrentHashMap<Long, Long>();
    private transient ConcurrentHashMap<Long, Boolean> connected=new ConcurrentHashMap<Long, Boolean>();
    private transient ConcurrentHashMap<Long, Long> touchTime = new ConcurrentHashMap<Long, Long>();



    protected String database;
    protected String collection;

    protected String password;
    protected String username;
    protected String connection;
    protected int port;
    private boolean remoteView=false;
    public boolean quiet=false;


    private ReentrantLock lock = new ReentrantLock();

    protected OutputType output=OutputType.Json;
    private int node=DBServer.getNodeNum();

    public MonsterClient (String connection, int port, String username, String password) throws NoDatabaseException {
	clients= new ConcurrentHashMap<Long, Connection.Client>();
	transports = new ConcurrentHashMap<Long, TTransport>();
	connectionIDs = new ConcurrentHashMap<Long, Long>();
	connected=new ConcurrentHashMap<Long, Boolean>();
	touchTime = new ConcurrentHashMap<Long, Long>();
	connect(connection,port, username, password);
    }

    public MonsterClient (String connection, int port, long connectionid) throws NoDatabaseException {
	clients= new ConcurrentHashMap<Long, Connection.Client>();
	transports = new ConcurrentHashMap<Long, TTransport>();
	connectionIDs = new ConcurrentHashMap<Long, Long>();
	connected=new ConcurrentHashMap<Long, Boolean>();
	touchTime = new ConcurrentHashMap<Long, Long>();
	connect(connection,port, connectionid);
    }

    public MonsterClient() {
	clients= new ConcurrentHashMap<Long, Connection.Client>();
	transports = new ConcurrentHashMap<Long, TTransport>();
	connectionIDs = new ConcurrentHashMap<Long, Long>();
	connected=new ConcurrentHashMap<Long, Boolean>();
	touchTime = new ConcurrentHashMap<Long, Long>();
    }


    public MonsterClient(String r) {
	clients= new ConcurrentHashMap<Long, Connection.Client>();
	transports = new ConcurrentHashMap<Long, TTransport>();
	connectionIDs = new ConcurrentHashMap<Long, Long>();
	connected=new ConcurrentHashMap<Long, Boolean>();
	touchTime = new ConcurrentHashMap<Long, Long>();
	int p=27018;
	String un=null, pw=null;
	if (r.split("@").length>1) {
	    un=r.split("@")[0];
	    r=r.split("@")[1];
	    if (un.indexOf(":")>-1) {
		pw=un.split(":")[1];
		un=un.split(":")[0];
	    }
	}
	if (r.split(":").length>1) {
	    p = Integer.parseInt(r.split(":")[1]);
	    r=r.split(":")[0];
	}
	connect(r,p,un,pw);
    }

    public MonsterClient clone() {
	MonsterClient client =  new MonsterClient(connection, port, username, password);
	client.database=database;
	client.collection=collection;
	client.remoteView=remoteView;
	return client;
    }

    public long getInternalConnectionID() {
	return connectionIDs.get(Thread.currentThread().getId());
    }
    
    
    public void interrupt(long connectionID) throws TException {
	//gets a new client connection to close a different one...
	 getClient().closeConnection(connectionID);
    }



    public void connect(String connection, int port, long connectionid) {
	try {
	    connected.put(Thread.currentThread().getId(), false);

	    this.connection=connection;
	    this.port=port;
	    if (transports.containsKey(Thread.currentThread().getId()))
		transports.remove(Thread.currentThread().getId());

	    reconnect(connectionid);
	} catch (TException x) {
	    throw new NoDatabaseException(x.toString());
	} 

    }

    public void connect(String connection, int port, String username, String password) {
	try {
	    connected.put(Thread.currentThread().getId(), false);
	    this.username=username;
	    this.password=password;
	    this.connection=connection;
	    this.port=port;
	    if (transports.containsKey(Thread.currentThread().getId()))
		transports.remove(Thread.currentThread().getId());

	    reconnect();
	} catch (TException x) {
	    throw new NoDatabaseException(x.toString());
	} 

    }

    long reconnect(long handlerID) throws InvalidOperation, TException {
	lock.lock();
	if (transports.containsKey(Thread.currentThread().getId()))
	    try {
		transports.remove(Thread.currentThread().getId()).close();
	    }
	catch (Exception e) 
	{};
	TSocket transport = new TSocket(connection, port);
	transport.open();
	TProtocol protocol = new  TBinaryProtocol(transport);
	transports.put(Thread.currentThread().getId(), transport);
	Client client = new Connection.Client(protocol);
	clients.put(Thread.currentThread().getId(), client);

	connectionIDs.put(Thread.currentThread().getId(), handlerID);
	lock.unlock();
	if (!quiet) {
	    if (username!=null)
		System.out.println("Connected to " + connection+":"+port + " as " + username);
	    else
		System.out.println("Connected to " + connection +":"+port);
	}
	connected.put(Thread.currentThread().getId(), true);
	touchTime.put(Thread.currentThread().getId(),System.currentTimeMillis());
	return handlerID;
    }


    boolean reconnect() throws InvalidOperation, TException {
	lock.lock();
	if (transports.containsKey(Thread.currentThread().getId()))
	    try {
		transports.remove(Thread.currentThread().getId()).close();
	    }
	catch (Exception e) 
	{};
	TSocket transport = new TSocket(connection, port);
	transport.open();
	TProtocol protocol = new  TBinaryProtocol(transport);
	transports.put(Thread.currentThread().getId(), transport);
	Client client = new Connection.Client(protocol);
	clients.put(Thread.currentThread().getId(), client);


	long connectionID=client.connect(username, password);
	connectionIDs.put(Thread.currentThread().getId(), connectionID);

	lock.unlock();
	if (connectionID!=-1) {
	    if (!quiet) {
		if (username!=null)
		    System.out.println("Connected to " + connection+":"+port + " as " + username);
		else
		    System.out.println("Connected to " + connection +":"+port);
	    }
	    connected.put(Thread.currentThread().getId(), true);
	    touchTime.put(Thread.currentThread().getId(),System.currentTimeMillis());
	    return true;
	}
	else {
	    if (username!=null)
		System.err.println("Failed to authenticate, try again");
	    else
		System.err.println("Username null but remote system expects one");
	    try {
		disconnect();
	    } catch (Exception e) {
	    }
	    return false;
	}

    }

    private Client getClient() {

	Client client = clients.get(Thread.currentThread().getId());
	if (client==null)
	    try {
		reconnect();
		client = clients.get(Thread.currentThread().getId());
	    } catch (TException e) {
		e.printStackTrace();
	    }
	if (client!=null) {
	    touchTime.put(Thread.currentThread().getId(), System.currentTimeMillis());
	}
	return client;

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#disconnect()
     */
    @Override
    public void disconnect() throws Exception {
	try {
	    lock.lock();
	    if (clients.containsKey(Thread.currentThread().getId()))
		clients.remove(Thread.currentThread().getId()).closeConnection(connectionIDs.get(Thread.currentThread().getId()));
	    lock.unlock();
	} catch (Exception e) {}
	if (transports.containsKey(Thread.currentThread().getId())) {
	    transports.remove(Thread.currentThread().getId()).close();
	}
	connected.remove(Thread.currentThread().getId());
    }

    private long runCommand(String database, String collection, String command) throws Exception {
	//System.out.println("Sending command to " + connection+":"+port+ " command="+command + ", remoteView? " + remoteView+" database="+database+" collection="+collection);
	//.out.println("locking "+ command);
	boolean ok=false;
	if (connectionIDs.containsKey(Thread.currentThread().getId()))
	    ok=connectionIDs.get(Thread.currentThread().getId())!=-1;
	long retur=-1;
	try {

	    if (!isConnected()) {
		ok=reconnect();
	    }

	    if (ok) {
		lock.lock();
		try {
		    retur= getClient().command(connectionIDs.get(Thread.currentThread().getId()), database, collection, command, remoteView);
		} catch (TTransportException ttt) {
		    ok=reconnect();
		    if (ok)
			retur= getClient().command(connectionIDs.get(Thread.currentThread().getId()), database, collection, command, remoteView);

		}
		lock.unlock();
	    }
	    else retur=-1;

	} catch (Exception e) {
	    System.err.println("I'm trying to send an command to " + connectionIDs.get(Thread.currentThread().getId()) + ", but a error occurs: "+e.toString());
	    disconnect();
	    if (lock.isLocked())
		lock.unlock();
	    retur= -1;
	}
	//System.out.println("unlocking "+ command);


	return retur;
    }

    public static Document toDocument(ByteBuffer bb) {

	if (bb.capacity()!=0) {
	    String string = new String(bb.array());

	    Document out = null;
	    if (string!=null) {
		out = Document.parse(string);
	    }
	    return out;
	}
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#useDatabase(java.lang.String)
     */
    @Override
    public IBasicDB useDatabase(String db) {
	Iterator it = this.listDatabaseNames().iterator();
	while (it.hasNext()) {
	    Document d = (Document) it.next();
	    if (d.containsKey("value") && d.getString("value").equalsIgnoreCase(db)) {
		this.database=db;
		return this;
	    }
	}
	createDatabase(db);
	return this;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#useCollection(java.lang.String)
     */
    @Override
    public IBasicDB useCollection(String coll) {
	this.collection=coll;

	return this;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#listDatabaseNames()
     */
    @Override
    public DBCursor listDatabaseNames() {
	BasicDBList list = null;
	try {
	    long cursor = runCommand(null, null, (new Document("MethodName", "listDatabaseNames")).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    Document d = toDocument(bb);
		    list = new BasicDBList(d.getList("list"));
		}
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor(list);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#listCollectionNames()
     */
    @Override
    public DBCursor listCollectionNames() {
	BasicDBList list = null;
	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "listCollectionNames")).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    Document d = toDocument(bb);
		    if (d !=null)
			list = new BasicDBList(d.getList("list"));
		}
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor(list);

    }

    @Override
    public String createDatabase(String db) {


	try {
	    long cursor = runCommand( null, null, (new Document("MethodName", "createDatabase").append("1",db)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String dbname = new String(bb.array());
		    this.database=dbname;
		    return dbname;
		} else this.database=db;
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;


    }

    @Override
    public String createCollection(String coll) {

	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "getCollection").append("1",coll)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} else this.collection=coll;
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;


    }

    @Override 
    public Document describeCollection(String coll) {
	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "describeCollection").append("1",coll)).toJson());
	    if (cursor!=-1) {

		ByteBuffer bb = cursorNext( cursor, -1);

		if (bb.capacity()!=0) {
		    String doc = new String(bb.array());
		    return Document.parse(doc);
		} 
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public String createCollection(String coll, String hashKey) {

	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "getCollection").append("1",coll).append("2",hashKey)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} else this.collection=coll;
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public String createCollection(String coll, Document ranges) {

	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "createCollection").append("1",coll).append("2",ranges)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} else this.collection=coll;
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;


    }

    @Override
    public String createFuzzyCollection(String name, String definitionFile, String ruleSetPath) {
	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "createFuzzyCollection").append("1",name).append("2", definitionFile).append("3", ruleSetPath)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} 
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;

    }

    @Override
    public String createFuzzyCollection(String name, String definitionFile, String ruleSetPath, String hashKey) {
	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "createFuzzyCollection").append("1",name).append("2", definitionFile).append("3", ruleSetPath).append("4",hashKey)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} 
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;

    }

    @Override
    public String createFuzzyCollection(String name, String definitionFile, String ruleSetPath, Document ranges) {
	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "createFuzzyCollection").append("1",name).append("2", definitionFile).append("3", ruleSetPath).append("4",ranges)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} 
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;

    }

    @Override
    public String createCollection(Document collDoc) {

	try {
	    long cursor = runCommand( database, null, (new Document("MethodName", "createCollection").append("1",collDoc)).toJson());
	    if (cursor!=-1) {
		ByteBuffer bb = cursorNext( cursor, -1);
		if (bb.capacity()!=0) {
		    String collname = new String(bb.array());
		    this.collection=collname;
		    return collection;
		} 
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;


    }

    private ByteBuffer cursorNext(long cursor, int i) throws InvalidOperation, TException {

	try {
	    lock.lock();
	    long id=-1;
	    if (connectionIDs.containsKey(Thread.currentThread().getId()))
		id=connectionIDs.get(Thread.currentThread().getId());
	    ByteBuffer bb=null;
	    if (id!=-1)
		bb = getClient().cursorNext(id, cursor, -1);
	    lock.unlock();
	    if (bb==null)
		bb=ByteBuffer.allocate(0);
	    return bb;
	} catch (StopIterationException ignore) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    return ByteBuffer.allocate(0);
	} catch (TTransportException e) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    System.out.println("Transport Exception has occurred, like the server has gone offline, or you have been disconnected");
	    return ByteBuffer.allocate(0);

	} catch (TException e) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    System.out.println("Session has been disconnected");

	    return ByteBuffer.allocate(0);

	}
    }


    public Stream<Document> remoteGetStream(long connectionID, long cursorID) {

	try {

	    //start the browser here - but point it to the server at this cursor

	    final MonsterClient client = this;
	    AtomicBoolean shouldreturnnull=new AtomicBoolean(true);
	    try {
		Stream<Document> ret= Stream.generate(new Supplier<Object>() {
		    @Override
		    public Object get(){
			Object o=null;

			try {

			    ByteBuffer bb = remoteGetCursorNext(connectionID, cursorID, 100);
			    if (bb.capacity()!=0) {
				String ss = new String(bb.array());
				o= Mocument.parseListOrDocument(client, ss);
				shouldreturnnull.set(false);
			    } else return null;

			} catch (Exception ex) {
			    ex.printStackTrace();
			}
			return o;
		    }

		}).takeWhile(o -> o!=null)
			.map(o -> {
			    List<Document> l=null;
			    if (o instanceof BasicDBList)
				l=((BasicDBList)o).toList();
			    if (o instanceof Document)
				l=Collections.singletonList(new Mocument((Document)o, client));
			    return l.stream();
			})
			.filter(o -> o!=null)
			.flatMap(in -> in);
		if (shouldreturnnull.get())
		    return null;
		else
		    return ret;
	    } catch (Exception eof) {
		System.out.println("EOF");
		//ignored
	    }



	} catch (Exception e) {
	    e.printStackTrace();
	}
	return Stream.empty();


    }

    private ByteBuffer remoteGetCursorNext(long id, long cursor, int i) throws InvalidOperation, TException {

	try {
	    lock.lock();
	    ByteBuffer bb=null;
	    if (id!=-1)
		bb = getClient().cursorNext(id, cursor, -1);
	    lock.unlock();
	    if (bb==null)
		bb=ByteBuffer.allocate(0);
	    return bb;
	} catch (StopIterationException ignore) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    return ByteBuffer.allocate(0);
	} catch (TTransportException e) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    System.out.println("Transport Exception has occurred, like the server has gone offline, or you have been disconnected");
	    return ByteBuffer.allocate(0);

	} catch (TException e) {
	    if (lock.isHeldByCurrentThread() || lock.isLocked())
		lock.unlock();
	    System.out.println("Exception has occurred, like the server has gone offline, or you have been disconnected");

	    return ByteBuffer.allocate(0);

	}
    }


    public DBCursor pushDocInner(String database, String collection,  String command, Document send) throws Exception {



	send.append("MethodName", command);
	BasicDBList list = null;



	long cursor = runCommand( database, collection, send.toJson());
	if (cursor!=-1) {
	    //start the browser here - but point it to the server at this cursor
	    if (output==OutputType.BROWSER)
		browser(cursor, connectionIDs.get(Thread.currentThread().getId()));
	    else {

		try {
		    Document dd = new Document();
		    list = new BasicDBList();
		    while (dd!=null) {
			ByteBuffer bb = cursorNext( cursor, 100);
			if (bb.capacity()!=0) {
			    String ss = new String(bb.array());
			    Object o = Mocument.parseListOrDocument(this, ss);
			    if (o instanceof BasicDBList)
				list.addAll((BasicDBList)o);
			    if (o instanceof Document)
				list.add(new Mocument(dd, this));
			} else dd=null;
		    }
		} catch (InvalidOperation ioe) {
		    ioe.printStackTrace();
		}

	    }
	} else return new DBCursor();

	return new DBCursor(list);
    }


    public Stream<Document> pushDocInnerStream(String database, String collection,  String command, Document send) {
	send.append("MethodName", command);
	try {
	    long cursor = runCommand( database, collection, send.toJson());
	    if (cursor!=-1) {
		//start the browser here - but point it to the server at this cursor
		if (output==OutputType.BROWSER)
		    browser(cursor, connectionIDs.get(Thread.currentThread().getId()));
		else {
		    final MonsterClient client = this;
		    try {
			return Stream.generate(new Supplier<Object>() {
			    @Override
			    public Object get(){
				Object o=null;

				try {

				    ByteBuffer bb = cursorNext( cursor, 100);
				    if (bb.capacity()!=0) {
					String ss = new String(bb.array());
					o= Mocument.parseListOrDocument(client, ss);

				    } else return null;

				} catch (Exception ex) {
				    ex.printStackTrace();
				}
				return o;
			    }

			}).takeWhile(o -> o!=null)
				.map(o -> {
				    List<Document> l=null;
				    if (o instanceof BasicDBList)
					l=((BasicDBList)o).toList();
				    if (o instanceof Document)
					l=Collections.singletonList(new Mocument((Document)o, client));
				    return l.stream();
				})
				.filter(o -> o!=null)
				.flatMap(in -> in);
		    } catch (Exception eof) {
			System.out.println("EOF");
			//ignored
		    }

		}
	    } else System.out.println("No cursor from " + send.toString());
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return Stream.empty();


    }


    public DBCursor pushDoc(String database, String collection,  String command, Object... docs) {
	return pushDoc(database, collection, command, null, docs);
    }

    public DBCursor pushDoc(String database, String collection,  String command, String nextParts, Object... docs) {

	Document send = new Document("MethodName", command);
	if (nextParts!=null)
	    send.append("nextParts", nextParts);
	try {

	    int pos=0;
	    if (docs!=null)
		for (Object doc: docs) {
		    if (doc!=null) {

			pos++;
			send.append(""+pos, doc);
		    }

		}
	    return pushDocInner(database, collection, command, send);
	} catch (Exception e) {
	    System.out.println(e.toString());
	    e.printStackTrace();
	}
	return new DBCursor();
    }


    public Stream<Document> pushDocStream(String database, String collection,  String command, String nextParts, Object... docs) {

	Document send = new Document("MethodName", command);
	if (nextParts!=null)
	    send.append("nextParts", nextParts);
	try {

	    int pos=0;
	    if (docs!=null)
		for (Object doc: docs) {
		    if (doc!=null) {

			pos++;
			send.append(""+pos, doc);
		    }

		}
	    return pushDocInnerStream(database, collection, command, send);
	} catch (Exception e) {
	    System.out.println(e.toString());
	    e.printStackTrace();
	}
	return Stream.empty();
    }

    /**
     * @param collection
     */
    private void browser(long cursor, long connID) {

	//generate a unique URL on the remote server for this collection and open the browser pointing to it
	String url = "http://"+username+":"+password+"@" + connection + ":" + (port+1) +"/api/" + database+ "/" +collection +"?cursor="+cursor;
	if (username==null)
	    url = "http://"+ connection + ":" + (port+1) +"/api/" + database+ "/" +collection +"?cursor="+cursor+"&connection="+connID;
	String os = System.getProperty("os.name").toLowerCase();
	Runtime rt = Runtime.getRuntime();

	try{

	    if (os.indexOf( "win" ) >= 0) {

		// this doesn't support showing urls in the form of "page.html#nameLink" 
		rt.exec( "rundll32 url.dll,FileProtocolHandler " + url);

	    } else if (os.indexOf( "mac" ) >= 0) {

		rt.exec( "open " + url);

	    } else if (os.indexOf( "nix") >=0 || os.indexOf( "nux") >=0) {

		// Do a best guess on unix until we get a platform independent way
		// Build a list of browsers to try, in this order.
		String[] browsers = {"epiphany", "firefox", "mozilla", "konqueror",
			"netscape","opera","links","lynx"};

		// Build a command string which looks like "browser1 "url" || browser2 "url" ||..."
		StringBuffer cmd = new StringBuffer();
		for (int i=0; i<browsers.length; i++)
		    cmd.append( (i==0  ? "" : " || " ) + browsers[i] +" \"" + url + "\" ");

		rt.exec(new String[] { "sh", "-c", cmd.toString() });

	    } else {
		return;
	    }
	}catch (Exception e){
	    return;
	}
	return;		


    }

    @Override
    public void createIndex(Document fields, Document options) {
	pushDoc(database, collection, "createIndex",null,  fields, options);
    }

    @Override
    public void createIndex(Document fields) {
	try {
	    pushDoc(database, collection, "createIndex", null, fields);
	} catch (Exception e) {

	    e.printStackTrace();
	}

    }

    @Override
    public void createUniqueIndex(Document fields) {
	try {
	    pushDoc(database, collection, "createUniqueIndex", null, fields);
	} catch (Exception e) {

	    e.printStackTrace();
	}

    }

    @Override
    public DBCursor find(Document query) {
	try {
	    return pushDoc(database, collection, "find", null, query);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }



    @Override
    public DBCursor find() {
	try {
	    return pushDoc(database, collection, "find", null, new Document());
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }

    @Override
    public DBCursor findFuzzy(Document filter) {
	try {
	    return pushDoc(database, collection, "findFuzzy",null,  filter);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#findFuzzy(java.lang.String)
     */
    @Override
    public DBCursor findFuzzy(String textQuery) {
	try {
	    return pushDoc(database, collection, "findFuzzy",null,  textQuery);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }


    @Override
    public Document save(Document doc) {
	try {
	    return pushDoc(database, collection, "save",null,  doc).first();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }




    @Override
    public Document insertOne(Document newDocument) {
	try {
	    return (Document)pushDoc(database, collection, "insertOne", null, newDocument).first();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public long count(Document query) {
	try { 
	    Document d = pushDoc(database, collection, "count",null,  query).first();
	    if (d!=null)
		return d.getInteger("Number");
	    else return 0;
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return 0;
    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement, Document options) {
	return pushDoc(database, collection, "findOneAndReplace",null,  filter, replacement, options).first();
    }

    @Override
    public Document findOneAndReplace(Document filter, Document replacement) {
	try {
	    return pushDoc(database, collection, "findOneAndReplace",null,  filter,replacement).first();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments) {
	return pushDoc(database, collection, "findOneAndUpdate",null,  filter, amendments).first();

    }

    @Override
    public Document findOneAndUpdate(Document filter, Document amendments, Document options) {
	return pushDoc(database, collection, "findOneAndUpdate",null,  filter, amendments, options).first();
    }

    @Override
    public Document updateOne(Document filter, Document amendments) {
	return pushDoc(database, collection, "updateOne",null,  filter, amendments).first();
    }

    @Override
    public Document updateOne(Document filter, Document amendments, Document options) {
	return pushDoc(database, collection, "updateOne",null,  filter, amendments, options).first();
    }

    @Override
    public int updateMany(Document filter, Document amendments, Document options) {
	return pushDoc(database, collection, "updateMany", null, filter, amendments, options).count();
    }

    @Override
    public int updateMany(Document filter, Document amendments) {
	return pushDoc(database, collection, "updateMany", null, filter, amendments, new Document()).count();
    }

    @Override
    public int insertMany(List<Document> listRecs) {
	return pushDoc(database, collection, "insertMany", null, listRecs).first().getInteger("Number", 0);
    }


    @Override
    public Document findOneAndDelete(Document filter) {
	try {
	    return pushDoc(database, collection, "findOneAndDelete", null, filter).first();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public Document deleteOne(Document filter) {
	try {
	    return pushDoc(database, collection, "deleteOne", null, filter).first();
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    @Override
    public int deleteMany(Document filter) {
	try {
	    return pushDoc(database, collection, "deleteMany", null, filter).first().getInteger("Number");
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return 0;
    }

    @Override
    public Stream<Document> aggregate(List<Document> in, List<Document> pipeline) {
	return pushDocStream(database, collection, "aggregate",null,  in, pipeline, new Document());
    }
    
    @Override
    public Document aggregateMetadata(List<Document> in, List<Document> pipeline) {
	return pushDocStream(database, collection, "aggregateMetadata",null,  in, pipeline, new Document()).findFirst().get();
    }

    @Override
    public Document aggregateMetadata(List<Document> pipeline) {
	return pushDocStream(database, collection, "aggregateMetadata",null, pipeline, new Document()).findFirst().get();
    }
    
    @Override
    public Stream<Document> aggregate(List<Document> pipeline, Document options) {
	return pushDocStream(database, collection, "aggregate", null, pipeline,options);
    }

    @Override
    public Stream<Document> aggregate(List<Document> pipeline) {
	return pushDocStream(database, collection, "aggregate", null, pipeline);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#listIndexes()
     */
    @Override
    public DBCursor listIndexes() {
	return pushDoc(database, collection, "listIndexes");
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#dropCollection()
     */
    @Override
    public void dropCollection() {
	pushDoc(database, null, "dropCollection", null, collection);

    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#dropDatabase()
     */
    @Override
    public void dropDatabase() {
	pushDoc(database, null, "drop");

    }

    @Override
    public String currentDatabase() {

	return database;
    }


    @Override
    public void rebuildIndex(String name) {
	pushDoc(database, null, "rebuildIndex", null, name);

    }


    @Override
    public void dropIndex(String name) {
	pushDoc(database, null, "dropIndex", null, name);

    }


    @Override
    public int saveMany(List<Document> records) {
	try {
	    pushDoc(database, collection, "saveMany", null, records);
	    return 0;
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return 0;
    }


    public boolean isConnected() {

	return transports.containsKey(Thread.currentThread().getId()) 
		&& transports.get(Thread.currentThread().getId()).peek() 
		&& connected.containsKey(Thread.currentThread().getId()) 
		&& connected.get(Thread.currentThread().getId()) 
		&& (System.currentTimeMillis() - touchTime.get(Thread.currentThread().getId()) < Session.MAXAGE);
    }


    @Override
    public void setExplaining(boolean explain) {
	try {
	    pushDoc(database, collection, "setExplaining", null, explain);

	} catch (Exception e) {

	    e.printStackTrace();
	}


    }

    @Override
    public void addTrigger(String name) {
	try {
	    pushDoc(database, collection, "addTrigger", null, name);

	} catch (Exception e) {

	    e.printStackTrace();
	}


    }

    @Override
    public DBCursor find(Document query, Document explain) {
	try {
	    return pushDoc(database, collection, "find", null, query,explain);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }


    @Override
    public DBCursor executeCommand(String command, User user, Session session) {
	try {
	    if (!command.contains("MethodName"))
		return pushDoc(database, collection, command);
	    else {
		Document d = Document.parse(command);
		if (d!=null)
		    return pushDocInner(database, collection, d.getString("MethodName"), d);
	    }
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }


    @Override
    public void drop() {
	try {
	    pushDoc(database, collection, "drop");
	} catch (Exception e) {

	    e.printStackTrace();
	}


    }


    /**
     * 
     */
    public void shutdown() {
	try {
	    pushDoc(null, null, "shutdown");
	} catch (Exception e) {

	    e.printStackTrace();
	}

    }


    @Override
    public <TResult> Iterable<TResult> distinct(String fieldName, Class<TResult> resultClass) {
	try {
	    return (Iterable<TResult>) pushDoc(database, collection, "distinct", null, fieldName, resultClass);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }




    @Override
    public Document getDocument(Object pointer) {
	return pushDoc(database, collection, "getDocument",null,  pointer).first();

    }

    @Override
    public Document getIndexPrefixSubMap(String indexName, String key, boolean b) {
	return pushDoc(database, collection, "getIndexPrefixSubMap", null, indexName, key, b).first();

    }


    @Override
    public Document keyCount(String indexName) {
	return pushDoc(database, collection, "keyCount", null, indexName).first();
    }
    @Override
    public Document conceptCount(String indexName) {
	return pushDoc(database, collection, "conceptCount", null, indexName).first();
    }

    @Override
    public Document getStandardised(String indexName, String id) {
	return pushDoc(database, collection, "getStandardised", null, indexName, id).first();
    }


    public boolean isRemoteView() {
	return remoteView;
    }


    public MonsterClient setRemoteView(boolean remoteView) {
	this.remoteView = remoteView;
	return this;
    }

    @Override
    public int getNodeNum() {
	return pushDoc(null, null, "getNodeNum").first().getInteger("Number");
	
    }


    @Override
    public void saveTable(Document def) {
	pushDoc(database, collection, "saveTable", null, def);

    }


    @Override
    public void setAutoMatch(boolean def) {
	pushDoc(database, collection, "setAutoMatch", null, def);

    }

    @Override
    public Document getTable(String def) {
	return pushDoc(database, collection, "getTable", null, def).first();

    }


    @Override
    public void deleteTable(String name) {
	pushDoc(database, collection, "deleteTable", null, name);

    }


    @Override
    public void saveConceptGroup(Document def) {
	pushDoc(database, collection, "saveConceptGroup", null, def);

    }


    @Override
    public void deleteConceptGroup(String name) {
	pushDoc(database, collection, "deleteConceptGroup", null, name);

    }


    @Override
    public void saveConcept(Document def) {
	pushDoc(database, collection, "saveConcept", null, def);

    }


    @Override
    public void deleteConcept(String purposeName, String purposeColumn) {
	pushDoc(database, collection, "deleteConcept", null, purposeName, purposeColumn);

    }


    @Override
    public void saveConceptMapping(Document def) {
	pushDoc(database, collection, "saveConceptMapping", null, def);

    }


    @Override
    public void deleteConceptMapping(Document def) {
	pushDoc(database, collection, "deleteConceptMapping", null, def);

    }


    @Override
    public void saveQualityRule(Document def) {
	pushDoc(database, null, "saveQualityRule", null, def);

    }


    @Override
    public DBCursor findQualityRules(Document def) {
	return pushDoc(database, null, "findQualityRules", null, def);

    }


    @Override
    public void saveMatchRule(Document def) {
	pushDoc(database, collection, "saveMatchRule", null, def);

    }
    @Override
    public void saveMatchRules(List<Document> defs) {
	pushDoc(database, collection, "saveMatchRules", null, defs);

    }

    @Override
    public void deleteMatchRule(long order) {
	pushDoc(database, collection, "deleteMatchRule", null, order);

    }


    @Override
    public void saveFuzzyIndex(Document def) {
	pushDoc(database, collection, "saveFuzzyIndex", null, def);

    }


    @Override
    public void deleteFuzzyIndex(String name) {
	pushDoc(database, collection, "deleteFuzzyIndex", null, name);

    }

    @Override
    public void removeFuzzy() {
	pushDoc(database, collection, "removeFuzzy",null);
    }

    @Override
    public void setRuleSetPath(String path) {
	pushDoc(null, null, "setRuleSetPath", null, path);
    }

    @Override
    public void saveRuleSet(String path) {
	pushDoc(null, null, "saveRuleSet", null, path);
    }	


    @Override
    public void setRuleSet(RuleSetMap map) {
	pushDoc(null, null, "setRuleSet", null, map);
    }	

    @Override
    public Document getDefinition() {
	return pushDoc(database, collection, "getDefinition", null).first();
    }

   
    @Override
    public void updateDefinition(Document def) {
	if (def==null)
	    def=new Document();
	pushDoc(database, collection, "updateDefinition", null, def);

    }

  
    @Override
    public void updateDelta(Document def) {
	if (def==null)
	    def=new Document();
	pushDoc(database, collection, "updateDelta", null, def);

    }
    
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findRelationships(Document filter) {
	if (filter==null)
	    filter=new Document();
	return pushDoc(database, collection, "findRelationships", null, filter);
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public DBCursor findTasks(Document filter) {
	if (filter==null)
	    filter=new Document();
	return pushDoc(database, collection, "findTasks", null, filter);
    }



    public void resolveTask(Document document) {
	pushDoc(database, collection, "resolveTask", null, document);
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#addRelationship(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveRelationship(Document relationship) {
	return  pushDoc(database, collection, "saveRelationship", null, relationship).first();
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#aggregateTasks(java.util.ArrayList)
     */
    @Override
    public DBCursor aggregateTasks(ArrayList<Document> andlist) {
	return pushDoc(database, collection, "aggregateTasks", null, andlist);
    }


    @Override
    public Document updateTask(String id, Document doc) {
	return pushDoc(database, collection, "updateTask", null, id, doc).first();

    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#relsDistinct(java.lang.String, java.lang.Class)
     */
    @Override
    public <TResult> Iterable<TResult> distinctRelationship(String fieldName, Class<TResult> resultClass) {
	try {
	    return (Iterable<TResult>) pushDoc(database, collection, "distinctRelationship", null, fieldName, resultClass);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#updateRelationship(com.entitystream.monster.db.Document, com.entitystream.monster.db.Document, com.entitystream.monster.db.Document)
     */
    @Override
    public Document updateRelationship(Document document, Document setUpdate, Document options) {
	return pushDoc(database, collection, "updateRelationship", null, document, setUpdate,options).first();
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#peekQueue()
     */
    @Override
    public DBCursor peekQueue() {
	return pushDoc(database, collection, "peekQueue", null);
    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#deleteTasks(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteTasks(Document document) {
	pushDoc(database, collection, "deleteTasks", document);

    }


    /* (non-Javadoc)
     * @see com.entitystream.monster.db.IBasicDB#deleteRelationships(com.entitystream.monster.db.Document)
     */
    @Override
    public void deleteRelationships(Document document) {
	pushDoc(database, collection, "deleteRelationships", document);
    }


    /**
     * @param code
     */
    @Override
    public void setScriptCode(String code) {
	pushDoc(database, null, "setScriptCode", code);

    }

    @Override
    public void setScriptPath(String path) {
	pushDoc(database, null, "setScriptPath", path);

    }


    /**
     * @return
     */
    public OutputType getOutputType() {

	return output;
    }

    public void setOutputType(OutputType o) {

	output=o;
    }


    @Override
    public Stream<Document> findStream(Document query) {
	try {
	    return pushDocStream(database, collection, "findStream", null, query);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getReplicaConnections()
     */
    @Override
    public Map<Integer, ICollection> getReplicaConnections() {
	return null;
    }
    
    @Override
    public Stream<Document> getNodes() {
	try {
	    return pushDocStream(database, collection, "getNodes", null);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
	
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#match(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> match(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("match", asDocument, in, options);
    }
    
    private Stream<Document> remoteAggregate(String command, Object asDocument, Stream<Document> in, Document options){
	try {
	    //package up in as a callable cursor on this side, pass the cursor id to the other side and then call the remote side to read it into a function
	    DBCursor curs = new DBCursor(in); 
	    long connectionID=DBServer.getHandler().connect(username, password);
	    long cursorID = DBServer.getHandler().addCursor(connectionID, curs, asDocument.toString());
	    Stream<Document> output= pushDocStream(database, collection, command, null, asDocument, node, connectionID, cursorID, options);
	    return output;
	    
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#analyse(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> analyse(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("analyse", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#limit(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> limit(long long1, Stream<Document> in, Document options) {
	return remoteAggregate("limit", long1, in, options);
	
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#lookup(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> lookup(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("lookup", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#join(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> join(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("join", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#minus(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> minus(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("minus", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#skip(long, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> skip(long long1, Stream<Document> in, Document options) {
	return remoteAggregate("skip", long1, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#group(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> group(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("group", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#out(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> out(String string, Stream<Document> in, Document options) {
	return remoteAggregate("out", string, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#sort(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> sort(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("sort", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#bucket(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> bucket(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("bucket", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzySearch(java.lang.String, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> fuzzySearch(String string, Stream<Document> in, Document options) {
	return remoteAggregate("fuzzySearch", string, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> fuzzyMatch(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("fuzzyMatch", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#unwind(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> unwind(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("unwind", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#spinOut(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> spinOut(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("spinOut", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getRelated(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> getRelated(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("getRelated", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#rematch(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> rematch(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("rematch", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierBuild(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierBuild(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("classifierBuild", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierPredict(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierPredict(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("classifierPredict", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#arrf(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> arrf(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("arrf", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#classifierTree(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> classifierTree(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("classifierTree", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#coerce(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> coerce(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("coerce", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#compare(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> compare(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("compare", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#count(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> count(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("count", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#first(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> first(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("first", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#last(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> last(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("last", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#between(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> between(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("between", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#cluster(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> cluster(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("cluster", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#project(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> project(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("project", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#evaluate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> evaluate(Document asDocument, Stream<Document> in, Document options) {
	return remoteAggregate("evaluate", asDocument, in, options);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#split(java.util.List, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Map<Document, List<Document>> split(List<Document> list, Stream<Document> in, Document options) {
	try {
	    throw new Exception("split function cannot be called remotely");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDatabase()
     */
    @Override
    public Database getDatabase() {
	try {
	    throw new Exception("getDatabase function cannot be called directly");
	} catch(Exception e) {
	    e.printStackTrace();
	}
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getTrigger()
     */
    @Override
    public String getTrigger() {

	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#fuzzyMatch(com.entitystream.monster.db.Document, java.util.List)
     */
    @Override
    public Document fuzzyMatch(Document matchNode, List<IIndex> matchIndexes) {
	return pushDoc(database, collection, "fuzzyMatch", matchNode, matchIndexes).first();

    }

    /**
     * @param value
     */
    public void connect(String value) {
	//monster://root:AKspt9SUT&Z7724H@mongodb:27018
	URI url;
	try {
	    url = new URI(value);
	    if (url.getScheme().equalsIgnoreCase("monsterdb")) {
		String host=url.getHost();
		int port = url.getPort();
		if (port==-1)
		    port=27018;
		String username=url.getUserInfo();
		String password=null;
		if (username!=null && username.indexOf(":")>-1) {
		  password=username.split(":")[1];
		  username=username.split(":")[0];
		}
		connect(host, port, username, password);
	    }
	} catch (URISyntaxException e) {
	    // TODO Auto-generated catch block
	    e.printStackTrace();
	}
	
	
    }

    /**
     * @return
     */
    public String getAddress() {
	
	return connection;
    }

  
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#traverseTop(com.entitystream.monster.db.Document, java.lang.String)
     */
    @Override
    public DBCursor traverseTop(Document from, String relType) {
	return pushDoc(database, collection, "traverseTop", null, from, relType);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#writeRel(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> writeRel(Document options, Stream<Document> in, Document goptions) {
	return remoteAggregate("writeRel", options, in, goptions);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#script(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> script(Document scriptStatements, Stream<Document> in, Document stageoptions) {
	return null;
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getMatchTypes()
     */
    @Override
    public DBCursor getMatchTypes() {
	try {
	    return pushDoc(database, collection, "getMatchTypes", null);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveFuzzyIndexes(java.util.List)
     */
    @Override
    public void saveFuzzyIndexes(List<Document> def) {
	pushDoc(database, collection, "saveFuzzyIndexes", def);
	
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlows()
     */
    @Override
    public DBCursor getFlows() {
	try {
	    return pushDoc(database, collection, "getFlows", null);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }
    
    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getBooks()
     */
    @Override
    public DBCursor getBooks() {
	try {
	    return pushDoc(database, collection, "getBooks", null);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getFlow(java.lang.String)
     */
    @Override
    public DBCursor getFlow(String name) {
	try {
	    return pushDoc(database, collection, "getFlow", null, name);
	} catch (Exception e) {

	    e.printStackTrace();
	}
	return new DBCursor();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#validate(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> validate(String type, Stream<Document> in, Document goptions){
	return remoteAggregate("validate", type, in, goptions);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#task(com.entitystream.monster.db.Document, java.util.stream.Stream, com.entitystream.monster.db.Document)
     */
    @Override
    public Stream<Document> task(Document options, Stream<Document> in, Document globalOptions) {
	return remoteAggregate("task", options, in, globalOptions);
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#saveTask(com.entitystream.monster.db.Document)
     */
    @Override
    public Document saveTask(Document task) {
	return  pushDoc(database, collection, "saveTask", null, task).first();
    }

    /* (non-Javadoc)
     * @see com.entitystream.monster.db.ICollection#getDeltas()
     */
    @Override
    public Stream<Document> getDeltas() {
	return  pushDoc(database, collection, "getDeltas", null).stream();
    }

  
  
  
}
