/**
 *
	MonsterDB - Collection Based Database with fuzzy matching
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;

import com.entitystream.identiza.db.INode;
import com.entitystream.identiza.db.Node;
import com.entitystream.identiza.entity.extract.TizaReader;
import com.entitystream.identiza.entity.resolve.metadata.ITable;
import com.entitystream.identiza.entity.resolve.metadata.Table;
import com.entitystream.identiza.entity.resolve.metadata.TableColumn;

import picocli.CommandLine;
import picocli.CommandLine.Option;

public class MonsterLoadCLI extends MonsterClient implements Runnable {
	@Option(names = {"-h", "--host"}, required = true, description = "Remote host IP or DNS Name")
	private  String host;
	@Option(names = {"-p", "--port"}, required = false, description = "Remote host port (default 27018)")
	private  int port;
	@Option(names = {"-u", "--user"}, required = false, description = "The username to connect with")
	private  String username;
	@Option(names = {"-pw", "--password"}, required = false, description = "The usernames password")
	private  String password;
	@Option(names = {"-c", "--collection"}, required = true, description = "Collection to load")
	private  String collection;
	@Option(names = {"-d", "--db"}, required = true, description = "DB to load")
	private  String db;
	@Option(names = {"-f", "--file"}, required = true, description = "File to load")
	private  String filename;
	@Option(names = {"-r", "--recordType"}, required = false, description = "Record Type to load (fuzzy tables only)")
	private  String tableName;
	@Option(names = {"-o", "--options"}, required = false, description = "Load options")
	private  String propertiestr;
	@Option(names = {"-pbp", "--postBatchPipeline"}, required = false, description = "Post Batch Pipeline")
	private  String postBatchPipeline;
	@Option(names = {"-plp", "--postLoadPipeline"}, required = false, description = "Post Load Pipeline")
	private  String postLoadPipeline;
	@Option(names = { "-q", "--quiet"}, description = "Be less verbose")
	private boolean quiet=false;
	private boolean terminate=false;

	public MonsterLoadCLI() throws NoDatabaseException {
		super();
	}

	public MonsterLoadCLI(String host, int port, String username, String password, String collection, String database, String fileName, String tableName, String options, String postBatchPipeline, String postLoadPipeline, boolean quiet, boolean terminate) throws NoDatabaseException {
		super();
		this.host=host;
		this.collection=collection;
		this.port=port;
		this.username=username;
		this.password=password;
		this.db=database;
		this.filename=fileName;
		this.tableName=tableName;
		this.propertiestr=options;
		this.postBatchPipeline=postBatchPipeline;
		this.postLoadPipeline=postLoadPipeline;
		this.quiet=quiet;
		this.terminate=terminate;
		new Thread (this).start();
	}

	public static void main(String... args) {
		CommandLine.run(new MonsterLoadCLI(), System.err, args);
	}

	@Override
	public void run() {

		try {
			this.connect(host, port, username, password);
			this.useDatabase(db);
			this.useCollection(collection);
			//loadFile fuzzy2 /Users/roberthaynes/Documents/data/leismall.xml GLEIF {fileType: "XML", rootNode: "LEIRecord"}
			
			System.out.println(propertiestr);
			
			Document properties = Document.parse(propertiestr);

			BasicDBList postBatch = null;
			if (postBatchPipeline!=null)
				postBatch = (BasicDBList) Document.parseListOrDoc(postBatchPipeline);
			BasicDBList postLoad =null;
			if (postLoadPipeline!=null)
				postLoad = (BasicDBList) Document.parseListOrDoc(postLoadPipeline);

			Properties props = new Properties();
			if (properties!=null)
			   props.putAll(properties.toMap());
			TizaReader tiza = TizaReader.createReader(filename, props, "");
			if (tiza!=null && tiza.isValid()) {
				try {
					long start = System.currentTimeMillis();
					ITable table = null;
					if (tableName==null) {
						Map<String, String> headings = new HashMap<String, String>();
						for (String heading : tiza.getHeadings()) {
							String[] p=heading.split("\\(");
							String type="text";
							if (p.length>1) {
								type=p[1].replaceAll("\\)","");
								if (type.equalsIgnoreCase("Group"))
									type="Structure";
							}
							headings.put(p[0].trim(), type.trim());
						}

						table = new Table();
						for (String heading : headings.keySet()) {
							TableColumn tc = new TableColumn();
							tc.setColName(heading);
							tc.setDisplayType(headings.get(heading));
							table.addColumn(tc);
						}
						table.init();
					} else {
						Document tableDoc = this.getTable(tableName);
						if (tableDoc!=null) 
							table = tableDoc.toObject(Table.class);
						if (table==null) {
							System.out.println("Unknown table "+tableName);
							return;
						}
					}
					tiza=TizaReader.createReader(filename, props, "");
					System.out.println((new Date()).toLocaleString()+" loadFile Process starting  ("+filename+")");

					List<Document> records = new ArrayList<Document>(1000);
					Map<String, Object> record;
					int count=0;
					while ((record = tiza.readNext())!=null) {
						count++;
						//convert to doc
						Document doc = Node.expandDoc(record, table);
						doc.append("Table", tableName);
						records.add(doc);
						if (count % 1000 == 0){
							saveMany(records);
							if (postBatch!=null)
							   postBatch(postBatch.toList(), records);
							records = new ArrayList<Document>(1000);
							count=0;
						}
					}
					if (records.size()>0) {
						saveMany(records);
						if (postBatch!=null)
						   postBatch(postBatch.toList(), records);
					}
					if (!quiet) {
						System.out.println((new Date()).toLocaleString()+ " loadFile Process completed ("+filename+") in " + ( System.currentTimeMillis()-start) + "ms");

					}

					if (postLoad!=null) {
						long startpl=System.currentTimeMillis();
						if (!quiet) {
							System.out.println((new Date()).toLocaleString()+" Post Load Pipeline Commencing...");
						}
						aggregate(records, postLoad.toList()).forEach(d -> System.out.println(((Document)d).toJson())); 
							

						if (!quiet) {
							System.out.println((new Date()).toLocaleString()+ " Post Load Pipeline Completed ("+filename+") in " + ( System.currentTimeMillis()-startpl) + "ms");
						}
					}

				}catch (Exception e) {

					e.printStackTrace();

				}

			}

		} catch (Exception e) {
			e.printStackTrace();
			if (terminate)
			System.exit(-1);
		}

		if (terminate)
			System.exit(0);
		

	}

	private void postBatch(List<Document> postBatch, List<Document> records) {
		if (postBatch!=null) {
			long startpl=System.currentTimeMillis();
			aggregate(records, postBatch).forEach(d -> System.out.println(((Document)d).toJson()));
			
			if (!quiet) {
				System.out.println((new Date()).toLocaleString()+" Post Batch Completed in " +(System.currentTimeMillis()-startpl)+ " ms");
			}
		}
	}


}
