/**
 *
	MonsterDB - Collection Based Database with fuzzy matching
    
    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.monster.db;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

import com.entitystream.identiza.entity.resolve.match.Matchable;

public class Session {
	protected static final long MAXAGE = (1l*60l*1000l);
	Map<Long, DBCursor> cursors = new HashMap<Long, DBCursor>(); 
	Map<Long, Long> cursorAge = new HashMap<Long, Long>(); 
	Map<Long, AtomicBoolean> cursorActive = new HashMap<Long, AtomicBoolean>(); 
	Thread ticker = null;
	private long id;
	public long lastTouch;
	private boolean remoteView;
	private List<Document> roles;
	private User user;
	private String lastStatement;
	public Session(long id, User user, List<Document> roles) {
		this.id=id;
		this.user=user;
		this.roles=roles;
		this.lastTouch=System.currentTimeMillis();
		ticker = new Thread(new Runnable() {
			@Override
			public void run() {
				if (user!=null)
				   System.out.println("Session Connected " +id + " as " + user.identifier);
				else
				   System.out.println("Session Connected " +id );
				while (cursors.size()!=-1) {
					try {
						for (long cur : cursorAge.keySet()) {
							if (cursors.containsKey(cur)) {
								if (cursorAge.get(cur) + MAXAGE < System.currentTimeMillis()) {
								     if (!cursorActive.get(cur).get())
									  cursors.remove(cur);
								     else //extend life due to active cursor
									 cursorAge.put(id, System.currentTimeMillis());
								}
							}
						}
						Thread.sleep(10000);
					}
					catch (Exception e) {
						System.out.println("Session ticker terminated " +id);
					}
				}

			}
		});
		ticker.start();
	}

	/**
	 * @param sessionid
	 */
	public Session(long sessionid) {
	    this(sessionid, null, null);
	}

	public DBCursor getCursor(long cursor) {
		this.lastTouch=System.currentTimeMillis();
		return cursors.get(cursor);
	}

	public void removeCursor(long cursor) {
		this.lastTouch=System.currentTimeMillis();
		DBCursor c = cursors.get(cursor);
		if (c!=null)
		   c.destroy();
		cursors.remove(cursor);

	}

	public void disconnect() {
		cursors.clear();
		ticker.interrupt();
		ticker=null;
		System.out.println("Session disconnected " +id);
	}

	public long execute(ICollection coll, String command) {
		this.lastTouch=-1; //prevents timeout
		DBCursor cursor = coll.executeCommand(command, user,this);
		return addCursor(cursor, command);
	}

	public long execute(Database db, String command) {
		this.lastTouch=-1; //prevents timeout
		db.setRemoteView(remoteView);
		DBCursor cursor = db.executeCommand(command, user, this);
		return addCursor(cursor, command);


	}

	public long execute(Container client, String command) {
		this.lastTouch=-1; //prevents timeout
		client.setRemoteView(remoteView);
		DBCursor cursor = client.executeCommand(command, user,this);
		return addCursor(cursor, command);
	}



	public long addCursor(DBCursor cursor, String statement) {
	    	
		this.lastTouch=System.currentTimeMillis();
		this.lastStatement=statement;
		
		long id = UUID.randomUUID().getLeastSignificantBits();
		
		//System.out.println("Adding Cursor "+ id+" for command " +statement);
		
		cursorAge.put(id, System.currentTimeMillis());
		AtomicBoolean ab = new AtomicBoolean(false);
		if (cursor!=null) {
		  cursor.setInProgress(ab);
		  cursor.setStatement(statement);
		}
		cursors.put(id,cursor);
		cursorActive.put(id, ab);
		return id;
	}

	public void isRemote(boolean remoteView) {
		this.remoteView=remoteView;
		
	}

	
}
