/**
 *
	MonsterDB - Collection Based Database with fuzzy matching

    Copyright (C) 2019  Robert James Haynes (EntityStream KFT), Budapest Hungary

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see https://www.gnu.org/licenses/agpl-3.0.en.html
 */
package com.entitystream.morphia;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

import com.entitystream.monster.db.Document;
import com.entitystream.monster.db.MonsterClient;
import com.entitystream.morphia.annotations.Entity;
import com.entitystream.morphia.annotations.Id;
import com.entitystream.morphia.annotations.Property;
import com.entitystream.morphia.annotations.Transient;
import com.google.gson.annotations.SerializedName;

import io.undertow.util.DateUtils;

/**
 * @author roberthaynes
 *
 */
public class MonsterDatastore implements Datastore {
    MonsterClient client = null;
    String packageNme;

    public MonsterDatastore(MonsterClient monClient, String packageNme) {
	client=monClient;
	this.packageNme=packageNme;
    }

    @Override
    public MonsterClient getMonsterDB() {
	return client;
    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#createQuery(java.lang.Class)
     */
    @Override
    public <T> Query createQuery(Class<T> class1) {
	return new Query(class1, this);
    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#save(java.lang.Object)
     */
    @Override
    public <T> T save(T entity) {
	//convert the object into a document
	Document out=convert(entity);
	String type=entity.getClass().getSimpleName();
	Entity ann = entity.getClass().getAnnotation(Entity.class);
	if (ann!=null)
	    type=ann.value();
	Document ret = client.useCollection(type).insertOne(out);

	return (T) convert(ret, entity.getClass());
    }


    private <T> Object convert(Document doc, Class<T> clazz) {
	Object object = null;
	try {
	    object = clazz.getConstructor().newInstance();
	    for (java.lang.reflect.Field field : clazz.getDeclaredFields()) {
		field.setAccessible(true);
		Transient istrans = field.getAnnotation(Transient.class);    

		if ((istrans==null 
			|| !istrans.value()) 
			&& !Modifier.isStatic(field.getModifiers())
			&& !Modifier.isFinal(field.getModifiers())) {
		    //non transient fields only
		    String name=field.getName();
		    SerializedName prop = field.getAnnotation(SerializedName.class);
		    if (prop!=null)
			name=prop.value();

		    Object value=doc.get(name);
		    try {
			if (value!=null){
			    if (value instanceof Document)
				field.set(object, convert(((Document)value), field.getType()));
			    else if (value instanceof List) {
				Class listOfClass=Object.class;
				if (field.getGenericType() instanceof java.lang.reflect.ParameterizedType)
				   listOfClass=(Class) ((java.lang.reflect.ParameterizedType) field.getGenericType()).getActualTypeArguments()[0];
				List newlist=new ArrayList();
				for (Object item : (List)value) {
				    if (listOfClass.isPrimitive()) {
					newlist.add(item);
				    } else if (item instanceof Document){
					
					newlist.add(convert((Document)item, listOfClass));
				    } 
				}
				field.set(object, newlist);
			    }
			    else if (!field.getType().isAssignableFrom(value.getClass())) {
				//not assignable
				Class<? extends Object> valClass = value.getClass();
				Class<? extends Object> fieldClass=field.getType();
				
				if (fieldClass.equals(long.class)) {
				    if (valClass.equals(Long.class))
					field.set(object, ((Long)value).longValue());
				    else if (valClass.equals(Integer.class))
					field.set(object, ((Integer)value).longValue());
				    else if (valClass.equals(int.class))
					field.set(object, Integer.valueOf((int)value).longValue());
				} else if (fieldClass.equals(Long.class)) {
				    if (valClass.equals(long.class))
					field.set(object, Long.valueOf((long) value));
				    else if (valClass.equals(int.class))
					field.set(object, Long.valueOf((int) value));
				    else if (valClass.equals(Integer.class))
					field.set(object, Long.valueOf(((Integer) value)).intValue());
				} else if (fieldClass.equals(Integer.class)) {
				    if (valClass.equals(Long.class))
					field.set(object, Integer.valueOf(((Long)value).intValue()));
				    else if (valClass.equals(int.class))
					field.set(object, Integer.valueOf((int)value));
				    else if (valClass.equals(long.class))
					field.set(object, Integer.valueOf(Long.valueOf((long)value).intValue()));
				} else if (fieldClass.equals(int.class)) {
				    if (valClass.equals(Long.class))
					field.set(object, ((Long)value).intValue());
				    else if (valClass.equals(Integer.class))
					field.set(object, ((Integer)value).intValue());
				    else if (valClass.equals(long.class))
					field.set(object, Long.valueOf((long)value).intValue());
				} else if (fieldClass.equals(boolean.class)) {
				    if (valClass.equals(Boolean.class))
					field.set(object, ((Boolean)value).booleanValue());
				} else if (fieldClass.equals(Boolean.class)) {
				    if (valClass.equals(boolean.class))
					field.set(object, Boolean.valueOf((boolean) value));
				} else if (fieldClass.isEnum()) {
				    //convert enum
				    if (valClass.equals(String.class)) {
					Class<Enum> e = (Class<Enum>)field.getType();
					field.set(object, Enum.valueOf(e, (String)value));
				    }
				} else if (fieldClass.equals(Date.class)) {
				    if (value instanceof Date)
					field.set(object, value);
				    else
				        field.set(object, Document.fromISO8601UTC(value));
					    
				} 
				  else //last ditch assign by casting
					field.set(object, field.getType().cast(value));
			    
			    } else 
				field.set(object, value);
			}
		    } catch (Exception e) {
			e.printStackTrace();
		    }
		}


	    }
	} catch (Exception e) {
	    e.printStackTrace();
	}
	return object;
    }
    /**
     * @param <T>
     * @param gson
     * @param entity
     * @return
     */
    private Document convert(Object entity) {
	Document out=new Document();

	for (java.lang.reflect.Field field : entity.getClass().getDeclaredFields()) {
	    field.setAccessible(true);
	    Transient istrans = field.getAnnotation(Transient.class);    

	    if (istrans==null || !istrans.value()) {
		//non transient fields only
		String name=field.getName();
		SerializedName prop = field.getAnnotation(SerializedName.class);
		if (prop!=null)
		    name=prop.value();

		try {
		    Object value=field.get(entity);
		    if (field.getType().isPrimitive() || 
			    value==null || 
			    field.getType().isAssignableFrom(String.class) ||
			    field.getType().isAssignableFrom(Number.class) ||
			    field.getType().isAssignableFrom(Boolean.class))
			out.append(name, value);
		    else if (field.getType().isAssignableFrom(Date.class)) {
			out.append(name, Document.toISO8601UTC((Date)value));
		    } 
		    else if (field.getType().isEnum())
			out.append(name, value.toString());
		    else if (field.getType().isArray()) {
			List newlist=new ArrayList();
			for (int i=0;i<((Object[])value).length;i++)
			    newlist.add(convert(((Object[])value)[i]));
			out.append(name, value.toString());
		    }
		    else if (field.getType().isAssignableFrom(List.class)) {
			List newlist=new ArrayList();
			for (Object item : (List)value)
			    newlist.add(convert(item));
			out.append(name, newlist);
		    } else if (field.getType().isAssignableFrom(Map.class) && value instanceof Map) {
			out.append(name, value);
		    } else if (field.getType().isAssignableFrom(Document.class) && value instanceof Document) {

			out.append(name, value);
		    } else if (field.getType().isAssignableFrom(Document.class) && value instanceof Map) {

			out.append(name, new Document(value));
		    }
		    else
			out.append(name, value);
		} catch (Exception e) {
		    e.printStackTrace();
		}
	    }


	}
	return out;


    }

    private String getIdField(Class forType) {
	for (java.lang.reflect.Field field : forType.getFields()) {
	    field.setAccessible(true);
	    Transient istrans = field.getAnnotation(Transient.class);    

	    if (istrans==null || !istrans.value()) {
		//non transient fields only
		String name=field.getName();

		if(field.getAnnotation(Id.class) != null) {
		    return name;
		}
	    }

	}
	return null;
    }


    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#createUpdateOperations(java.lang.Class)
     */
    @Override
    public <T> UpdateOperations createUpdateOperations(Class<T> clazz) {

	return new UpdateOperations();
    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#update(com.entitystream.morphia.Query, com.entitystream.morphia.UpdateOperations)
     */
    @Override
    public void update(Query q, UpdateOperations up) {
	client.updateMany(q.getQuery(), new Document("$set", up.getOperations()), up.getOptions());

    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#findAndDelete(com.entitystream.morphia.Query)
     */
    @Override
    public <T> T findAndDelete(Query q) {
	Document r = client.findOneAndDelete(q.getQuery());
	return (T) convert(r, (Class)q.getType());
    }
    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#find(java.lang.Class)
     */
    @Override
    public <T> Query<T> find(Class<T> class1) {
	return new Query(class1,this);
    }



    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#find(java.lang.Class, java.lang.String, java.lang.String)
     */
    @Override
    public <T> Query<T> find(Class<T> class1, String string, String value) {
	Query<T> q =  new Query(class1, this);
	return q.criteria(string).equal(value);
    }



    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#delete(java.lang.Object)
     */
    @Override
    public <T> T delete(T entity) {
	Document ret = client.deleteOne(convert(entity));
	return (T) convert(ret, entity.getClass());
    }


    private static Class[] getClasses(String packageName)
	    throws ClassNotFoundException, IOException {
	ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
	assert classLoader != null;
	String path = packageName.replace('.', '/');
	Enumeration resources = classLoader.getResources(path);
	List<File> dirs = new ArrayList();
	while (resources.hasMoreElements()) {
	    URL resource = (URL) resources.nextElement();
	    dirs.add(new File(resource.getFile()));
	}
	ArrayList<Class> classes = new ArrayList();
	for (File directory : dirs) {
	    classes.addAll(findClasses(directory, packageName));
	}
	return classes.toArray(new Class[classes.size()]);
    }


    private static List findClasses(File directory, String packageName) throws ClassNotFoundException {
	List<Class> classes = new ArrayList();
	if (!directory.exists()) {
	    return classes;
	}
	File[] files = directory.listFiles();
	for (File file : files) {
	    if (file.isDirectory()) {
		assert !file.getName().contains(".");
		classes.addAll(findClasses(file, packageName + "." + file.getName()));
	    } else if (file.getName().endsWith(".class")) {
		classes.add(Class.forName(packageName + '.' + file.getName().substring(0, file.getName().length() - 6)));
	    }
	}
	return classes;
    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#ensureIndexes()
     */
    @Override
    public void ensureIndexes() {
	try {
	    for (Class clazz : getClasses(packageNme)) {
		if (clazz.isAnnotationPresent(Entity.class)) {
		    String id = getIdField(clazz);

		    String[] types=clazz.toString().split("\\.");
		    String type=types[types.length-1];

		    Entity ann = (Entity) clazz.getAnnotation(Entity.class);
		    if (ann!=null)
			type=ann.value();
		    client.useCollection(type);
		    client.createIndex(new Document(id, 1), new Document("name", id+"_1").append("unique",true));
		}
	    }
	} catch (ClassNotFoundException | IOException e) {
	    e.printStackTrace();
	}

    }

    /**
     * @param <T>
     * @param query
     */
    public <T> List<T> find(Query<T> query) {
	List<T> out = new ArrayList();
	String fullName=query.getType().toString().replaceAll("class ", "");
	String[] types=fullName.split("\\.");
	String type=types[types.length-1];
	Entity ann = (Entity) query.getType().getClass().getAnnotation(Entity.class);
	if (ann!=null)
	    type=ann.value();
	client.useCollection(type);
	Class clazz=(Class)query.getType();

	try {
	    clazz = this.getClass().forName(fullName);
	} catch (Exception e) {
	    e.printStackTrace();
	}
	for (Document d : client.find(query.getQuery()).sort(query.getOrder())) {
	    out.add((T) convert(d, clazz));
	};

	return out;


    }

    /**
     * @param query
     * @param options
     * @return
     */
    public <T> List<T> find(Query<T> query, FindOptions options) {

	List<T> out = new ArrayList();
	long l=Long.MAX_VALUE;
	if (options.getLimit()>-1)
	    l=options.getLimit();

	String fullName=query.getType().toString().replaceAll("class ", "");
	Class clazz=(Class)query.getType();

	try {
	    clazz = this.getClass().forName(fullName);
	} catch (Exception e) {
	    e.printStackTrace();
	}

	for (Document d : client.find(query.getQuery()).limit(l).sort(query.getOrder())) {
	    out.add((T) convert(d, clazz));
	};

	return out;

    }

    /* (non-Javadoc)
     * @see com.entitystream.morphia.Datastore#replace(com.entitystream.morphia.Query, java.lang.Object)
     */
    @Override
    public <T> T replace(Query q, T entity) {
	Document r = client.findOneAndReplace(q.getQuery(), convert(entity));
	return (T) convert(r, (Class)q.getType());
    }

}
