package com.eniot.data.query;

import com.eniot.data.query.exception.SqlError;
import com.eniot.data.query.impl.ConnectionImpl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * @author jinghui.zhao
 * @date 2020/1/9
 */
public class Driver implements java.sql.Driver {
    private static final Logger log = LoggerFactory.getLogger(Driver.class);

    private static final String URL_PREFIX = "jdbc:datafederation://";

    public static final String PLATFORM = getPlatform();
    public static final String OS = getOSName();
    public static final String LICENSE = "@DATA_FEDERATION_LICENSE_TYPE@";
    public static final String RUNTIME_VENDOR = System.getProperty("java.vendor");
    public static final String RUNTIME_VERSION = System.getProperty("java.version");
    public static final String VERSION = "@DATA_FEDERATION_VERSION@";
    public static final String NAME = "@DATA_FEDERATION_DISPLAY_PROD_NAME@";

    public static final String SERVCER_NAME_TO_CONNECT = "data-query-proxy";



    /**
     * The following should be the last static initialization, so that any other
     * static initialization is completed before we create an instance and let
     * DriverManager access it:
     */
    static {
        // Upon loading of class, register an instance with DriverManager.
        try {
            DriverManager.registerDriver(new Driver());
        } catch (Error | SQLException e) {
            log.error("Error in registering data-query-proxy Driver driver {}: {}", Driver.class, e);
        }
    }

    /**
     * Construct a new driver and register it with DriverManager
     *
     * @throws SQLException if a database error occurs.
     */
    public Driver() throws SQLException {

    }

    /**
     * Ensures that class is loaded.
     * <p>
     * (Avoids extra instance of calling {@code new Driver();}; avoids verbosity
     * of {@code Class.forName("org.apache.drill.jdbc.Driver");}.)
     * </p>
     * @return boolean
     */
    public static boolean load() {
        return true;
    }


    public static String getOSName() {
        return System.getProperty("os.name");
    }

    /**
     * @return the transformed, standardized platform details
     */
    public static String getPlatform() {
        return System.getProperty("os.arch");
    }


    /**
     * Key used to retreive the database value from the properties instance
     * passed to the driver.
     * used for orgId
     */
    public static final String ORGID_PROPERTY_KEY = "orgId";

    private static final String CHID_PROPERTY_KEY = "chId";

    public static final String SOURCE_PROPERTY_KEY = "source";

    /**
     * Should the driver generate debugging output
     */
    public static final boolean DEBUG = false;

    /**
     * Key used to retreive the hostname value from the properties instance
     * passed to the driver.
     */
    private static final String HOST_PROPERTY_KEY = "HOST";

    /**
     * Key used to retreive the username value from the properties instance
     * passed to the driver.
     */
    public static final String USER_PROPERTY_KEY = "user";

    public static final String USE_CONFIG_PROPERTY_KEY = "useConfigs";

    /**
     * Gets the drivers major version number
     *
     * @return the drivers major version number
     */
    public static int getMajorVersionInternal() {
        return safeIntParse("@ENIOT_EDP_CJ_MAJOR_VERSION@");
    }

    /**
     * Get the drivers minor version number
     *
     * @return the drivers minor version number
     */
    public  static int getMinorVersionInternal() {
        return safeIntParse("@ENIOT_EDP_CJ_MINOR_VERSION@");
    }

    public static int safeIntParse(String intAsString) {
        try {
            return Integer.parseInt(intAsString);
        } catch (NumberFormatException nfe) {
            return 0;
        }
    }

    /**
     * @param url the URL of the driver
     * @return true if this driver accepts the given URL
     * @throws SQLException if a database access error occurs or the url is null
     * @see java.sql.Driver#acceptsURL
     */
    @Override
    public boolean acceptsURL(String url) throws SQLException {
        if (url == null) {
            throw SqlError.createSQLException("url is null!", SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }
        return url.startsWith(URL_PREFIX);
    }

    /**
     * @param url  the URL of the database to connect to
     * @param info a list of arbitrary tag/value pairs as connection arguments
     * @return a connection to the URL or null if it isn't us
     * @throws SQLException SQLException
     */
    @Override
    public java.sql.Connection connect(String url, Properties info) throws SQLException {
        if (!acceptsURL(url)) {
            return null;
        }

        Properties props = null;
        if ((props = parseURL(url, info)) == null) {
            return null;
        }

        if (!props.containsKey(HOST_PROPERTY_KEY)) {
            throw SqlError.createSQLException("url is invalid!",
                SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }
        if (!props.containsKey(USER_PROPERTY_KEY)) {
            throw SqlError.createSQLException("user is required to establish a connection!",
                SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }

        String psKeyName = "password";
        if (!props.containsKey(psKeyName)) {
            throw SqlError.createSQLException("password is required to establish a connection!",
                SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }

        if (!props.containsKey(ORGID_PROPERTY_KEY)) {
            throw SqlError.createSQLException("orgId is required to establish a connection!",
                SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }

        String channelPre = "ch-";
        if (!props.containsKey(CHID_PROPERTY_KEY) || !props.getProperty(CHID_PROPERTY_KEY).startsWith(channelPre)) {
            throw SqlError.createSQLException("channelId is required to establish a connection!",
                    SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }

        try {
            return ConnectionImpl.getInstance(host(props), props, database(props));
        } catch (SQLException sqlEx) {
            // Don't wrap SQLExceptions, throw
            // them un-changed.
            throw sqlEx;
        } catch (Exception ex) {
            log.error("NonRegisteringDriver.connect Exception", ex);
            SQLException sqlEx = SqlError.createSQLException("NonRegisteringDriver.connect Exception" + ex.toString(),
                SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
            sqlEx.initCause(ex);

            throw sqlEx;
        }
    }

    public Properties parseURL(String url, Properties defaults) throws SQLException {
        Properties urlProps = (defaults != null) ? defaults : new Properties();

        if (url == null) {
            return null;
        }

        if (!StringUtils.startsWithIgnoreCase(url, URL_PREFIX)) {
            return null;
        }

        int beginningOfSlashes = url.indexOf("//");

        /*
         * Parse parameters after the ? in the URL and remove them from the
         * original URL.
         */
        int index = url.indexOf("?");
        if (index == -1){
            index = url.length() - 1;
        }

        String paramString = url.substring(index + 1);
        url = url.substring(0, index);

        StringTokenizer queryParams = new StringTokenizer(paramString, "&");

        while (queryParams.hasMoreTokens()) {
            String parameterValuePair = queryParams.nextToken();

            int indexOfEquals = StringUtils.indexOfIgnoreCase(parameterValuePair, "=", 0);

            String parameter = null;
            String value = null;

            if (indexOfEquals != -1) {
                parameter = parameterValuePair.substring(0, indexOfEquals);

                if (indexOfEquals + 1 < parameterValuePair.length()) {
                    value = parameterValuePair.substring(indexOfEquals + 1);
                }
            }

            if (StringUtils.isNotBlank(value) && StringUtils.isNotBlank(parameter)) {
                urlProps.setProperty(parameter, value);
            }
        }

        url = url.substring(beginningOfSlashes + 2);
        urlProps.setProperty(HOST_PROPERTY_KEY, url);

        // split get chId
        String chId = url.substring(url.lastIndexOf('/') + 1);
        urlProps.setProperty(CHID_PROPERTY_KEY, chId);

        return urlProps;
    }

    /**
     * Returns the database property from <code>props</code>
     *
     * @param props the Properties to look for the database property.
     * @return the database name.
     */
    public String database(Properties props) {
        return props.getProperty(CHID_PROPERTY_KEY);
    }

    /**
     * Gets the drivers major version number
     *
     * @return the drivers major version number
     */
    @Override
    public int getMajorVersion() {
        return getMajorVersionInternal();
    }

    /**
     * Get the drivers minor version number
     *
     * @return the drivers minor version number
     */
    @Override
    public int getMinorVersion() {
        return getMinorVersionInternal();
    }

    /**
     * @param url  the Url of the database to connect to
     * @param info a proposed list of tag/value pairs that will be sent on
     *             connect open.
     * @return An array of DriverPropertyInfo objects describing possible
     * properties. This array may be an empty array if no properties are
     * required
     * @throws SQLException if a database-access error occurs
     * @see java.sql.Driver#getPropertyInfo
     */
    @Override
    public DriverPropertyInfo[] getPropertyInfo(String url, Properties info) throws SQLException {
        if (info == null) {
            info = new Properties();
        }

        if ((url != null) && url.startsWith(URL_PREFIX)) {
            info = parseURL(url, info);
        }
        DriverPropertyInfo[] ret = new DriverPropertyInfo[4];
        DriverPropertyInfo hostProp = new DriverPropertyInfo(HOST_PROPERTY_KEY, info.getProperty(HOST_PROPERTY_KEY));
        hostProp.required = true;
        hostProp.description = "url";

        DriverPropertyInfo dbProp = new DriverPropertyInfo(CHID_PROPERTY_KEY, info.getProperty(CHID_PROPERTY_KEY));
        dbProp.required = true;
        dbProp.description = "chId name";

        DriverPropertyInfo userProp = new DriverPropertyInfo(USER_PROPERTY_KEY, info.getProperty(USER_PROPERTY_KEY));
        userProp.required = true;
        userProp.description = "user";

        DriverPropertyInfo passwordProp = new DriverPropertyInfo("password", info.getProperty("password"));
        userProp.required = true;
        userProp.description = "password";

        ret[0] = hostProp;
        ret[1] = dbProp;
        ret[2] = userProp;
        ret[3] = passwordProp;

        return ret;
    }

    /**
     * Returns the hostname property
     *
     * @param props the java.util.Properties instance to retrieve the hostname
     *              from.
     * @return the hostname
     */
    public String host(Properties props) {
        return props.getProperty(HOST_PROPERTY_KEY, "localhost");
    }

    /**
     * Report whether the driver is a genuine Driver compliant driver. A driver
     * may only report "true" here if it passes the Driver compliance tests,
     * otherwise it is required to return false. Driver compliance requires full
     * support for the Driver API and full support for SQL 92 Entry Level.
     *
     * <p>
     * MySQL is not SQL92 compliant
     * </p>
     *
     * @return is this driver Driver compliant?
     */
    @Override
    public boolean jdbcCompliant() {
        return false;
    }

    @Override
    public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
        return null;
    }

    /**
     * Returns the given property from <code>props</code>
     *
     * @param name  the property name
     * @param props the property instance to look in
     * @return the property value, or null if not found.
     */
    public String property(String name, Properties props) {
        return props.getProperty(name);
    }

}
