package com.eniot.data.query.impl;

import com.eniot.data.query.EniotConnect;
import com.eniot.data.query.Driver;
import com.eniot.data.query.entity.QueryResponse;
import com.eniot.data.query.exception.SqlError;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.*;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;

/**
 * @author jinghui.zhao
 */
public class ConnectionImpl implements EniotConnect {

    private static final Logger log = LoggerFactory.getLogger(ConnectionImpl.class);

    /**
     * The point in time when this connection was created
     */
    private long connectionCreationTimeMillis = 0;

    /**
     * ID used when profiling
     */
    private long connectionId;

    /**
     * We are in read-only mode
     */
    private boolean readOnly = true;

    private boolean isClosed = true;

    /**
     * Internal DBMD to use for various database-version specific features
     */
    private DatabaseMetaData dbmd = null;

    /**
     * The hostname we're connected to
     */
    private String host = null;

    private String chId = null;

    private String user = null;

    private String source = null;

    private RestClient restClient;

    private static final int DEFAULT_RESULT_SET_TYPE = ResultSet.TYPE_FORWARD_ONLY;

    private static final int DEFAULT_RESULT_SET_CONCURRENCY = ResultSet.CONCUR_READ_ONLY;

    private String serverVersion = "0.0.1";

    /**
     * private Statement for Driver query
     */
    private volatile Statement statement = null;

    private volatile QueryResponse response = null;


    /**
     * An array of currently open statements.
     * Copy-on-write used here to avoid ConcurrentModificationException when statements unregister themselves while we iterate over the list.
     */
    private final CopyOnWriteArrayList<Statement> openStatements = new CopyOnWriteArrayList<Statement>();

    private boolean transactionsSupported = false;

    /**
     * add "`" to dbName or not
     */
    private boolean dbEscapeNeeded = false;

    /**
     * Map mysql transaction isolation level name to
     * java.sql.Connection.TRANSACTION_XXX
     */
    private static Map<String, Integer> mapTransIsolationNameToValue = null;

    static {
        mapTransIsolationNameToValue = new HashMap<String, Integer>(8);
        mapTransIsolationNameToValue.put("READ-UNCOMMITTED", TRANSACTION_READ_UNCOMMITTED);
        mapTransIsolationNameToValue.put("READ-COMMITTED", TRANSACTION_READ_COMMITTED);
        mapTransIsolationNameToValue.put("REPEATABLE-READ", TRANSACTION_REPEATABLE_READ);
        mapTransIsolationNameToValue.put("SERIALIZABLE", TRANSACTION_SERIALIZABLE);
    }


    /**
     * Creates a connection instance
     *
     * @param hostToConnectTo host
     * @param info properties
     * @param chIdToConnectTo chIdToConnectTo
     * @return Connection
     * @throws SQLException if a database access error occurs
     */
    public static Connection getInstance(String hostToConnectTo, Properties info, String chIdToConnectTo)
            throws SQLException {
        return new ConnectionImpl(hostToConnectTo, info, chIdToConnectTo);

    }

    /**
     * Creates a connection to a DataQuery Server.
     *
     * @param hostToConnectTo the hostname of the database server
     * @param info            a Properties[] list holding the user and password
     * @param chIdToConnectTo the channel to connect to
     * @throws SQLException if a database access error occurs
     */
    public ConnectionImpl(String hostToConnectTo, Properties info, String chIdToConnectTo) throws SQLException {
        log.info("Connect server start.");
        this.connectionCreationTimeMillis = System.currentTimeMillis();

        this.host = hostToConnectTo;
        this.chId = chIdToConnectTo;
        this.user = info.getProperty(Driver.USER_PROPERTY_KEY);
        this.source = info.getProperty(Driver.SOURCE_PROPERTY_KEY);
        dbEscapeNeeded = Boolean.valueOf(info.getProperty("dbEscapeNeeded", "false"));

        restClient = RestClient.getRestClient(host, chId, info);

        isClosed = false;
    }

    public void checkClosed() throws SQLException {
        synchronized (this){
            if (this.isClosed) {
                log.warn("The connection has closed!");
                throw SqlError.createSQLException("No operations allowed after connection closed.", SqlError.SQL_STATE_CONNECTION_NOT_OPEN);
            }
        }
    }

    @Override
    public boolean isClosed() {
        return this.isClosed;
    }

    /**
     * JDBC1.0
     *
     * @return Statement
     * @throws SQLException if a database access error occurs
     */
    @Override
    public Statement createStatement() throws SQLException {
        return this.createStatement(DEFAULT_RESULT_SET_TYPE, DEFAULT_RESULT_SET_CONCURRENCY);
    }

    /**
     * JDBC2.0
     *
     * @param resultSetType resultSetType
     * @param resultSetConcurrency resultSetConcurrency
     * @return Statement
     * @throws SQLException if a database access error occurs
     */
    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        log.info("ConnectionImpl.createStatement, resultSetType:{}, resultSetConcurrency:{}", resultSetType, resultSetConcurrency);
        checkClosed();
        if (resultSetType != java.sql.ResultSet.TYPE_FORWARD_ONLY && resultSetType != java.sql.ResultSet.TYPE_SCROLL_INSENSITIVE) {
            throw SqlError.createSQLException("TYPE_FORWARD_ONLY or TYPE_SCROLL_INSENSITIVE is only supported resultSetType level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        if (resultSetConcurrency != java.sql.ResultSet.CONCUR_READ_ONLY) {
            throw SqlError.createSQLException("CONCUR_READ_ONLY is only supported resultSetConcurrency level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }

        return new StatementImpl(this, resultSetType, resultSetConcurrency);
    }

    /**
     * JDBC3.0
     *
     * @param resultSetType resultSetType
     * @param resultSetConcurrency resultSetConcurrency
     * @param resultSetHoldability resultSetHoldability
     * @return Statement
     * @throws SQLException SQLException
     */
    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        log.info("ConnectionImpl.createStatement, resultSetType:{}, resultSetConcurrency:{}, resultSetHoldability:{}", resultSetType, resultSetConcurrency, resultSetHoldability);
        if (resultSetHoldability != java.sql.ResultSet.HOLD_CURSORS_OVER_COMMIT) {
            throw SqlError.createSQLException("HOLD_CUSRORS_OVER_COMMIT is only supported holdability level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        return createStatement(resultSetType, resultSetConcurrency);
    }

    @Override
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        return this.prepareStatement(sql, DEFAULT_RESULT_SET_TYPE, DEFAULT_RESULT_SET_CONCURRENCY);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        log.info("ConnectionImpl.prepareStatement, sql:{}, resultSetType:{}, resultSetConcurrency:{}.", sql, resultSetType, resultSetConcurrency);
        checkClosed();
        if (resultSetType != java.sql.ResultSet.TYPE_FORWARD_ONLY && resultSetType != java.sql.ResultSet.TYPE_SCROLL_INSENSITIVE) {
            throw SqlError.createSQLException("TYPE_FORWARD_ONLY or TYPE_SCROLL_INSENSITIVE is only supported resultSetType level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        if (resultSetConcurrency != java.sql.ResultSet.CONCUR_READ_ONLY) {
            throw SqlError.createSQLException("CONCUR_READ_ONLY is only supported resultSetConcurrency level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        return new PreparedStatementImpl(this, sql, resultSetType, resultSetConcurrency);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        if (resultSetHoldability != java.sql.ResultSet.HOLD_CURSORS_OVER_COMMIT) {
            throw SqlError.createSQLException("HOLD_CUSRORS_OVER_COMMIT is only supported holdability level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        return this.prepareStatement(sql, resultSetType, resultSetConcurrency);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        log.info("ConnectionImpl.prepareStatement, sql:{}, autoGeneratedKeys:{}.", sql, autoGeneratedKeys);
        throw SqlError.createSQLFeatureNotSupportedException("prepareStatement");
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        log.info("ConnectionImpl.prepareStatement, sql:{}, columnIndexes:{}.", sql, columnIndexes);
        throw SqlError.createSQLFeatureNotSupportedException("prepareStatement");
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        log.info("ConnectionImpl.prepareStatement, sql:{}, columnNames:{}.", sql, columnNames);
        throw SqlError.createSQLFeatureNotSupportedException("prepareStatement");
    }

    @Override
    public CallableStatement prepareCall(String sql) throws SQLException {
        log.info("ConnectionImpl.prepareCall, sql:{}.", sql);
        throw SqlError.createSQLFeatureNotSupportedException("prepareCall");
    }

    @Override
    public String nativeSQL(String sql) throws SQLException {
        if (StringUtils.isNotBlank(sql)) {
            return sql.replace("\"", "`").replaceAll("[\r\n]", " ");
        }
        return sql;
    }

    @Override
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        return;
    }

    @Override
    public boolean getAutoCommit() throws SQLException {
        return true;
    }

    @Override
    public void commit() throws SQLException {
        log.info("ConnectionImpl.commit");
        return;
    }

    @Override
    public void rollback() throws SQLException {
        log.info("ConnectionImpl.rollback");
        return;

    }

    @Override
    public void close() throws SQLException {
        synchronized (this) {
            this.isClosed = true;
            this.restClient = null;
            this.statement = null;
            this.response = null;
        }
    }

    /**
     * @return a <code>DatabaseMetaData</code> object for this <code>Connection</code> object
     * @throws SQLException SQLException
     */
    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        return getMetaData(true);
    }

    private DatabaseMetaData getMetaData(boolean checkClosed) throws SQLException {
        if (checkClosed) {
            checkClosed();
        }
        return DatabaseMetaDataImpl.getInstance(this, this.chId);
    }

    @Override
    public void setReadOnly(boolean readOnly) throws SQLException {
        checkClosed();
        return;
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        return true;
    }

    @Override
    public void setCatalog(String catalog) throws SQLException {
        checkClosed();
        return;
    }

    @Override
    public String getCatalog() throws SQLException {
        return this.chId;
    }

    @Override
    public void setTransactionIsolation(int level) throws SQLException {
        log.info("ConnectionImpl.setTransactionIsolation, level:{}.", level);
        if (Connection.TRANSACTION_NONE != level) {
            throw SqlError.createSQLException("Not support transaction isolation level:" + level, SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public int getTransactionIsolation() throws SQLException {
        return Connection.TRANSACTION_NONE;
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        log.info("ConnectionImpl.prepareCall");
        throw SqlError.createSQLFeatureNotSupportedException("prepareCall");
    }

    @Override
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        log.info("ConnectionImpl.getTypeMap");
        return new HashMap<>(0);
    }

    @Override
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setTypeMap");
    }

    @Override
    public void setHoldability(int holdability) throws SQLException {
        if (ResultSet.HOLD_CURSORS_OVER_COMMIT != holdability) {
            throw SqlError.createSQLException("HOLD_CUSRORS_OVER_COMMIT is only supported holdability level", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public int getHoldability() throws SQLException {
        return ResultSet.HOLD_CURSORS_OVER_COMMIT;
    }

    @Override
    public Savepoint setSavepoint() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setSavepoint");
    }

    @Override
    public Savepoint setSavepoint(String name) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setSavepoint");
    }

    @Override
    public void rollback(Savepoint savepoint) throws SQLException {
    }

    @Override
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("releaseSavepoint");
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        log.info("ConnectionImpl.prepareCall");
        throw SqlError.createSQLFeatureNotSupportedException("prepareCall");
    }

    @Override
    public Clob createClob() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("createClob");
    }

    @Override
    public Blob createBlob() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("createBlob");
    }

    @Override
    public NClob createNClob() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("createNClob");
    }

    @Override
    public SQLXML createSQLXML() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("createSQLXML");
    }

    @Override
    public boolean isValid(int timeout) throws SQLException {
        return !this.isClosed;
    }

    private Properties clientInfo = null;

    @Override
    public void setClientInfo(String name, String value) throws SQLClientInfoException {
        log.info("ConnectionImpl.setClientInfo");
        synchronized (this) {
            if (this.clientInfo == null) {
                this.clientInfo = new Properties();
            }
            this.clientInfo.setProperty(name, value);
        }
    }

    @Override
    public void setClientInfo(Properties properties) throws SQLClientInfoException {
        log.info("ConnectionImpl.setClientInfo");
        synchronized (this) {
            if (this.clientInfo == null) {
                this.clientInfo = new Properties();
            }
            this.clientInfo.putAll(properties);
        }
    }

    @Override
    public String getClientInfo(String name) throws SQLException {
        log.info("ConnectionImpl.getClientInfo");
        synchronized (this) {
            if (this.clientInfo != null) {
                return this.clientInfo.getProperty(name);
            }
            return null;
        }
    }

    @Override
    public Properties getClientInfo() throws SQLException {
        log.info("ConnectionImpl.getClientInfo");
        synchronized (this) {
            return this.clientInfo;
        }
    }

    @Override
    public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
        log.info("ConnectionImpl.createArrayOf");
        throw SqlError.createSQLFeatureNotSupportedException("createArrayOf");
    }

    @Override
    public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
        log.info("ConnectionImpl.createStruct");
        throw SqlError.createSQLFeatureNotSupportedException("createStruct");
    }

    @Override
    public void setSchema(String schema) throws SQLException {
    }

    @Override
    public String getSchema() throws SQLException {
        checkClosed();
        return this.chId;
    }

    @Override
    public void abort(Executor executor) throws SQLException {

    }

    /**
     * Sets the maximum period a <code>Connection</code> or
     * objects created from the <code>Connection</code>
     * will wait for the database to reply to any one request. If any
     * request remains unanswered, the waiting method will
     * return with a <code>SQLException</code>, and the <code>Connection</code>
     * or objects created from the <code>Connection</code>  will be marked as
     * closed.
     *
     * @param executor executor
     * @param milliseconds milliseconds
     * @throws SQLException SQLException
     */
    @Override
    public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        log.info("ConnectionImpl.setNetworkTimeout");
    }

    @Override
    public int getNetworkTimeout() throws SQLException {
        return 300000;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        log.info("ConnectionImpl.unwrap");
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        log.info("ConnectionImpl.isWrapperFor");
        return false;
    }

    @Override
    public ResultSet execQuerySql(Statement statement, String sql) throws SQLException {
        checkClosed();
        QueryResponse response = restClient.execQuery(sql);
        return new ResultSetImpl(this, statement, response);

    }

    @Override
    public ResultSet execQuerySql(String sql) throws SQLException {
        checkClosed();
        if (null == this.statement) {
            this.statement = this.createStatement();
        }
        QueryResponse response = restClient.execQuery(sql);
        return new ResultSetImpl(this, this.statement, response);

    }

    @Override
    public ResultSet showSchemas() throws SQLException {
        checkClosed();
        if (null == this.statement) {
            this.statement = this.createStatement();
        }
        if (null == this.response) {
            String sql = "show schemas";
            this.response = restClient.execQuery(sql);
            //replace columnName
            String originSchemaKey = "SCHEMA_NAME";
            String finalSchemaKey = "TABLE_SCHEM";
            String catalogKey = "TABLE_CATALOG";
            List<String> newColumns = new ArrayList<>(2);
            newColumns.add(finalSchemaKey);
            newColumns.add(catalogKey);
            List<Map<String, Object>> newRows = new ArrayList<>(response.getRows().size());
            for (Map<String, Object> item : response.getRows()) {
                Map<String, Object> subRow = new HashMap<>(2);
                subRow.put(catalogKey, this.chId);
                Object schemaNameObject = item.get(originSchemaKey);
                String schemaName = schemaNameObject == null ? "" : schemaNameObject.toString();
                //ignore system schema
                if (this.source != null) {
                    subRow.put(finalSchemaKey, schemaName);
                    newRows.add(subRow);
                    continue;
                }
                if (StringUtils.isBlank(schemaName) || schemaName.contains("information_schema") || schemaName.contains(".mysql")
                        || schemaName.contains(".performance_schema") || schemaName.contains("sys")) {
                    continue;
                }
                String[] schemaTableName = schemaName.split("\\.");
                if (schemaTableName.length != 2) {
                    continue;
                }
                if (dbEscapeNeeded) {
                    schemaName = schemaTableName[0] + ".`" + schemaTableName[1] + "`";
                }
                subRow.put(finalSchemaKey, schemaName);

                newRows.add(subRow);
            }
            response.setColumns(newColumns);
            response.setRows(newRows);
            response.getMetadata().add("VARCHAR");
        }
        return new ResultSetImpl(this, this.statement, this.response);
    }

    @Override
    public ResultSet showCatalogs() throws SQLException {
        checkClosed();
        QueryResponse queryResponse = new QueryResponse();
        List<String> columns = new ArrayList<>(1);
        columns.add("TABLE_CAT");
        List<String> metadata = new ArrayList<>(1);
        metadata.add("VARCHAR");
        List<Map<String, Object>> rows = new ArrayList<>(1);
        Map<String, Object> row1 = new HashMap<>(1);
        row1.put("TABLE_CAT", this.chId);
        rows.add(row1);
        queryResponse.setColumns(columns);
        queryResponse.setMetadata(metadata);
        queryResponse.setRows(rows);
        return new ResultSetImpl(this, statement, queryResponse);
    }

    @Override
    public ResultSet getTableTypes() throws SQLException {
        checkClosed();
        QueryResponse queryResponse = new QueryResponse();
        List<String> columns = new ArrayList<>(1);
        columns.add("TABLE_TYPE");
        List<String> metadata = new ArrayList<>(1);
        metadata.add("VARCHAR");
        List<Map<String, Object>> rows = new ArrayList<>(1);
        Map<String, Object> row1 = new HashMap<>(1);
        row1.put("TABLE_TYPE", "TABLE");
        rows.add(row1);
        queryResponse.setColumns(columns);
        queryResponse.setMetadata(metadata);
        queryResponse.setRows(rows);
        return new ResultSetImpl(this, statement, queryResponse);
    }

    @Override
    public String getURL() {
        return this.host;
    }

    @Override
    public String getUser() {
        return this.user;
    }

    @Override
    public String getServerVersion() {
        return this.serverVersion;
    }

    @Override
    public Statement getPrivateStatement() throws SQLException {
        checkClosed();
        if (null == this.statement) {
            this.statement = this.createStatement();
        }
        return this.statement;
    }
}
