package com.eniot.data.query.impl;

import com.eniot.data.query.exception.SqlError;
import com.eniot.data.query.util.JdbcUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.ParameterMetaData;
import java.sql.SQLException;
import java.sql.Types;

/**
 * @author jinghui.zhao
 * @date 2020/2/28
 */
public class ParameterMetaDataImpl implements ParameterMetaData {

    private static final Logger log = LoggerFactory.getLogger(ParameterMetaDataImpl.class);

    private boolean[] isNull = null;

    private int[] parameterTypes = null;

    private int parameterCount = 0;

    public ParameterMetaDataImpl(int parameterCount, int[] parameterTypes, boolean[] isNull) {
        this.parameterCount = parameterCount;
        this.parameterTypes = parameterTypes;
        this.isNull = isNull;
    }

    @Override
    public int getParameterCount() throws SQLException {
        return this.parameterCount;
    }

    @Override
    public int isNullable(int param) throws SQLException {
        log.info("ParameterMetaDataImpl.isNullable, param:{}", param);
        return ParameterMetaData.parameterNoNulls;
    }

    @Override
    public boolean isSigned(int param) throws SQLException {
        checkBounds(param);
        int type = this.parameterTypes[param];
        switch (type) {
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
            case Types.NUMERIC:
            case Types.DECIMAL:
                return true;
            default:
                return false;
        }
    }

    @Override
    public int getPrecision(int param) throws SQLException {
        checkBounds(param);
        int type = this.parameterTypes[param];
        return JdbcUtils.getPrecisionFromTypes(type);
    }

    @Override
    public int getScale(int param) throws SQLException {
        checkBounds(param);
        int type = this.parameterTypes[param];
        return JdbcUtils.getPrecisionFromTypes(type);
    }

    @Override
    public int getParameterType(int param) throws SQLException {
        log.info("ParameterMetaDataImpl.getParameterType, param:{}", param);
        checkBounds(param);
        return this.parameterTypes[param - 1];
    }

    private void checkBounds(int paramIndex) throws SQLException {
        synchronized (this) {
            if ((paramIndex < 1)) {
                throw SqlError.createSQLException(String.format("Param index out of range, %d < 1.", paramIndex), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            } else if (paramIndex > this.parameterCount) {
                throw SqlError.createSQLException(String.format("Param index out of range, %d > %d.", paramIndex, this.parameterCount),
                        SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
    }

    @Override
    public String getParameterTypeName(int param) throws SQLException {
        int type = this.getParameterType(param);
        return JdbcUtils.getTypeNameFromTypes(type);
    }

    @Override
    public String getParameterClassName(int param) throws SQLException {
        int type = this.getParameterType(param);
        return JdbcUtils.getClassNameFromTypes(type);
    }

    @Override
    public int getParameterMode(int param) throws SQLException {
        return parameterModeUnknown;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
