package com.eniot.data.query.impl;

import com.eniot.data.query.EniotConnect;
import com.eniot.data.query.entity.QueryResponse;
import com.eniot.data.query.exception.SqlError;
import com.eniot.data.query.util.JdbcUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URL;
import java.sql.*;
import java.sql.Date;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

import static java.sql.ResultSet.FETCH_FORWARD;

/**
 * @author jinghui.zhao
 * @date 2020/2/25
 */
public class PreparedStatementImpl implements PreparedStatement {

    private static final Logger log = LoggerFactory.getLogger(PreparedStatementImpl.class);

    protected volatile EniotConnect connection = null;

    protected int statementId;

    /**
     * Used to generate IDs when profiling.
     */
    static int statementCounter = 1;

    protected int resultSetConcurrency = 0;

    /**
     * The type of this result set (scroll sensitive or in-sensitive)
     */
    protected int resultSetType = 0;

    private boolean isClosed = false;

    private volatile ResultSet results = null;

    private int maxFieldSize = Integer.MAX_VALUE;

    private int maxRows = 640000;

    private int fetchDirection = FETCH_FORWARD;

    private int fetchSize = 640000;

    /**
     * The SQL that was passed in to 'prepare'
     */
    private String originalSql = null;

    private boolean[] isNull = null;

    private String[] staticSql = null;

    private String[] parameterValues = null;

    private int[] parameterTypes = null;

    /**
     * The number of parameters in this PreparedStatement
     */
    private int parameterCount = 0;

    private int hasSpecifiedParamCount = 0;

    private String asSql() throws SQLException {
        synchronized (this) {
            checkClosed();
            StringBuilder buf = new StringBuilder();

            for (int i = 0; i < this.parameterCount; ++i) {
                buf.append(this.staticSql[i]);

                if (this.parameterValues[i] == null) {
                    buf.append("** NOT SPECIFIED **");
                } else {
                    buf.append(this.parameterValues[i]);
                }
            }
            buf.append(this.staticSql[this.parameterCount]);

            return buf.toString();
        }
    }

    public PreparedStatementImpl(EniotConnect c, String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        if ((c == null) || c.isClosed()) {
            throw SqlError.createSQLException("No operations allowed after connection closed.", SqlError.SQL_STATE_CONNECTION_NOT_OPEN);
        }

        JdbcUtils.checkSql(sql);

        this.connection = c;
        this.statementId = StatementImpl.statementCounter++;
        this.resultSetType = resultSetType;
        this.resultSetConcurrency = resultSetConcurrency;
        this.originalSql = this.connection.nativeSQL(sql);

        initializeSql(this.originalSql);

    }

    /**
     * 1、初始化，解析sql
     * 2、记录paramVal
     * 3、合并sql
     *
     * @throws SQLException
     */

    private void initializeSql(String sql) throws SQLException {
        synchronized (this) {
            int statementLength = sql.length();

            ArrayList<int[]> endpointList = new ArrayList<int[]>();
            int lastParmEnd = 0;
            for (int i = 0; i < statementLength; ++i) {
                char c = sql.charAt(i);
                if ((c == '?')) {
                    endpointList.add(new int[]{lastParmEnd, i});
                    lastParmEnd = i + 1;
                }
            }
            endpointList.add(new int[]{lastParmEnd, statementLength});

            this.staticSql = new String[endpointList.size()];
            for (int i = 0; i < this.staticSql.length; i++) {
                int[] ep = endpointList.get(i);
                int begin = ep[0];
                int end = ep[1];
                this.staticSql[i] = sql.substring(begin, end);
            }

            this.parameterCount = this.staticSql.length - 1;
            this.parameterValues = new String[parameterCount];
            this.isNull = new boolean[this.parameterCount];
            this.parameterTypes = new int[this.parameterCount];

            this.clearParameters();
        }
    }

    private void checkClosed() throws SQLException {
        synchronized (this) {
            if (isClosed) {
                log.warn("The preparedStatement has closed!");
                throw SqlError.createSQLException("No operations allowed after preparedStatement closed.", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
    }

    private void checkBounds(int paramIndex) throws SQLException {
        synchronized (this) {
            if ((paramIndex < 1)) {
                throw SqlError.createSQLException(String.format("Param index out of range, %d < 1.", paramIndex), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            } else if (paramIndex > this.parameterCount) {
                throw SqlError.createSQLException(String.format("Param index out of range, %d > %d.", paramIndex, this.parameterCount),
                        SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
    }

    private void checkParamSpecifiedFinished() throws SQLException {
        synchronized (this) {
            if (this.hasSpecifiedParamCount != this.parameterCount) {
                throw SqlError.createSQLException("There are unspecified parameters!", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
    }

    @Override
    public ResultSet executeQuery() throws SQLException {
        checkClosed();

        this.checkParamSpecifiedFinished();

        //build sql
        String sql = this.asSql();

        return connection.execQuerySql(this, sql);

    }

    @Override
    public int executeUpdate() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, "null", sqlType);
            this.isNull[parameterIndex - 1] = true;
        }
    }

    @Override
    public void setBoolean(int parameterIndex, boolean x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, x ? "1" : "0", Types.BOOLEAN);
        }
    }

    @Override
    public void setByte(int parameterIndex, byte x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.TINYINT);
        }
    }

    @Override
    public void setShort(int parameterIndex, short x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.SMALLINT);
        }
    }

    @Override
    public void setInt(int parameterIndex, int x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.INTEGER);
        }
    }

    @Override
    public void setLong(int parameterIndex, long x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.BIGINT);
        }
    }

    @Override
    public void setFloat(int parameterIndex, float x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.FLOAT);
        }
    }

    @Override
    public void setDouble(int parameterIndex, double x) throws SQLException {
        synchronized (this) {
            this.setInternal(parameterIndex, String.valueOf(x), Types.DOUBLE);
        }
    }

    @Override
    public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.DECIMAL);
                return;
            }
            this.setInternal(parameterIndex, x.toString(), Types.DECIMAL);
        }
    }

    @Override
    public void setString(int parameterIndex, String x) throws SQLException {
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.VARCHAR);
                return;
            }

            x = "'" + x + "'";

            this.setInternal(parameterIndex, x, Types.VARCHAR);
        }
    }

    private boolean isEscapeNeededForString(String x) {
        if (StringUtils.isBlank(x)) {
            return false;
        }
        boolean needsEscape = false;
        for (int i = 0; i < x.length(); ++i) {
            char c = x.charAt(i);
            switch (c) {
                case 0:
                case ':':
                case ' ':
                case '?':
                case '"':
                case '~':
                case '!':
                case '@':
                case '#':
                case '$':
                case '%':
                case '^':
                case '&':
                case '*':
                case '(':
                case ')':
                case '_':
                case '+':
                case '=':
                case '|':
                case '\\':
                case '{':
                case '}':
                case ';':
                case ',':
                case '<':
                case '>':
                    needsEscape = true;
                    break;
                default:
                    break;
            }

            if (needsEscape) {
                break;
            }
        }
        return needsEscape;
    }

    /**
     * @param paramIndex
     * @param val
     * @param type       java.sql.Types
     * @throws SQLException
     */
    private void setInternal(int paramIndex, String val, int type) throws SQLException {
        synchronized (this) {
            checkClosed();
            checkBounds(paramIndex);
            this.parameterValues[paramIndex - 1] = val;
            this.parameterTypes[paramIndex - 1] = type;
            this.hasSpecifiedParamCount++;
        }
    }

    @Override
    public void setBytes(int parameterIndex, byte[] x) throws SQLException {
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.ARRAY);
                return;
            }
            this.setInternal(parameterIndex, Arrays.toString(x), Types.ARRAY);
        }
    }

    @Override
    public void setDate(int parameterIndex, Date x) throws SQLException {
        this.setDate(parameterIndex, x, null);
    }

    @Override
    public void setTime(int parameterIndex, Time x) throws SQLException {
        this.setTime(parameterIndex, x, null);
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
        this.setTimestamp(parameterIndex, x, null);
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.setAsciiStream");
    }

    @Override
    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.setUnicodeStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.setBinaryStream");
    }

    @Override
    public void clearParameters() throws SQLException {
        synchronized (this) {
            this.checkClosed();
            for (int i = 0; i < this.parameterValues.length; ++i) {
                this.parameterValues[i] = null;
                this.isNull[i] = false;
                this.parameterTypes[i] = 0;
                this.hasSpecifiedParamCount = 0;
            }
        }
    }

    @Override
    public void setObject(int parameterIndex, Object parameterObj, int targetSqlType) throws SQLException {
        if (!(parameterObj instanceof BigDecimal)) {
            setObject(parameterIndex, parameterObj, targetSqlType, 0);
        } else {
            setObject(parameterIndex, parameterObj, targetSqlType, ((BigDecimal) parameterObj).scale());
        }
    }

    @Override
    public void setObject(int parameterIndex, Object parameterObj) throws SQLException {
        synchronized (this) {
            if (parameterObj == null) {
                setNull(parameterIndex, java.sql.Types.OTHER);
            } else {
                if (parameterObj instanceof Byte) {
                    setInt(parameterIndex, ((Byte) parameterObj).intValue());
                } else if (parameterObj instanceof String) {
                    setString(parameterIndex, (String) parameterObj);
                } else if (parameterObj instanceof BigDecimal) {
                    setBigDecimal(parameterIndex, (BigDecimal) parameterObj);
                } else if (parameterObj instanceof Short) {
                    setShort(parameterIndex, ((Short) parameterObj).shortValue());
                } else if (parameterObj instanceof Integer) {
                    setInt(parameterIndex, ((Integer) parameterObj).intValue());
                } else if (parameterObj instanceof Long) {
                    setLong(parameterIndex, ((Long) parameterObj).longValue());
                } else if (parameterObj instanceof Float) {
                    setFloat(parameterIndex, ((Float) parameterObj).floatValue());
                } else if (parameterObj instanceof Double) {
                    setDouble(parameterIndex, ((Double) parameterObj).doubleValue());
                } else if (parameterObj instanceof byte[]) {
                    setBytes(parameterIndex, (byte[]) parameterObj);
                } else if (parameterObj instanceof java.sql.Date) {
                    setDate(parameterIndex, (java.sql.Date) parameterObj);
                } else if (parameterObj instanceof Time) {
                    setTime(parameterIndex, (Time) parameterObj);
                } else if (parameterObj instanceof Timestamp) {
                    setTimestamp(parameterIndex, (Timestamp) parameterObj);
                } else if (parameterObj instanceof Boolean) {
                    setBoolean(parameterIndex, ((Boolean) parameterObj).booleanValue());
                } else if (parameterObj instanceof InputStream) {
                    setBinaryStream(parameterIndex, (InputStream) parameterObj, -1);
                } else if (parameterObj instanceof java.sql.Blob) {
                    setBlob(parameterIndex, (java.sql.Blob) parameterObj);
                } else if (parameterObj instanceof java.sql.Clob) {
                    setClob(parameterIndex, (java.sql.Clob) parameterObj);
                } else if (parameterObj instanceof java.util.Date) {
                    setTimestamp(parameterIndex, new Timestamp(((java.util.Date) parameterObj).getTime()));
                } else if (parameterObj instanceof BigInteger) {
                    setString(parameterIndex, parameterObj.toString());
                } else {
                    throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.setObject, parameterObj:" + parameterObj);
                }
            }
        }
    }

    @Override
    public boolean execute() throws SQLException {
        ResultSet newResults = this.executeQuery();
        synchronized (this) {
            this.results = newResults;
            return this.results != null;
        }
    }

    @Override
    public void addBatch() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.addBatch");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setCharacterStream");
    }

    @Override
    public void setRef(int parameterIndex, Ref x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setRef");
    }

    @Override
    public void setBlob(int parameterIndex, Blob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setBlob");
    }

    @Override
    public void setClob(int parameterIndex, Clob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setClob");
    }

    @Override
    public void setArray(int parameterIndex, Array x) throws SQLException {
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.ARRAY);
                return;
            }
            this.setInternal(parameterIndex, x.toString(), Types.ARRAY);
        }
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        if (null != this.results) {
            return this.results.getMetaData();
        } else {
            log.info("PreparedStatementImpl.getMetaData, results is null, try execute()");
            boolean hasResult = this.execute();
            if (hasResult) {
                return this.results.getMetaData();
            } else {
                throw SqlError.createSQLException("getMetaData, execute sql but get null, sql:" + asSql(), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
    }

    @Override
    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
        log.info("PreparedStatementImpl.setDate, parameterIndex:{}, Date:{}, cal:{}", parameterIndex, x, cal == null ? null : cal.getTimeZone());
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.DATE);
                return;
            }
            DateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd");
            if (cal == null) {
                String date = dateFormatter.format(x);
                this.setInternal(parameterIndex, date, Types.DATE);
                return;
            }
            dateFormatter.setCalendar(cal);
            String date = dateFormatter.format(x);
            this.setInternal(parameterIndex, date, Types.DATE);
        }
    }


    @Override
    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
        log.info("PreparedStatementImpl.setTime, parameterIndex:{}, Time:{}, cal:{}", parameterIndex, x, cal == null ? null : cal.getTimeZone());
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.TIME);
                return;
            }
            DateFormat timeFormatter = new SimpleDateFormat("HH:mm:ss");
            if (cal == null) {
                String time = timeFormatter.format(x);
                this.setInternal(parameterIndex, time, Types.TIME);
                return;
            }
            timeFormatter.setCalendar((Calendar) cal.clone());
            String time = timeFormatter.format(x);
            this.setInternal(parameterIndex, time, Types.TIME);
        }
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
        log.info("PreparedStatementImpl.setTimestamp, parameterIndex:{}, Timestamp:{}, cal:{}", parameterIndex, x, cal == null ? null : cal.getTimeZone());
        synchronized (this) {
            if (x == null) {
                this.setNull(parameterIndex, Types.TIMESTAMP);
                return;
            }
            DateFormat dateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            if (cal == null) {
                String timestamp = dateTimeFormatter.format(x);
                this.setInternal(parameterIndex, timestamp, Types.TIMESTAMP);
                return;
            }
            dateTimeFormatter.setCalendar((Calendar) cal.clone());
            String timestamp = dateTimeFormatter.format(x);
            this.setInternal(parameterIndex, timestamp, Types.TIMESTAMP);
        }
    }

    @Override
    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
        this.setNull(parameterIndex, sqlType);
    }

    @Override
    public void setURL(int parameterIndex, URL x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setURL");
    }

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
        log.info("PrepareStatementImpl.getParameterMetaData");
        synchronized (this) {
            return new ParameterMetaDataImpl(this.parameterCount, this.parameterTypes, this.isNull);
        }
    }

    @Override
    public void setRowId(int parameterIndex, RowId x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setRef");
    }

    @Override
    public void setNString(int parameterIndex, String value) throws SQLException {
        this.setString(parameterIndex, value);
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setNCharacterStream");
    }

    @Override
    public void setNClob(int parameterIndex, NClob value) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setNClob");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setClob");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setBlob");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setNClob");
    }

    @Override
    public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.setSQLXML");
    }

    @Override
    public void setObject(int parameterIndex, Object parameterObj, int targetSqlType, int scale) throws SQLException {
        synchronized (this) {
            if (parameterObj == null) {
                setNull(parameterIndex, java.sql.Types.OTHER);
                return;
            }
            try {
                switch (targetSqlType) {
                    case Types.BOOLEAN:
                        if (parameterObj instanceof Boolean) {
                            setBoolean(parameterIndex, ((Boolean) parameterObj).booleanValue());
                            break;
                        } else if (parameterObj instanceof String) {
                            setBoolean(parameterIndex, "true".equalsIgnoreCase((String) parameterObj) || !"0".equalsIgnoreCase((String) parameterObj));
                            break;
                        } else if (parameterObj instanceof Number) {
                            int intValue = ((Number) parameterObj).intValue();
                            setBoolean(parameterIndex, intValue != 0);
                            break;
                        } else {
                            throw SqlError.createSQLException("No conversion from " + parameterObj.getClass().getName() + " to Types.BOOLEAN possible.",
                                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
                        }
                    case Types.BIT:
                    case Types.TINYINT:
                    case Types.SMALLINT:
                    case Types.INTEGER:
                    case Types.BIGINT:
                    case Types.REAL:
                    case Types.FLOAT:
                    case Types.DOUBLE:
                    case Types.DECIMAL:
                    case Types.NUMERIC:
                        setNumericObject(parameterIndex, parameterObj, targetSqlType, scale);
                        break;
                    case Types.CHAR:
                    case Types.VARCHAR:
                    case Types.LONGVARCHAR:
                        if (parameterObj instanceof BigDecimal) {
                            setString(parameterIndex, JdbcUtils.fixDecimalExponent(parameterObj.toString()));
                        } else {
                            setString(parameterIndex, parameterObj.toString());
                        }
                        break;
                    case Types.CLOB:
                        if (parameterObj instanceof java.sql.Clob) {
                            setClob(parameterIndex, (java.sql.Clob) parameterObj);
                        } else {
                            setString(parameterIndex, parameterObj.toString());
                        }
                        break;
                    case Types.BINARY:
                    case Types.VARBINARY:
                    case Types.LONGVARBINARY:
                    case Types.BLOB:
                        if (parameterObj instanceof byte[]) {
                            setBytes(parameterIndex, (byte[]) parameterObj);
                        } else if (parameterObj instanceof java.sql.Blob) {
                            setBlob(parameterIndex, (java.sql.Blob) parameterObj);
                        } else {
                            setString(parameterIndex, parameterObj.toString());
                        }
                        break;
                    case Types.DATE:
                    case Types.TIMESTAMP:
                        java.util.Date parameterAsDate;
                        if (parameterObj instanceof String) {
                            java.text.DateFormat sdf = new SimpleDateFormat(getDateTimePattern((String) parameterObj, true));
                            ;
                            parameterAsDate = sdf.parse((String) parameterObj);
                        } else {
                            parameterAsDate = (java.util.Date) parameterObj;
                        }
                        switch (targetSqlType) {
                            case Types.DATE:
                                if (parameterAsDate instanceof java.sql.Date) {
                                    setDate(parameterIndex, (java.sql.Date) parameterAsDate);
                                } else {
                                    setDate(parameterIndex, new java.sql.Date(parameterAsDate.getTime()));
                                }
                                break;
                            case Types.TIMESTAMP:
                                if (parameterAsDate instanceof java.sql.Timestamp) {
                                    setTimestamp(parameterIndex, (java.sql.Timestamp) parameterAsDate);
                                } else {
                                    setTimestamp(parameterIndex, new java.sql.Timestamp(parameterAsDate.getTime()));
                                }
                                break;
                            default:

                        }
                        break;
                    case Types.TIME:
                        if (parameterObj instanceof String) {
                            java.text.DateFormat sdf = new SimpleDateFormat(getDateTimePattern((String) parameterObj, true));
                            ;
                            setTime(parameterIndex, new java.sql.Time(sdf.parse((String) parameterObj).getTime()));
                        } else if (parameterObj instanceof Timestamp) {
                            Timestamp xT = (Timestamp) parameterObj;
                            setTime(parameterIndex, new java.sql.Time(xT.getTime()));
                        } else {
                            setTime(parameterIndex, (java.sql.Time) parameterObj);
                        }
                        break;
                    case Types.OTHER:
                        setObject(parameterIndex, parameterObj);
                        break;
                    default:
                        throw SqlError.createSQLException("PreparedStatement.setObject unknown targetSqlType:" + targetSqlType, SqlError.SQL_STATE_GENERAL_ERROR);
                }
            } catch (Exception ex) {
                if (ex instanceof SQLException) {
                    throw (SQLException) ex;
                }
                SQLException sqlEx = SqlError.createSQLException(
                        "PreparedStatement.setObject Exception, parameterObj:" + parameterObj.getClass().toString() + ex.getMessage(),
                        SqlError.SQL_STATE_GENERAL_ERROR);
                sqlEx.initCause(ex);
                throw sqlEx;
            }
        }
    }

    private void setNumericObject(int parameterIndex, Object parameterObj, int targetSqlType, int scale) throws SQLException {
        Number parameterAsNum;
        if (parameterObj instanceof Boolean) {
            parameterAsNum = ((Boolean) parameterObj).booleanValue() ? Integer.valueOf(1) : Integer.valueOf(0);
        } else if (parameterObj instanceof String) {
            switch (targetSqlType) {
                case Types.BIT:
                    String stringOne = "1";
                    String stringZero = "0";
                    if (stringOne.equals(parameterObj) || stringZero.equals(parameterObj)) {
                        parameterAsNum = Integer.valueOf((String) parameterObj);
                    } else {
                        boolean parameterAsBoolean = "true".equalsIgnoreCase((String) parameterObj);
                        parameterAsNum = parameterAsBoolean ? Integer.valueOf(1) : Integer.valueOf(0);
                    }
                    break;
                case Types.TINYINT:
                case Types.SMALLINT:
                case Types.INTEGER:
                    parameterAsNum = Integer.valueOf((String) parameterObj);
                    break;
                case Types.BIGINT:
                    parameterAsNum = Long.valueOf((String) parameterObj);
                    break;
                case Types.REAL:
                    parameterAsNum = Float.valueOf((String) parameterObj);
                    break;
                case Types.FLOAT:
                case Types.DOUBLE:
                    parameterAsNum = Double.valueOf((String) parameterObj);
                    break;
                case Types.DECIMAL:
                case Types.NUMERIC:
                default:
                    parameterAsNum = new java.math.BigDecimal((String) parameterObj);
            }
        } else {
            parameterAsNum = (Number) parameterObj;
        }

        switch (targetSqlType) {
            case Types.BIT:
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
                setInt(parameterIndex, parameterAsNum.intValue());
                break;
            case Types.BIGINT:
                setLong(parameterIndex, parameterAsNum.longValue());
                break;
            case Types.REAL:
                setFloat(parameterIndex, parameterAsNum.floatValue());
                break;
            case Types.FLOAT:
            case Types.DOUBLE:
                setDouble(parameterIndex, parameterAsNum.doubleValue());
                break;
            case Types.DECIMAL:
            case Types.NUMERIC:
                if (parameterAsNum instanceof java.math.BigDecimal) {
                    BigDecimal scaledBigDecimal = null;
                    try {
                        scaledBigDecimal = ((java.math.BigDecimal) parameterAsNum).setScale(scale);
                    } catch (ArithmeticException ex) {
                        try {
                            scaledBigDecimal = ((java.math.BigDecimal) parameterAsNum).setScale(scale, BigDecimal.ROUND_HALF_UP);
                        } catch (ArithmeticException arEx) {
                            throw SqlError.createSQLException("Can't set scale of '" + scale + "' for DECIMAL argument '" + parameterAsNum + "'",
                                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
                        }
                    }
                    setBigDecimal(parameterIndex, scaledBigDecimal);
                } else if (parameterAsNum instanceof java.math.BigInteger) {
                    setBigDecimal(parameterIndex, new java.math.BigDecimal((java.math.BigInteger) parameterAsNum, scale));
                } else {
                    setBigDecimal(parameterIndex, new java.math.BigDecimal(parameterAsNum.doubleValue()));
                }
                break;
            default:
        }
    }

    private String getDateTimePattern(String dt, boolean toTime) throws Exception {
        //
        // Special case
        //
        int dtLength = (dt != null) ? dt.length() : 0;
        int dtLengthMin = 8;
        int dtLengthMax = 10;
        if ((dtLength >= dtLengthMin) && (dtLength <= dtLengthMax)) {
            int dashCount = 0;
            boolean isDateOnly = true;

            for (int i = 0; i < dtLength; i++) {
                char c = dt.charAt(i);

                if (!Character.isDigit(c) && (c != '-')) {
                    isDateOnly = false;

                    break;
                }

                if (c == '-') {
                    dashCount++;
                }
            }
            int dtDashCount = 2;
            if (isDateOnly && (dashCount == dtDashCount)) {
                return "yyyy-MM-dd";
            }
        }

        //
        // Special case - time-only
        //
        boolean colonsOnly = true;

        for (int i = 0; i < dtLength; i++) {
            char c = dt.charAt(i);

            if (!Character.isDigit(c) && (c != ':')) {
                colonsOnly = false;

                break;
            }
        }

        if (colonsOnly) {
            return "HH:mm:ss";
        }

        int n;
        int z;
        int count;
        int maxvecs;
        char c;
        char separator;
        StringReader reader = new StringReader(dt + " ");
        ArrayList<Object[]> vec = new ArrayList<Object[]>();
        ArrayList<Object[]> vecRemovelist = new ArrayList<Object[]>();
        Object[] nv = new Object[3];
        Object[] v;
        nv[0] = Character.valueOf('y');
        nv[1] = new StringBuilder();
        nv[2] = Integer.valueOf(0);
        vec.add(nv);

        if (toTime) {
            nv = new Object[3];
            nv[0] = Character.valueOf('h');
            nv[1] = new StringBuilder();
            nv[2] = Integer.valueOf(0);
            vec.add(nv);
        }

        while ((z = reader.read()) != -1) {
            separator = (char) z;
            maxvecs = vec.size();

            for (count = 0; count < maxvecs; count++) {
                v = vec.get(count);
                n = ((Integer) v[2]).intValue();
                c = getSuccessor(((Character) v[0]).charValue(), n);

                if (!Character.isLetterOrDigit(separator)) {
                    if ((c == ((Character) v[0]).charValue()) && (c != 'S')) {
                        vecRemovelist.add(v);
                    } else {
                        ((StringBuilder) v[1]).append(separator);

                        if ((c == 'X') || (c == 'Y')) {
                            v[2] = Integer.valueOf(4);
                        }
                    }
                } else {
                    if (c == 'X') {
                        c = 'y';
                        nv = new Object[3];
                        nv[1] = (new StringBuilder(((StringBuilder) v[1]).toString())).append('M');
                        nv[0] = Character.valueOf('M');
                        nv[2] = Integer.valueOf(1);
                        vec.add(nv);
                    } else if (c == 'Y') {
                        c = 'M';
                        nv = new Object[3];
                        nv[1] = (new StringBuilder(((StringBuilder) v[1]).toString())).append('d');
                        nv[0] = Character.valueOf('d');
                        nv[2] = Integer.valueOf(1);
                        vec.add(nv);
                    }

                    ((StringBuilder) v[1]).append(c);

                    if (c == ((Character) v[0]).charValue()) {
                        v[2] = Integer.valueOf(n + 1);
                    } else {
                        v[0] = Character.valueOf(c);
                        v[2] = Integer.valueOf(1);
                    }
                }
            }

            int size = vecRemovelist.size();

            for (int i = 0; i < size; i++) {
                v = vecRemovelist.get(i);
                vec.remove(v);
            }

            vecRemovelist.clear();
        }

        int size = vec.size();

        for (int i = 0; i < size; i++) {
            v = vec.get(i);
            c = ((Character) v[0]).charValue();
            n = ((Integer) v[2]).intValue();

            boolean bk = getSuccessor(c, n) != c;
            boolean atEnd = (((c == 's') || (c == 'm') || ((c == 'h') && toTime)) && bk);
            boolean finishesAtDate = (bk && (c == 'd') && !toTime);
            boolean containsEnd = (((StringBuilder) v[1]).toString().indexOf('W') != -1);

            boolean res = (!atEnd && !finishesAtDate) || (containsEnd);
            if (res) {
                vecRemovelist.add(v);
            }
        }

        size = vecRemovelist.size();

        for (int i = 0; i < size; i++) {
            vec.remove(vecRemovelist.get(i));
        }

        vecRemovelist.clear();
        v = vec.get(0);

        StringBuilder format = (StringBuilder) v[1];
        format.setLength(format.length() - 1);

        return format.toString();
    }

    private final char getSuccessor(char c, int n) {
        int two = 2;
        int three = 3;
        int four = 4;
        char y = 'y';
        char d = 'd';
        char m = 'm';
        char s = 's';
        char mCaps = 'M';
        char hCaps = 'H';

        if ((c == y) && (n == two)) {
            return 'X';
        }
        if (((c == y) && (n < four))) {
            return 'y';
        }
        if (c == y) {
            return 'M';
        }
        if ((c == mCaps) && (n == two)) {
            return 'Y';
        }
        if ((c == mCaps) && (n < three)) {
            return 'M';
        }
        if (c == mCaps) {
            return 'd';
        }
        if ((c == d) && (n < two)) {
            return 'd';
        }
        if (c == d) {
            return 'H';
        }
        if ((c == hCaps) && (n < two)) {
            return 'H';
        }
        if (c == hCaps) {
            return 'm';
        }
        if ((c == m) && (n < two)){
            return 'm';
        }
        if (c == m) {
            return 's';
        }
        if ((c == s) && (n < two)) {
            return 's';
        }else {
            return 'W';
        }
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setAsciiStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setBinaryStream");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setCharacterStream");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setAsciiStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setBinaryStream");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setCharacterStream");
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setNCharacterStream");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setClob");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setBlob");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("setNClob");
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PrepareStatementImpl.executeQuery");
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public void close() throws SQLException {
        log.info("PrepareStatementImpl.close");
        synchronized (this) {
            this.isClosed = true;
            this.results = null;
            this.connection = null;
            this.parameterValues = null;
            this.isNull = null;
            this.parameterTypes = null;
            this.staticSql = null;
        }
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        return this.maxFieldSize;
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        this.maxFieldSize = max;
    }

    @Override
    public int getMaxRows() throws SQLException {
        return this.maxRows;
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        this.maxRows = max;
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        log.info("PreparedStatementImpl.setEscapeProcessing");
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        return 300;
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        log.info("PreparedStatementImpl.setQueryTimeout");
    }

    @Override
    public void cancel() throws SQLException {
        log.info("PreparedStatementImpl.cancel");
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        log.info("StatementImpl.getWarnings");
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {

    }

    @Override
    public void setCursorName(String name) throws SQLException {
        log.info("StatementImpl.setCursorName");
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.execute");
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
        synchronized (this) {
            ResultSet res = this.results;
            this.results = null;
            return res;
        }
    }

    @Override
    public int getUpdateCount() throws SQLException {
        return -1;
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        return false;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        synchronized (this) {
            ResultSetImpl.checkSupportFetchDirection(direction);
            this.fetchDirection = direction;
        }
    }

    @Override
    public int getFetchDirection() throws SQLException {
        return this.fetchDirection;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        this.fetchSize = rows;
    }

    @Override
    public int getFetchSize() throws SQLException {
        return this.fetchSize;
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        return this.resultSetConcurrency;
    }

    @Override
    public int getResultSetType() throws SQLException {
        return this.resultSetType;
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.addBatch");
    }

    @Override
    public void clearBatch() throws SQLException {
        log.info("PreparedStatementImpl.clearBatch");
    }

    @Override
    public int[] executeBatch() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.executeBatch");
    }

    @Override
    public Connection getConnection() throws SQLException {
        return this.connection;
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        log.info("PreparedStatementImpl.getMoreResults");
        return false;
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        QueryResponse queryResponse = new QueryResponse();
        List<String> columns = new ArrayList<>(4);
        columns.add("GENERATED_KEY");
        List<String> metadata = new ArrayList<>(4);
        metadata.add("VARCHAR");
        List<Map<String, Object>> rows = new ArrayList<>(0);
        queryResponse.setColumns(columns);
        queryResponse.setMetadata(metadata);
        queryResponse.setRows(rows);
        return new ResultSetImpl(this.connection, this.connection.getPrivateStatement(), queryResponse);
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.executeUpdate");
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.executeUpdate");
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.executeUpdate");
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.execute");
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.execute");
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("PreparedStatementImpl.execute");
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        return java.sql.ResultSet.HOLD_CURSORS_OVER_COMMIT;
    }

    @Override
    public boolean isClosed() throws SQLException {
        return this.isClosed;
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {

    }

    @Override
    public boolean isPoolable() throws SQLException {
        return false;
    }

    @Override
    public void closeOnCompletion() throws SQLException {

    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        return false;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
