package com.eniot.data.query.impl;

import com.eniot.data.query.Driver;
import com.eniot.data.query.entity.EnosResponse;
import com.eniot.data.query.entity.ListChannelResp;
import com.eniot.data.query.entity.QueryResponse;
import com.eniot.data.query.entity.SupportedDataSoureTypeEnum;
import com.eniot.data.query.exception.SqlError;
import com.envision.apim.poseidon.config.PConfig;
import com.envision.apim.poseidon.core.Poseidon;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.util.*;

import static com.eniot.data.query.entity.DataQueryExceptionCode.*;

/**
 * exec every api
 *
 * @author jinghui.zhao
 * @date 2020/1/9
 */
public class RestClient {

    private static final Logger log = LoggerFactory.getLogger(RestClient.class);

    private String orgId = null;

    private String host;

    private String chId;

    private String source = null;

    private volatile boolean connected = false;

    private String user;

    private String password;

    private PConfig.ProxyConfig proxyConfig;

    private String httpPrefix = "https://";

    private String dataStorageManager = "data-storage-manager";

    private String bear = "Bearer ";

    private Gson gson = new Gson();

    private static RestClient restClient;

    private String apimQueryUrl;

    private String apimLoginUrl;

    private String apimRefreshTokenUrl;

    private String apimListChannelsUrl;

    private Set<String> supportedDataSourceNameSet = new HashSet<>(2);

    private static final String SHOW_DATABASES = "show databases";
    private static final String SHOW_SCHEMAS = "show schemas";
    private static final String SCHEMA_NAME = "SCHEMA_NAME";
    private static final String SHOW_FILES = "show files";
    private static final String SHOW_TABLES = "show tables";
    private static final String DESC = "desc";

    private RestClient(String host, String chId, Properties info) {

        this.user = info.getProperty(Driver.USER_PROPERTY_KEY);
        this.password = info.getProperty("password");
        this.chId = chId;
        this.source = info.getProperty(Driver.SOURCE_PROPERTY_KEY);
        this.host = host;
        this.orgId = info.getProperty(Driver.ORGID_PROPERTY_KEY);
        String proxyIp = info.getProperty("proxyIp");
        String proxyPort = info.getProperty("proxyPort");
        if (proxyIp != null && proxyPort != null) {
            this.proxyConfig = new PConfig.ProxyConfig(proxyIp, Integer.parseInt(proxyPort));
        }

        this.apimQueryUrl = httpPrefix + host + "?orgId=" + orgId;

        this.apimLoginUrl = httpPrefix + host.substring(0, host.indexOf('/')) + "/apim-token-service/v2.0/token/get";
        this.apimRefreshTokenUrl = httpPrefix + host.substring(0, host.indexOf('/')) + "/apim-token-service/v2.0/token/refresh";
        this.apimListChannelsUrl = httpPrefix + host.substring(0, host.lastIndexOf("/read")) + "?orgId=" + this.orgId + "&channelType=READ";

        log.info("apimQueryUrl:{}", apimQueryUrl);
        log.info("apimLoginUrl:{}", apimLoginUrl);
        log.info("apimRefreshTokenUrl:{}", apimRefreshTokenUrl);
        log.info("apimListChannelsUrl:{}", apimListChannelsUrl);

    }

    public synchronized static RestClient getRestClient(String host, String chId, Properties info) throws SQLException {
        if (RestClient.restClient == null || resetLoginInfoNeeded(host, chId, info)) {
            RestClient.restClient = new RestClient(host, chId, info);
            RestClient.restClient.connect();
        }
        return RestClient.restClient;
    }

    private static boolean resetLoginInfoNeeded(String host, String chId, Properties info) {
        if (!RestClient.restClient.host.equals(host) || !RestClient.restClient.chId.equals(chId)) {
            return true;
        }
        if (!RestClient.restClient.orgId.equals(info.getProperty(Driver.ORGID_PROPERTY_KEY))) {
            return true;
        }
        return !RestClient.restClient.user.equals(info.getProperty(Driver.USER_PROPERTY_KEY)) ||
                !RestClient.restClient.password.equals(info.getProperty("password"));
    }

    /**
     * Start's a connection from client to server
     *
     * @throws SQLException
     */
    private void connect() throws SQLException {
        if (connected) {
            return;
        }

        apimCheckChannelsInfo();

        connected = true;
        log.info("Successfully connected to query server");
    }

    public QueryResponse execQuery(String sql) throws SQLException {
        return this.execApimQuery(sql);
    }

    private void apimCheckChannelsInfo() throws SQLException {
        log.info("RestClient.apimCheckChannelsInfo");
        try {
            String response = Poseidon.config(PConfig.init().appKey(this.user).appSecret(this.password).proxyConfig(this.proxyConfig)
                    .connectTimeout(0).readTimeout(0).writeTimeout(0))
                    .url(this.apimListChannelsUrl)
                    .header("Content-Type", "application/json")
                    .method("GET")
                    .sync();
            ListChannelResp listChannelResp = gson.fromJson(response, ListChannelResp.class);
            if (listChannelResp.getCode() != 0) {
                log.error("RestClient apimCheckChannelsInfo failed!, apimListChannelsUrl = {}, response = {}", apimListChannelsUrl, response);
                throw SqlError.createSQLException(String.format("CheckChannelsInfo failed! code = %d, msg = %s.", listChannelResp.getCode(), listChannelResp.getMsg()),
                        SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
            }

            if (listChannelResp.getData() != null && !listChannelResp.getData().isEmpty()) {
                for (ListChannelResp.ChannelDataInfo channelDataInfo : listChannelResp.getData()) {
                    if (this.chId.equals(channelDataInfo.getChannelId())) {
                        if (channelDataInfo.getDataSourceInfo() != null && !listChannelResp.getData().isEmpty()) {
                            for (ListChannelResp.DataSourceInfo dataSourceInfo : channelDataInfo.getDataSourceInfo()) {
                                if (SupportedDataSoureTypeEnum.getSupportedDataSoureType(dataSourceInfo.getDataSourceType()) != null) {
                                    supportedDataSourceNameSet.add(dataSourceInfo.getAlias());
                                }
                            }
                        }

                    }
                }
            }
            if (supportedDataSourceNameSet.isEmpty()) {
                log.error("No valid data source was found for chId = {}!", this.chId);
                throw SqlError.createSQLException(String.format("No valid data source was found for chId = %s!", this.chId),
                        SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
            }
        } catch (SQLException sqlException) {
            throw sqlException;
        } catch (Exception e) {
            log.error("RestClient apimCheckChannelsInfo error, apimListChannelsUrl = {}, exception: {}", apimListChannelsUrl, e);
            throw SqlError.createSQLException("apimCheckChannelsInfo error! " + e.getMessage(),
                    SqlError.SQL_STATE_UNABLE_TO_CONNECT_TO_DATASOURCE);
        }
    }

    private QueryResponse execApimQuery(String sql) throws SQLException {
        log.info("RestClient.execApimQuery, sql:{}", sql);
        String resp = "";
        try {
            long refreshDiffTimeMax = 600000L;
            resp = apimQuery(sql, this.source);
            EnosResponse enosResponse = gson.fromJson(resp, EnosResponse.class);
            if (enosResponse.getCode() == SC_FORBIDDEN) {
                resp = apimQuery(sql, this.source);
                enosResponse = gson.fromJson(resp, EnosResponse.class);
            }
            if (enosResponse.getCode() == SUCCESS) {
                if (SHOW_SCHEMAS.equals(sql.toLowerCase(Locale.ENGLISH)) || SHOW_DATABASES.equals(sql.toLowerCase(Locale.ENGLISH))) {
                    if (source == null) {
                        return filterDatabaseRows(enosResponse.getData());
                    }
                }
                return enosResponse.getData();
            }
            if (enosResponse.getCode() == EXEC_SQL_ERROR) {
                log.error("RestClient execApimQuery failed, status = {}, msg = {}", enosResponse.getCode(), "Illegal sql");
                throw SqlError.createSQLException(String.format("execApimQuery failed! status = %d, msg = %s.", enosResponse.getCode(), "Illegal sql"),
                        SqlError.SQL_STATE_QUERY_ERROR);
            }
            log.error("RestClient execApimQuery failed, status = {}, msg = {}", enosResponse.getCode(), enosResponse.getMsg());
            throw SqlError.createSQLException(String.format("execApimQuery failed! status = %d, msg = %s.", enosResponse.getCode(), enosResponse.getMsg()),
                    SqlError.SQL_STATE_QUERY_ERROR);
        } catch (SQLException sqlException) {
            throw sqlException;
        } catch (Exception e) {
            log.error("RestClient execApimQuery error!, apimQueryUrl:{}, resp:{}, exception:{}", apimQueryUrl, resp, e);
            throw SqlError.createSQLException("execQuery error! " + e.getMessage(),
                    SqlError.SQL_STATE_QUERY_ERROR);
        }
    }

    private String apimQuery(String sql, String source) {
        JsonObject requestBody = new JsonObject();
        requestBody.addProperty("sqlQuery", sql);
        if (source != null) {
            requestBody.addProperty("source", source);
        }
        return Poseidon.config(PConfig.init().appKey(this.user).appSecret(this.password).proxyConfig(this.proxyConfig)
                .connectTimeout(0).readTimeout(0).writeTimeout(0))
                .url(this.apimQueryUrl)
                .requestBody(requestBody.toString())
                .method("POST")
                .sync();
    }

    private QueryResponse filterDatabaseRows(QueryResponse queryResponse) {
        if (queryResponse == null || queryResponse.getRows() == null || queryResponse.getRows().isEmpty()) {
            return queryResponse;
        }
        List<Map<String, Object>> rowsNew = new ArrayList<>();
        for (Map<String, Object> row : queryResponse.getRows()) {
            String schema = (String) row.get(SCHEMA_NAME);
            if (StringUtils.isNotBlank(schema) && schema.contains(".")) {
                String dataSourceName = schema.substring(0, schema.indexOf('.'));
                if (supportedDataSourceNameSet.contains(dataSourceName)) {
                    rowsNew.add(row);
                }
            }
        }
        queryResponse.setRows(rowsNew);
        return queryResponse;
    }
}
