package com.eniot.data.query.impl;

import com.eniot.data.query.EniotConnect;
import com.eniot.data.query.EniotResultSet;
import com.eniot.data.query.entity.DataType;
import com.eniot.data.query.entity.QueryResponse;
import com.eniot.data.query.exception.SqlError;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.*;
import java.sql.Date;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Pattern;

/**
 * @author jinghui.zhao
 * @date 2020/1/17
 */
public class ResultSetImpl implements EniotResultSet {
    private static final Logger log = LoggerFactory.getLogger(ResultSetImpl.class);

    /**
     * raw index, start from 1
     */
    private Integer index = 0;

    private EniotConnect connect;

    private Statement statement;

    private String catalog;

    private boolean isClosed = false;

    private boolean wasNullFlag = false;

    private boolean isBeforeFirst = false;

    private boolean isAfterLast = false;

    private int fetchSize = 640000;

    /**
     * column name
     */
    private List<String> columns;

    /**
     * column metadata
     */
    private List<Integer> metaType;

    /**
     * data
     */
    private List<Map<String, Object>> rows;

    private Map<String, Integer> columnNameToIndex;

    private int fetchDirection = FETCH_FORWARD;

    Calendar defaultCalendar = new GregorianCalendar();

    private static Pattern DATE_TIME_PATTERN = Pattern.compile("^(\\d{4}-\\d{2}-\\d{2}\\s\\d{2}:\\d{2}:\\d{2})(\\.\\d{1,3}?)?$");

    private static Pattern DATE_TIME_PATTERN_T = Pattern.compile("^(\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2})(\\.\\d{1,3}?)?$");

    private static Pattern DATE_PATTERN = Pattern.compile("^\\d{4}-\\d{2}-\\d{2}$");

    private static Pattern TIME_PATTERN = Pattern.compile("^(\\d{2}:\\d{2}:\\d{2})(\\.\\d{1,3}?)?$");

    private static Pattern DATE_PATTERN_T = Pattern.compile("^\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}$");

    private static Pattern DATE_PATTERN_Z = Pattern.compile("^(\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2})(\\.\\d{1,3}\\+\\d{1,4})?$");

    public ResultSetImpl(EniotConnect connect, Statement statement, QueryResponse queryResponse) throws SQLException {
        this.connect = connect;
        this.statement = statement;
        this.columns = queryResponse.getColumns();
        this.rows = queryResponse.getRows();
        this.catalog = connect.getCatalog();

        /**
         * init meta
         */
        this.metaType = formatDataMetaType(queryResponse.getColumns(), queryResponse.getMetadata());

        columnNameToIndex = initColumnNameMap(this.columns);

    }

    private List<Integer> formatDataMetaType(List<String> columns, List<String> metadata) {
        if (null == columns || columns.isEmpty()) {
            return null;
        }
        if (null == metadata || metadata.isEmpty()) {
            return null;
        }
        List<Integer> metaType = new ArrayList<>(columns.size());
        for (int i = 0; i < columns.size(); i++) {
            String meta = metadata.get(i);
            if (meta.startsWith(DataType.VARCHAR.getType())) {
                metaType.add(Types.VARCHAR);
            } else if (meta.startsWith(DataType.BIT.getType())) {
                metaType.add(Types.BIT);
            } else if (meta.startsWith(DataType.INT.getType())) {
                metaType.add(Types.INTEGER);
            } else if (meta.startsWith(DataType.BIGINT.getType())) {
                metaType.add(Types.BIGINT);
            } else if (meta.startsWith(DataType.DOUBLE.getType())) {
                metaType.add(Types.DOUBLE);
            } else if (meta.startsWith(DataType.FLOAT.getType())) {
                metaType.add(Types.FLOAT);
            } else if (meta.startsWith(DataType.DATE.getType())) {
                metaType.add(Types.DATE);
            } else if (meta.startsWith(DataType.TIMESTAMP.getType())) {
                metaType.add(Types.TIMESTAMP);
            } else if (meta.startsWith(DataType.TIME.getType())) {
                metaType.add(Types.TIME);
            } else {
                metaType.add(Types.VARCHAR);
            }
        }
        return metaType;
    }

    private Map<String, Integer> initColumnNameMap(List<String> columns) {
        if (null == columns || columns.isEmpty()) {
            return null;
        }
        Map<String, Integer> columnNameToIndex = new HashMap<>(columns.size());
        for (int i = 0; i < columns.size(); i++) {
            columnNameToIndex.put(columns.get(i), i);
        }
        return columnNameToIndex;
    }

    @Override
    public String getCatalog() {
        return this.catalog;
    }

    /**
     * Ensures that the result set is not closed
     *
     * @throws SQLException if the result set is closed
     */
    private void checkClosed() throws SQLException {
        if (this.isClosed) {
            log.warn("ResultSet has closed!");
            throw SqlError.createSQLException("Operation not allowed after ResultSet closed",
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    /**
     * Checks if columnIndex is within the number of columns in this result set.
     *
     * @param columnIndex the index to check
     * @throws SQLException if the index is out of bounds
     */
    private void checkColumnBounds(int columnIndex) throws SQLException {
        if (columnIndex < 1) {
            throw SqlError.createSQLException(String.format("Column index out of range, %d < 1.", columnIndex), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        } else if (columnIndex > this.columns.size()) {
            throw SqlError.createSQLException(String.format("Column index out of range, %d > %d.", columnIndex, columns.size()),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    private Boolean onValidRow = false;
    private String invalidRowReason = "On valid row";

    private void setRowPositionValidity() throws SQLException {
        synchronized (this) {
            if (this.rows == null || this.rows.isEmpty()) {
                onValidRow = false;
                invalidRowReason = "Illegal operation on empty result set";
            } else if (this.isBeforeFirst) {
                onValidRow = false;
                invalidRowReason = "Before start of result set";
            } else if (this.isAfterLast) {
                onValidRow = false;
                invalidRowReason = "After end of result set";
            }
            this.onValidRow = true;
        }
    }

    /**
     * Ensures that the cursor is positioned on a valid row and that the result
     * set is not closed
     *
     * @throws SQLException if the result set is not in a valid state for traversal
     */
    private void checkRowPos() throws SQLException {
        synchronized (this) {
            checkClosed();
            if (!this.onValidRow) {
                throw SqlError.createSQLException(this.invalidRowReason, SqlError.SQL_STATE_GENERAL_ERROR);
            }
        }
    }

    @Override
    public boolean next() throws SQLException {
        if (this.rows == null || this.rows.isEmpty()) {
            return false;
        }
        synchronized (this) {
            if (this.index + 1 <= this.rows.size()) {
                setRowPositionValidity();
                this.index++;
                return true;
            }
        }
        return false;
    }

    @Override
    public void close() throws SQLException {
        log.info("ResultSetImpl.close");
        synchronized (this) {
            if (this.isClosed) {
                return;
            }
            this.columns = null;
            this.rows = null;
            this.statement = null;
            this.connect = null;
            this.metaType = null;
            this.onValidRow = false;
            this.columnNameToIndex = null;
            this.isClosed = true;
        }
    }

    @Override
    public boolean wasNull() throws SQLException {
        synchronized (this) {
            boolean wnf = this.wasNullFlag;
            this.wasNullFlag = false;
            return wnf;
        }
    }

    /**
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL <code>NULL</code>, the
     * value returned is <code>null</code>
     * @throws SQLException if the columnIndex is not valid;
     *                      if a database access error occurs or this method is
     *                      called on a closed result set
     */
    @Override
    public String getString(int columnIndex) throws SQLException {
        Object value = this.getObject(columnIndex);
        return value == null ? null : String.valueOf(value);
    }

    private String getColumnNameByIndex(int columnIndex) throws SQLException {
        if (null == this.columns || this.columns.isEmpty()) {
            throw SqlError.createSQLException("Illegal operation on empty result set!",
                    SqlError.SQL_STATE_EMPTY_RESULT);
        }
        checkColumnBounds(columnIndex);
        return this.columns.get(columnIndex - 1);
    }


    private Integer getColumnDataTypeByIndex(int columnIndex) throws SQLException {
        if (null == this.metaType || this.metaType.isEmpty()) {
            throw SqlError.createSQLException("ResultSet is empty!",
                    SqlError.SQL_STATE_EMPTY_RESULT);
        }
        checkColumnBounds(columnIndex);
        return this.metaType.get(columnIndex - 1);
    }

    /**
     * <P>If the designated column has a datatype of CHAR or VARCHAR
     * and contains a "0" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 0, a value of <code>false</code> is returned.  If the designated column has a datatype
     * of CHAR or VARCHAR
     * and contains a "1" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 1, a value of <code>true</code> is returned.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL <code>NULL</code>, the
     * value returned is <code>false</code>
     * @throws SQLException if the columnIndex is not valid;
     *                      if a database access error occurs or this method is
     *                      called on a closed result set
     */
    @Override
    public boolean getBoolean(int columnIndex) throws SQLException {
        Integer columnDataType = getColumnDataTypeByIndex(columnIndex);
        switch (columnDataType) {
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.DECIMAL:
            case Types.NUMERIC:
            case Types.REAL:
            case Types.FLOAT:
            case Types.DOUBLE:
                long longVal = getLong(columnIndex);
                return (longVal == -1 || longVal > 0);
            default:
                String stringVal = getString(columnIndex);
                return getBooleanFromString(stringVal);
        }
    }

    private boolean getBooleanFromString(String stringVal) throws SQLException {
        if ((stringVal != null) && (stringVal.length() > 0)) {
            int c = Character.toLowerCase(stringVal.charAt(0));
            return ((c == 't') || (c == 'y') || (c == '1') || "-1".equals(stringVal));
        }
        return false;
    }

    @Override
    public byte getByte(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            double valueAsDouble = Double.parseDouble(val);
            return (byte) valueAsDouble;
        } catch (NumberFormatException nfe) {
            throw SqlError.createSQLException(String.format("Invalid value for getByte(), value:%s", val),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public short getShort(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            double valueAsDouble = Double.parseDouble(val);
            return (short) valueAsDouble;
        } catch (NumberFormatException nfe) {
            throw SqlError.createSQLException(String.format("Invalid value for getShort(), value:%s", val),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this <code>ResultSet</code> object as
     * an <code>int</code> in the Java programming language.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL <code>NULL</code>, the
     * value returned is <code>0</code>
     * @throws SQLException if the columnIndex is not valid;
     *                      if a database access error occurs or this method is
     *                      called on a closed result set
     */
    @Override
    public int getInt(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            double valueAsDouble = Double.parseDouble(val);
            return (int) valueAsDouble;
        } catch (NumberFormatException nfe) {
            String yesKey = "YES";
            String noKey = "NO";
            if (yesKey.equalsIgnoreCase(val)) {
                return 1;
            } else if (noKey.equalsIgnoreCase(val)) {
                return 0;
            }
            throw SqlError.createSQLException(String.format("Invalid value for getInt(), column:%s, value:%s, type:%d", this.getColumnNameByIndex(columnIndex), val, this.getColumnDataTypeByIndex(columnIndex)),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    private void convertToZeroWithEmptyCheck() throws SQLException {
        throw SqlError.createSQLException("Can't convert empty string ('') to numeric", SqlError.SQL_STATE_INVALID_CHARACTER_VALUE_FOR_CAST);
    }

    @Override
    public long getLong(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            double valueAsDouble = Double.parseDouble(val);
            return (long) valueAsDouble;
        } catch (NumberFormatException nfe) {
            throw SqlError.createSQLException(String.format("Invalid value for getLong(), value:%s", val),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public float getFloat(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            double valueAsDouble = Double.parseDouble(val);
            return (float) valueAsDouble;
        } catch (NumberFormatException nfe) {
            throw SqlError.createSQLException(String.format("Invalid value for getFloat(), value:%s", val),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public double getDouble(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return 0;
        }
        if (val.length() == 0) {
            convertToZeroWithEmptyCheck();
        }
        try {
            return Double.parseDouble(val);
        } catch (NumberFormatException nfe) {
            throw SqlError.createSQLException(String.format("Invalid value for getDouble(), value:%s", val),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public BigDecimal getBigDecimal(int columnIndex, int scale) throws SQLException {
        BigDecimal val = this.getBigDecimal(columnIndex);
        return val == null ? null : val.setScale(scale);
    }

    @Override
    public byte[] getBytes(int columnIndex) throws SQLException {
        String val = this.getString(columnIndex);
        return val == null ? null : val.getBytes();
    }

    @Override
    public Date getDate(int columnIndex) throws SQLException {
        return this.getDate(columnIndex, null);
    }

    private java.util.Date formatUtilDate(String dateValue, Calendar cal) throws SQLException {
        if (dateValue == null) {
            return null;
        }
        java.util.Date utilDate = null;
        try {
            if (cal == null) {
                cal = defaultCalendar;
            }
            if (DATE_TIME_PATTERN.matcher(dateValue).matches()) {
                DateFormat dateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                dateTimeFormatter.setCalendar(cal);
                utilDate = dateTimeFormatter.parse(dateValue);
            } else if (DATE_TIME_PATTERN_T.matcher(dateValue).matches()) {
                DateFormat dateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                dateTimeFormatter.setCalendar(cal);
                utilDate = dateTimeFormatter.parse(dateValue.replace("T", " "));
            } else if (DATE_PATTERN.matcher(dateValue).matches()) {
                DateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd");
                dateFormatter.setCalendar(cal);
                utilDate = dateFormatter.parse(dateValue);
            } else if (DATE_PATTERN_T.matcher(dateValue).matches()) {
                dateValue = dateValue.replace("T", " ");
                dateValue = dateValue + ":00";
                DateFormat dateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                dateTimeFormatter.setCalendar(cal);
                utilDate = dateTimeFormatter.parse(dateValue);
            } else if (TIME_PATTERN.matcher(dateValue).matches()) {
                DateFormat timeFormatter = new SimpleDateFormat("HH:mm:ss");
                timeFormatter.setCalendar(cal);
                utilDate = new Date(timeFormatter.parse(dateValue).getTime());
            } else if(DATE_PATTERN_Z.matcher(dateValue).matches()) {
                SimpleDateFormat timeFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS+SSSS");
                timeFormatter.setCalendar(cal);
                utilDate = new Date(timeFormatter.parse(dateValue).getTime());
            } else {
                log.warn("ResultSetImpl.formatUtilDate, nonsupport date format, dateValue:{}", dateValue);
            }
        } catch (ParseException parseException) {
            log.warn("ResultSetImpl.formatUtilDate parse date exception", parseException);
            throw SqlError.createSQLException(
                    "ResultSetImpl.formatUtilDate parse date exception" + parseException.getMessage(),
                    SqlError.SQL_STATE_ILLEGAL_OPERATION);
        }
        return utilDate;
    }

    @Override
    public Time getTime(int columnIndex) throws SQLException {
        return this.getTime(columnIndex, null);
    }

    @Override
    public Timestamp getTimestamp(int columnIndex) throws SQLException {
        return this.getTimestamp(columnIndex, null);
    }

    @Override
    public InputStream getAsciiStream(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getAsciiStream");
        throw SqlError.createSQLFeatureNotSupportedException("getAsciiStream");
    }

    @Override
    public InputStream getUnicodeStream(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getUnicodeStream");
        throw SqlError.createSQLFeatureNotSupportedException("getUnicodeStream");
    }

    @Override
    public InputStream getBinaryStream(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getBinaryStream");
        throw SqlError.createSQLFeatureNotSupportedException("getBinaryStream");
    }

    private int findColumnIndex(String column) throws SQLException {
        Integer index = this.columnNameToIndex.get(column);
        if (null != index) {
            return index + 1;
        } else {
            for (int i = 0; i < this.columns.size(); i++) {
                if (columns.get(i).equalsIgnoreCase(column)) {
                    return i + 1;
                }
            }
        }
        throw SqlError.createSQLException(String.format("ResultSet columnName: %s, not found.", column), SqlError.SQL_STATE_COLUMN_NOT_FOUND);

    }

    @Override
    public String getString(String columnLabel) throws SQLException {
        return this.getString(this.findColumnIndex(columnLabel));
    }

    @Override
    public boolean getBoolean(String columnLabel) throws SQLException {
        return this.getBoolean(this.findColumnIndex(columnLabel));
    }

    @Override
    public byte getByte(String columnLabel) throws SQLException {
        return this.getByte(this.findColumnIndex(columnLabel));
    }

    @Override
    public short getShort(String columnLabel) throws SQLException {
        return this.getShort(this.findColumnIndex(columnLabel));
    }

    @Override
    public int getInt(String columnLabel) throws SQLException {
        return this.getInt(this.findColumnIndex(columnLabel));
    }

    @Override
    public long getLong(String columnLabel) throws SQLException {
        return this.getLong(this.findColumnIndex(columnLabel));
    }

    @Override
    public float getFloat(String columnLabel) throws SQLException {
        return this.getFloat(this.findColumnIndex(columnLabel));
    }

    @Override
    public double getDouble(String columnLabel) throws SQLException {
        return this.getDouble(this.findColumnIndex(columnLabel));
    }

    @Override
    public BigDecimal getBigDecimal(String columnLabel, int scale) throws SQLException {
        return this.getBigDecimal(this.findColumnIndex(columnLabel), scale);
    }

    @Override
    public byte[] getBytes(String columnLabel) throws SQLException {
        return this.getBytes(this.findColumnIndex(columnLabel));
    }

    @Override
    public Date getDate(String columnLabel) throws SQLException {
        return this.getDate(this.findColumnIndex(columnLabel));
    }

    @Override
    public Time getTime(String columnLabel) throws SQLException {
        return this.getTime(this.findColumnIndex(columnLabel));
    }

    @Override
    public Timestamp getTimestamp(String columnLabel) throws SQLException {
        return this.getTimestamp(this.findColumnIndex(columnLabel));
    }

    @Override
    public InputStream getAsciiStream(String columnLabel) throws SQLException {
        return this.getAsciiStream(this.findColumnIndex(columnLabel));
    }

    @Override
    public InputStream getUnicodeStream(String columnLabel) throws SQLException {
        return this.getUnicodeStream(this.findColumnIndex(columnLabel));
    }

    @Override
    public InputStream getBinaryStream(String columnLabel) throws SQLException {
        return this.getBinaryStream(this.findColumnIndex(columnLabel));
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        log.info("ResultSetImpl.getWarnings");
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {
        log.info("ResultSetImpl.clearWarnings");
    }

    @Override
    public String getCursorName() throws SQLException {
        log.info("ResultSetImpl.getCursorName");
        return null;
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        log.info("ResultSetImpl.getMetaData");
//        checkClosed();
        return new ResultSetMetaDataImpl(this, this.columns, this.metaType);
    }

    @Override
    public Object getObject(int columnIndex) throws SQLException {
        checkRowPos();
        String columnName = getColumnNameByIndex(columnIndex);
        synchronized (this){
            Map<String, Object> row = rows.get(this.index - 1);
            Object value = row.get(columnName);
            if (null == value) {
                this.wasNullFlag = true;
                return null;
            }
            return value;
        }
    }

    @Override
    public Object getObject(String columnLabel) throws SQLException {
        return this.getObject(this.findColumnIndex(columnLabel));
    }

    @Override
    public int findColumn(String columnLabel) throws SQLException {
        return this.findColumnIndex(columnLabel);
    }

    @Override
    public Reader getCharacterStream(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getCharacterStream");
        throw SqlError.createSQLFeatureNotSupportedException("getCharacterStream");
    }

    @Override
    public Reader getCharacterStream(String columnLabel) throws SQLException {
        return this.getCharacterStream(this.findColumnIndex(columnLabel));
    }

    @Override
    public BigDecimal getBigDecimal(int columnIndex) throws SQLException {
        String stringValue = this.getString(columnIndex);
        BigDecimal val = null;
        if (null != stringValue) {
            if (stringValue.length() == 0) {
                val = new BigDecimal("0");
            } else {
                try {
                    val = new BigDecimal(stringValue);
                } catch (NumberFormatException ex) {
                    throw SqlError.createSQLException(String.format("ResultSet.Bad_format_for_BigDecimal, columnIndex = %d, value = %s.", columnIndex, stringValue),
                            SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
                }
            }

        }
        return val;
    }

    @Override
    public BigDecimal getBigDecimal(String columnLabel) throws SQLException {
        return this.getBigDecimal(this.findColumnIndex(columnLabel));
    }

    @Override
    public boolean isBeforeFirst() throws SQLException {
        log.info("ResultSetImpl.isBeforeFirst");
        synchronized (this) {
            checkClosed();
            return this.isBeforeFirst;
        }
    }

    @Override
    public boolean isAfterLast() throws SQLException {
        log.info("ResultSetImpl.isAfterLast");
        synchronized (this) {
            checkClosed();
            return this.isAfterLast;
        }
    }

    @Override
    public boolean isFirst() throws SQLException {
        log.info("ResultSetImpl.isFirst");
        synchronized (this) {
            checkClosed();
            return this.index == 1;
        }
    }

    @Override
    public boolean isLast() throws SQLException {
        log.info("ResultSetImpl.isLast");
        synchronized (this) {
            checkClosed();
            if (this.rows == null || this.rows.isEmpty()) {
                return false;
            }
            return this.index == this.rows.size();
        }
    }

    /**
     * Moves the cursor to the front of
     * this <code>ResultSet</code> object, just before the
     * first row. This method has no effect if the result set contains no rows.
     *
     * @throws SQLException if a database access error
     *                      occurs; this method is called on a closed result set or the
     *                      result set type is <code>TYPE_FORWARD_ONLY</code>
     */
    @Override
    public void beforeFirst() throws SQLException {
        log.info("ResultSetImpl.beforeFirst");
        synchronized (this) {
            checkClosed();
            if (this.statement.getResultSetType() == ResultSet.TYPE_FORWARD_ONLY) {
                throw SqlError.createSQLException("Illegal operation for TYPE_FORWARD_ONLY ResultSet!",
                        SqlError.SQL_STATE_ILLEGAL_OPERATION);
            }
            if (null == this.rows) {
                return;
            }
            this.isBeforeFirst = true;
            setRowPositionValidity();
            this.index = 0;
        }
    }

    @Override
    public void afterLast() throws SQLException {
        log.info("ResultSetImpl.afterLast");
        synchronized (this) {
            checkClosed();
            if (this.statement.getResultSetType() == ResultSet.TYPE_FORWARD_ONLY) {
                throw SqlError.createSQLException("Illegal operation for TYPE_FORWARD_ONLY ResultSet!",
                        SqlError.SQL_STATE_ILLEGAL_OPERATION);
            }
            if (null == this.rows) {
                return;
            }
            this.isAfterLast = true;
            setRowPositionValidity();
            this.index = this.rows.size() + 1;
        }
    }

    @Override
    public boolean first() throws SQLException {
        log.info("ResultSetImpl.first");
        synchronized (this) {
            checkClosed();
            if (this.statement.getResultSetType() == ResultSet.TYPE_FORWARD_ONLY) {
                throw SqlError.createSQLException("Illegal operation for TYPE_FORWARD_ONLY ResultSet!",
                        SqlError.SQL_STATE_ILLEGAL_OPERATION);
            }
            setRowPositionValidity();
            this.index = 1;
            return this.onValidRow;
        }
    }

    @Override
    public boolean last() throws SQLException {
        log.info("ResultSetImpl.last");
        synchronized (this) {
            checkClosed();
            if (this.statement.getResultSetType() == ResultSet.TYPE_FORWARD_ONLY) {
                throw SqlError.createSQLException("Illegal operation for TYPE_FORWARD_ONLY ResultSet!",
                        SqlError.SQL_STATE_ILLEGAL_OPERATION);
            }
            setRowPositionValidity();
            this.index = this.rows.size();
            return this.onValidRow;
        }
    }

    @Override
    public int getRow() throws SQLException {
        log.info("ResultSetImpl.getRow");
        return this.index;
    }

    @Override
    public boolean absolute(int row) throws SQLException {
        log.info("ResultSetImpl.absolute, row:{}", row);
        synchronized (this) {
            checkClosed();
            if (this.statement.getResultSetType() == ResultSet.TYPE_FORWARD_ONLY) {
                throw SqlError.createSQLException("Illegal operation for TYPE_FORWARD_ONLY ResultSet!",
                        SqlError.SQL_STATE_ILLEGAL_OPERATION);
            }
            if (this.rows.size() == 0) {
                return false;
            }
            if (row > 0) {
                if (row <= this.rows.size()) {
                    setRowPositionValidity();
                    this.index = row;
                } else {
                    this.afterLast();
                }
            } else if (row < 0) {
                int cursor = this.rows.size() + row + 1;
                if (cursor > 0) {
                    setRowPositionValidity();
                    this.index = cursor;
                } else {
                    this.beforeFirst();
                }
            } else {
                this.beforeFirst();
            }
            return this.onValidRow;
        }
    }

    @Override
    public boolean relative(int rows) throws SQLException {
        log.info("ResultSetImpl.relative, rows:{}", rows);
        return false;
    }

    @Override
    public boolean previous() throws SQLException {
        log.info("ResultSetImpl.previous");
        return false;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        log.info("ResultSetImpl.setFetchDirection, direction:{}", direction);
        synchronized (this) {
            ResultSetImpl.checkSupportFetchDirection(direction);
            this.fetchDirection = direction;
        }
    }

    public static void checkSupportFetchDirection(int direction) throws SQLException {
        if ((direction != FETCH_FORWARD) && (direction != FETCH_REVERSE) && (direction != FETCH_UNKNOWN)) {
            throw SqlError.createSQLException("ResultSet.Illegal_value_for_fetch_direction:" + direction, SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        if (direction != FETCH_FORWARD) {
            throw SqlError.createSQLFeatureNotSupportedException("not support fetchDirection for:" + direction, "setFetchDirection");
        }
    }

    @Override
    public int getFetchDirection() throws SQLException {
        return this.fetchDirection;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        this.fetchSize = rows;
    }

    @Override
    public int getFetchSize() throws SQLException {
        return this.fetchSize;
    }

    @Override
    public int getType() throws SQLException {
        return this.statement.getResultSetType();
    }

    @Override
    public int getConcurrency() throws SQLException {
        return this.statement.getResultSetConcurrency();
    }

    @Override
    public boolean rowUpdated() throws SQLException {
        return false;
    }

    @Override
    public boolean rowInserted() throws SQLException {
        return false;
    }

    @Override
    public boolean rowDeleted() throws SQLException {
        return false;
    }

    @Override
    public void updateNull(int columnIndex) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNull");
    }

    @Override
    public void updateBoolean(int columnIndex, boolean x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBoolean");
    }

    @Override
    public void updateByte(int columnIndex, byte x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateByte");
    }

    @Override
    public void updateShort(int columnIndex, short x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateShort");
    }

    @Override
    public void updateInt(int columnIndex, int x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateInt");
    }

    @Override
    public void updateLong(int columnIndex, long x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateLong");
    }

    @Override
    public void updateFloat(int columnIndex, float x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateFloat");
    }

    @Override
    public void updateDouble(int columnIndex, double x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateDouble");
    }

    @Override
    public void updateBigDecimal(int columnIndex, BigDecimal x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBigDecimal");
    }

    @Override
    public void updateString(int columnIndex, String x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateString");
    }

    @Override
    public void updateBytes(int columnIndex, byte[] x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBytes");
    }

    @Override
    public void updateDate(int columnIndex, Date x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateDate");
    }

    @Override
    public void updateTime(int columnIndex, Time x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateTime");
    }

    @Override
    public void updateTimestamp(int columnIndex, Timestamp x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateTimestamp");
    }

    @Override
    public void updateAsciiStream(int columnIndex, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(int columnIndex, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(int columnIndex, Reader x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateObject(int columnIndex, Object x, int scaleOrLength) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateObject");
    }

    @Override
    public void updateObject(int columnIndex, Object x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateObject");
    }

    @Override
    public void updateNull(String columnLabel) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNull");
    }

    @Override
    public void updateBoolean(String columnLabel, boolean x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBoolean");
    }

    @Override
    public void updateByte(String columnLabel, byte x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateByte");
    }

    @Override
    public void updateShort(String columnLabel, short x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateShort");
    }

    @Override
    public void updateInt(String columnLabel, int x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateInt");
    }

    @Override
    public void updateLong(String columnLabel, long x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateLong");
    }

    @Override
    public void updateFloat(String columnLabel, float x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateFloat");
    }

    @Override
    public void updateDouble(String columnLabel, double x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateDouble");
    }

    @Override
    public void updateBigDecimal(String columnLabel, BigDecimal x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBigDecimal");
    }

    @Override
    public void updateString(String columnLabel, String x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateString");
    }

    @Override
    public void updateBytes(String columnLabel, byte[] x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBytes");
    }

    @Override
    public void updateDate(String columnLabel, Date x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateDate");
    }

    @Override
    public void updateTime(String columnLabel, Time x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateTime");
    }

    @Override
    public void updateTimestamp(String columnLabel, Timestamp x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateTimestamp");
    }

    @Override
    public void updateAsciiStream(String columnLabel, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(String columnLabel, InputStream x, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(String columnLabel, Reader reader, int length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateObject(String columnLabel, Object x, int scaleOrLength) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateObject");
    }

    @Override
    public void updateObject(String columnLabel, Object x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateObject");
    }

    @Override
    public void insertRow() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("insertRow");
    }

    @Override
    public void updateRow() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateRow");
    }

    @Override
    public void deleteRow() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("deleteRow");
    }

    @Override
    public void refreshRow() throws SQLException {

    }

    @Override
    public void cancelRowUpdates() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("cancelRowUpdates");
    }

    @Override
    public void moveToInsertRow() throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("moveToInsertRow");
    }

    @Override
    public void moveToCurrentRow() throws SQLException {
    }

    @Override
    public Statement getStatement() throws SQLException {
        return this.statement;
    }

    @Override
    public Object getObject(int columnIndex, Map<String, Class<?>> map) throws SQLException {
        log.info("ResultSetImpl.getObject, columnIndex:{}, map:{}", columnIndex, map);
        throw SqlError.createSQLFeatureNotSupportedException("updateRef");
    }

    @Override
    public Ref getRef(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getRef, columnIndex:{}", columnIndex);
        throw SqlError.createSQLFeatureNotSupportedException("getRef");
    }

    @Override
    public Blob getBlob(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getBlob");
        throw SqlError.createSQLFeatureNotSupportedException("getBlob");
    }

    @Override
    public Clob getClob(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getBlob");
        throw SqlError.createSQLFeatureNotSupportedException("getClob");
    }

    @Override
    public Array getArray(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getArray, columnIndex:{}", columnIndex);
        throw SqlError.createSQLFeatureNotSupportedException("getArray");
    }

    @Override
    public Object getObject(String columnLabel, Map<String, Class<?>> map) throws SQLException {
        log.info("ResultSetImpl.getObject, columnLabel:{}, map:{}", columnLabel, map);
        throw SqlError.createSQLFeatureNotSupportedException("getObject");
    }

    @Override
    public Ref getRef(String columnLabel) throws SQLException {
        log.info("ResultSetImpl.getRef, columnLabel:{}", columnLabel);
        throw SqlError.createSQLFeatureNotSupportedException("getRef");
    }

    @Override
    public Blob getBlob(String columnLabel) throws SQLException {
        log.info("ResultSetImpl.getBlob, columnLabel:{}", columnLabel);
        throw SqlError.createSQLFeatureNotSupportedException("getBlob");
    }

    @Override
    public Clob getClob(String columnLabel) throws SQLException {
        log.info("ResultSetImpl.getClob, columnLabel:{}", columnLabel);
        throw SqlError.createSQLFeatureNotSupportedException("getClob");
    }

    @Override
    public Array getArray(String columnLabel) throws SQLException {
        log.info("ResultSetImpl.getArray, columnLabel:{}", columnLabel);
        throw SqlError.createSQLFeatureNotSupportedException("getArray");
    }

    @Override
    public Date getDate(int columnIndex, Calendar cal) throws SQLException {
        String dateValue = this.getString(columnIndex);
        java.util.Date utilDate = formatUtilDate(dateValue, cal);
        return utilDate == null ? null : new Date(utilDate.getTime());
    }

    @Override
    public Date getDate(String columnLabel, Calendar cal) throws SQLException {
        return this.getDate(this.findColumnIndex(columnLabel), cal);
    }

    @Override
    public Time getTime(int columnIndex, Calendar cal) throws SQLException {
        String dateValue = this.getString(columnIndex);
        java.util.Date utilDate = formatUtilDate(dateValue, cal);
        return utilDate == null ? null : new Time(utilDate.getTime());
    }

    @Override
    public Time getTime(String columnLabel, Calendar cal) throws SQLException {
        return this.getTime(this.findColumnIndex(columnLabel), cal);
    }

    @Override
    public Timestamp getTimestamp(int columnIndex, Calendar cal) throws SQLException {
        String dateValue = this.getString(columnIndex);
        java.util.Date utilDate = formatUtilDate(dateValue, cal);
        return utilDate == null ? null : new Timestamp(utilDate.getTime());
    }

    @Override
    public Timestamp getTimestamp(String columnLabel, Calendar cal) throws SQLException {
        return this.getTimestamp(this.findColumnIndex(columnLabel), cal);
    }

    @Override
    public URL getURL(int columnIndex) throws SQLException {
        String val = getString(columnIndex);
        if (val == null) {
            return null;
        }
        try {
            return new URL(val);
        } catch (MalformedURLException mfe) {
            throw SqlError.createSQLException("ResultSet.Bad_format_for_URL, value:" + val, SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public URL getURL(String columnLabel) throws SQLException {
        return this.getURL(this.findColumnIndex(columnLabel));
    }

    @Override
    public void updateRef(int columnIndex, Ref x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateRef");
    }

    @Override
    public void updateRef(String columnLabel, Ref x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateRef");
    }

    @Override
    public void updateBlob(int columnIndex, Blob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateBlob(String columnLabel, Blob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateClob(int columnIndex, Clob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateClob(String columnLabel, Clob x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateArray(int columnIndex, Array x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateArray");
    }

    @Override
    public void updateArray(String columnLabel, Array x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateArray");
    }

    @Override
    public RowId getRowId(int columnIndex) throws SQLException {
        log.info("ResultSetImpl.getRowId, columnIndex:{}", columnIndex);
        throw SqlError.createSQLFeatureNotSupportedException("getRowId");
    }

    @Override
    public RowId getRowId(String columnLabel) throws SQLException {
        return this.getRowId(this.findColumnIndex(columnLabel));
    }

    @Override
    public void updateRowId(int columnIndex, RowId x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateRowId");
    }

    @Override
    public void updateRowId(String columnLabel, RowId x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateRowId");
    }

    @Override
    public int getHoldability() throws SQLException {
        return this.statement.getResultSetHoldability();
    }

    @Override
    public boolean isClosed() throws SQLException {
        return this.isClosed;
    }

    @Override
    public void updateNString(int columnIndex, String nString) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNString");
    }

    @Override
    public void updateNString(String columnLabel, String nString) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNString");
    }

    @Override
    public void updateNClob(int columnIndex, NClob nClob) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public void updateNClob(String columnLabel, NClob nClob) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public NClob getNClob(int columnIndex) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNClob");
    }

    @Override
    public NClob getNClob(String columnLabel) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNClob");
    }

    @Override
    public SQLXML getSQLXML(int columnIndex) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getSQLXML");
    }

    @Override
    public SQLXML getSQLXML(String columnLabel) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getSQLXML");
    }

    @Override
    public void updateSQLXML(int columnIndex, SQLXML xmlObject) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateSQLXML");
    }

    @Override
    public void updateSQLXML(String columnLabel, SQLXML xmlObject) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateSQLXML");
    }

    @Override
    public String getNString(int columnIndex) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNString");
    }

    @Override
    public String getNString(String columnLabel) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNString");
    }

    @Override
    public Reader getNCharacterStream(int columnIndex) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNCharacterStream");
    }

    @Override
    public Reader getNCharacterStream(String columnLabel) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("getNCharacterStream");
    }

    @Override
    public void updateNCharacterStream(int columnIndex, Reader x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNCharacterStream");
    }

    @Override
    public void updateNCharacterStream(String columnLabel, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNCharacterStream");
    }

    @Override
    public void updateAsciiStream(int columnIndex, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(int columnIndex, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(int columnIndex, Reader x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateAsciiStream(String columnLabel, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(String columnLabel, InputStream x, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(String columnLabel, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateBlob(int columnIndex, InputStream inputStream, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateBlob(String columnLabel, InputStream inputStream, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateClob(int columnIndex, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateClob(String columnLabel, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateNClob(int columnIndex, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public void updateNClob(String columnLabel, Reader reader, long length) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public void updateNCharacterStream(int columnIndex, Reader x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNCharacterStream");
    }

    @Override
    public void updateNCharacterStream(String columnLabel, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNCharacterStream");
    }

    @Override
    public void updateAsciiStream(int columnIndex, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(int columnIndex, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(int columnIndex, Reader x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateAsciiStream(String columnLabel, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateAsciiStream");
    }

    @Override
    public void updateBinaryStream(String columnLabel, InputStream x) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBinaryStream");
    }

    @Override
    public void updateCharacterStream(String columnLabel, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateCharacterStream");
    }

    @Override
    public void updateBlob(int columnIndex, InputStream inputStream) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateBlob(String columnLabel, InputStream inputStream) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateBlob");
    }

    @Override
    public void updateClob(int columnIndex, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateClob(String columnLabel, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateClob");
    }

    @Override
    public void updateNClob(int columnIndex, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public void updateNClob(String columnLabel, Reader reader) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("updateNClob");
    }

    @Override
    public <T> T getObject(int columnIndex, Class<T> type) throws SQLException {
        log.info("ResultSetImpl.getObject, columnIndex:{}, type:{}", columnIndex, type);
        if (type == null) {
            throw SqlError.createSQLException("Type parameter can not be null", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }

        if (type.equals(String.class)) {
            return (T) getString(columnIndex);
        } else if (type.equals(BigDecimal.class)) {
            return (T) getBigDecimal(columnIndex);
        } else if (type.equals(Boolean.class) || type.equals(Boolean.TYPE)) {
            return (T) Boolean.valueOf(getBoolean(columnIndex));
        } else if (type.equals(Integer.class) || type.equals(Integer.TYPE)) {
            return (T) Integer.valueOf(getInt(columnIndex));
        } else if (type.equals(Long.class) || type.equals(Long.TYPE)) {
            return (T) Long.valueOf(getLong(columnIndex));
        } else if (type.equals(Float.class) || type.equals(Float.TYPE)) {
            return (T) Float.valueOf(getFloat(columnIndex));
        } else if (type.equals(Double.class) || type.equals(Double.TYPE)) {
            return (T) Double.valueOf(getDouble(columnIndex));
        } else if (type.equals(byte[].class)) {
            return (T) getBytes(columnIndex);
        } else if (type.equals(java.sql.Date.class)) {
            return (T) getDate(columnIndex);
        } else if (type.equals(Time.class)) {
            return (T) getTime(columnIndex);
        } else if (type.equals(Timestamp.class)) {
            return (T) getTimestamp(columnIndex);
        } else if (type.equals(Array.class)) {
            return (T) getArray(columnIndex);
        } else if (type.equals(URL.class)) {
            return (T) getURL(columnIndex);
        } else {
            throw SqlError.createSQLException("Conversion not supported for type:" + type.getName(), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    @Override
    public <T> T getObject(String columnLabel, Class<T> type) throws SQLException {
        log.info("ResultSetImpl.getObject, columnLabel:{}, type:{}", columnLabel, type);
        return this.getObject(findColumnIndex(columnLabel), type);
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
