package com.eniot.data.query.impl;

import com.eniot.data.query.EniotResultSet;
import com.eniot.data.query.exception.SqlError;
import com.eniot.data.query.util.JdbcUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

/**
 * @author jinghui.zhao
 * @date 2020/1/20
 */
public class ResultSetMetaDataImpl implements ResultSetMetaData {

    private static final Logger log = LoggerFactory.getLogger(ResultSetMetaDataImpl.class);
    /**
     * column name
     */
    private List<String> columns;

    /**
     * column metadata
     */
    private List<Integer> metaType;

    private EniotResultSet resultSet;

    public ResultSetMetaDataImpl(EniotResultSet resultSet, List<String> columns, List<Integer> metaType) {
        this.resultSet = resultSet;
        this.columns = columns;
        this.metaType = metaType;
    }

    @Override
    public int getColumnCount() throws SQLException {
        return (this.columns == null) ? 0 : this.columns.size();
    }

    /**
     * Indicates whether the designated column is automatically numbered.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return <code>true</code> if so; <code>false</code> otherwise
     * @throws SQLException if a database access error occurs
     */
    @Override
    public boolean isAutoIncrement(int column) throws SQLException {
        return true;
    }

    private void checkColumnBounds(int columnIndex) throws SQLException {
        if (columnIndex < 1) {
            throw SqlError.createSQLException(String.format("Column Index out of range, %d < 1.", columnIndex), SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        } else if (columnIndex > this.columns.size()) {
            throw SqlError.createSQLException(String.format("Column Index out of range, %d > %d.", columnIndex, columns.size()),
                    SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    /**
     * Indicates whether a column's case matters.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return <code>true</code> if so; <code>false</code> otherwise
     * @throws SQLException if a database access error occurs
     */
    @Override
    public boolean isCaseSensitive(int column) throws SQLException {
        int columnType = this.getColumnType(column);
        switch (columnType) {
            case Types.BIT:
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
            case Types.DATE:
            case Types.TIME:
            case Types.TIMESTAMP:
                return false;

            case Types.CHAR:
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
                return true;
            default:
                return true;
        }
    }

    @Override
    public boolean isSearchable(int column) throws SQLException {
        return true;
    }

    /**
     * Indicates whether the designated column is a cash value.
     */
    @Override
    public boolean isCurrency(int column) throws SQLException {
        return false;
    }

    @Override
    public int isNullable(int column) throws SQLException {
        return java.sql.ResultSetMetaData.columnNoNulls;
    }

    /**
     * Indicates whether values in the designated column are signed numbers.
     */
    @Override
    public boolean isSigned(int column) throws SQLException {
        int columnType = this.getColumnType(column);
        switch (columnType) {
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
            case Types.NUMERIC:
            case Types.DECIMAL:
                return true;

            case Types.DATE:
            case Types.TIME:
            case Types.TIMESTAMP:
                return false;

            default:
                return false;
        }
    }

    @Override
    public int getColumnDisplaySize(int column) throws SQLException {
        return Integer.MAX_VALUE;
    }

    /**
     * Gets the designated column's suggested title for use in printouts and
     * displays. The suggested title is usually specified by the SQL <code>AS</code>
     * clause.  If a SQL <code>AS</code> is not specified, the value returned from
     * <code>getColumnLabel</code> will be the same as the value returned by the
     * <code>getColumnName</code> method.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return the suggested column title
     * @throws SQLException if a database access error occurs
     */
    @Override
    public String getColumnLabel(int column) throws SQLException {
        checkColumnBounds(column);
        return this.columns.get(column - 1);
    }

    @Override
    public String getColumnName(int column) throws SQLException {
        return this.getColumnLabel(column);
    }

    @Override
    public String getSchemaName(int column) throws SQLException {
        return "";
    }

    @Override
    public int getPrecision(int column) throws SQLException {
        int type = getColumnType(column);
        return JdbcUtils.getPrecisionFromTypes(type);
    }

    @Override
    public int getScale(int column) throws SQLException {
        int type = getColumnType(column);
        return JdbcUtils.getScaleFromTypes(type);
    }

    @Override
    public String getTableName(int column) throws SQLException {
        return "";
    }

    @Override
    public String getCatalogName(int column) throws SQLException {
        return this.resultSet.getCatalog();
    }

    @Override
    public int getColumnType(int column) throws SQLException {
        checkColumnBounds(column);
        return this.metaType.get(column - 1);
    }

    @Override
    public String getColumnTypeName(int column) throws SQLException {
        int columnType = this.getColumnType(column);
        return JdbcUtils.getTypeNameFromTypes(columnType);
    }

    @Override
    public boolean isReadOnly(int column) throws SQLException {
        return true;
    }

    @Override
    public boolean isWritable(int column) throws SQLException {
        return false;
    }

    @Override
    public boolean isDefinitelyWritable(int column) throws SQLException {
        return false;
    }

    @Override
    public String getColumnClassName(int column) throws SQLException {
        int type = this.getColumnType(column);
        return JdbcUtils.getClassNameFromTypes(type);
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
