package com.eniot.data.query.impl;

import com.eniot.data.query.EniotConnect;
import com.eniot.data.query.entity.QueryResponse;
import com.eniot.data.query.exception.SqlError;
import com.eniot.data.query.util.JdbcUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.ref.Reference;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import static java.sql.ResultSet.FETCH_FORWARD;

/**
 * @author jinghui.zhao
 * @date 2020/1/15
 */
public class StatementImpl implements java.sql.Statement {
    private static final Logger log = LoggerFactory.getLogger(StatementImpl.class);

    protected Object cancelTimeoutMutex = new Object();

    /**
     * Used to generate IDs when profiling.
     */
    static int statementCounter = 1;

    public final static byte USES_VARIABLES_FALSE = 0;

    public final static byte USES_VARIABLES_TRUE = 1;

    public final static byte USES_VARIABLES_UNKNOWN = -1;

    protected boolean wasCancelled = false;
    protected boolean wasCancelledByTimeout = false;

    /**
     * Holds batched commands
     */
    protected List<Object> batchedArgs;

    /** The character converter to use (if available) */
//    protected SingleByteCharsetConverter charConverter = null;

    /**
     * The character encoding to use (if available)
     */
    protected String charEncoding = null;

    /**
     * The connection that created us
     */
    protected volatile EniotConnect connection = null;

    /**
     * The physical connection used to effectively execute the statement
     */
    protected Reference<Connection> physicalConnection = null;

    /**
     * Should we process escape codes?
     */
    protected boolean doEscapeProcessing = true;

    /**
     * The number of rows to fetch at a time (currently ignored)
     */
    private int fetchSize = 640000;

    /**
     * Has this statement been closed?
     */
    protected boolean isClosed = false;

    /**
     * The auto_increment value for the last insert
     */
    protected long lastInsertId = -1;

    /**
     * The max field size for this statement
     */
    private int maxFieldSize = Integer.MAX_VALUE;

    /**
     * The maximum number of rows to return for this statement (-1 means _all_
     * rows)
     */
    private int maxRows = 640000;

    /** Set of currently-open ResultSets */
//    protected Set<ResultSetInternalMethods> openResults = new HashSet<ResultSetInternalMethods>();

    /**
     * Are we in pedantic mode?
     */
    protected boolean pedantic = false;

    /**
     * Should we profile?
     */
    protected boolean profileSQL = false;

    /** The current results */
//    protected ResultSetInternalMethods results = null;
//
//    protected ResultSetInternalMethods generatedKeysResults = null;

    /**
     * The concurrency for this result set (updatable or not)
     */
    private int resultSetConcurrency = 0;

    /**
     * The type of this result set (scroll sensitive or in-sensitive)
     */
    private int resultSetType = 0;

    /**
     * Used to identify this statement when profiling.
     */
    private int statementId;

    /**
     * The timeout for a query
     */
    protected int timeoutInMillis = 0;

    /**
     * The update count for this statement
     */
    protected long updateCount = -1;

    /**
     * Should we use the usage advisor?
     */
    protected boolean useUsageAdvisor = false;

    /**
     * The warnings chain.
     */
    private SQLWarning warningChain = null;

    /**
     * Has clearWarnings() been called?
     */
    protected boolean clearWarningsCalled = false;

    /**
     * Should this statement hold results open over .close() irregardless of
     * connection's setting?
     */
    protected boolean holdResultsOpenOverClose = false;

//    protected ArrayList<ResultSetRow> batchedGeneratedKeys = null;

    protected boolean retrieveGeneratedKeys = false;

    protected boolean continueBatchOnError = false;

    protected boolean useLegacyDatetimeCode;

    protected boolean sendFractionalSeconds;

    /**
     * Whether or not the last query was of the form ON DUPLICATE KEY UPDATE
     */
    protected boolean lastQueryIsOnDupKeyUpdate = false;

    /**
     * Are we currently executing a statement?
     */
    protected final AtomicBoolean statementExecuting = new AtomicBoolean(false);

    /**
     * Are we currently closing results implicitly (internally)?
     */
    private boolean isImplicitlyClosingResults = false;

    private volatile ResultSet results = null;

    private int fetchDirection = FETCH_FORWARD;


    public StatementImpl(EniotConnect c, int resultSetType, int resultSetConcurrency) throws SQLException {
        if ((c == null) || c.isClosed()) {
            throw SqlError.createSQLException("No operations allowed after connection closed.", SqlError.SQL_STATE_CONNECTION_NOT_OPEN);
        }
        this.connection = c;
        this.statementId = StatementImpl.statementCounter++;
        this.resultSetType = resultSetType;
        this.resultSetConcurrency = resultSetConcurrency;
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        JdbcUtils.checkSql(sql);

        return connection.execQuerySql(this, this.connection.nativeSQL(sql));

    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        log.info("StatementImpl.executeUpdate, sql:{}", sql);
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public void close() throws SQLException {
        log.info("StatementImpl.close");
        synchronized (this) {
            this.isClosed = true;
            this.results = null;
            this.connection = null;
            this.warningChain = null;
        }
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        return this.maxFieldSize;
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        this.maxFieldSize = max;

    }

    @Override
    public int getMaxRows() throws SQLException {
        return this.maxRows;
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        this.maxRows = max;
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        log.info("StatementImpl.setEscapeProcessing");
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        return 300;
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        log.info("StatementImpl.setQueryTimeout");
    }

    @Override
    public void cancel() throws SQLException {
        log.info("StatementImpl.cancel");
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        log.info("StatementImpl.getWarnings");
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {
        log.info("StatementImpl.clearWarnings");
    }

    @Override
    public void setCursorName(String name) throws SQLException {
        log.info("StatementImpl.setCursorName");
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        ResultSet newResults = this.executeQuery(sql);
        synchronized (this) {
            this.results = newResults;
            return this.results != null;
        }
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
        log.info("StatementImpl.getResultSet");
        synchronized (this) {
            ResultSet res = this.results;
            this.results = null;
            return res;
        }
    }

    @Override
    public int getUpdateCount() throws SQLException {
        log.info("StatementImpl.getUpdateCount");
        return -1;
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        log.info("StatementImpl.getMoreResults");
        return false;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        log.info("StatementImpl.setFetchDirection, direction:{}.", direction);
        synchronized (this) {
            ResultSetImpl.checkSupportFetchDirection(direction);
            this.fetchDirection = direction;
        }
    }

    @Override
    public int getFetchDirection() throws SQLException {
        return this.fetchDirection;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        this.fetchSize = rows;
    }

    @Override
    public int getFetchSize() throws SQLException {
        return this.fetchSize;
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        return this.resultSetConcurrency;
    }

    @Override
    public int getResultSetType() throws SQLException {
        return this.resultSetType;
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        log.info("StatementImpl.addBatch");
        throw SqlError.createSQLFeatureNotSupportedException("addBatch");
    }

    @Override
    public void clearBatch() throws SQLException {
        log.info("StatementImpl.clearBatch");
    }

    @Override
    public int[] executeBatch() throws SQLException {
        log.info("StatementImpl.executeBatch");
        throw SqlError.createSQLFeatureNotSupportedException("executeBatch");
    }

    @Override
    public Connection getConnection() throws SQLException {
        log.info("StatementImpl.getConnection");
        return this.connection;
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        log.info("StatementImpl.getMoreResults, current:{}", current);
        return false;
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        log.info("StatementImpl.getGeneratedKeys");
        QueryResponse queryResponse = new QueryResponse();
        List<String> columns = new ArrayList<>(4);
        columns.add("GENERATED_KEY");
        List<String> metadata = new ArrayList<>(4);
        metadata.add("VARCHAR");
        List<Map<String, Object>> rows = new ArrayList<>(0);
        queryResponse.setColumns(columns);
        queryResponse.setMetadata(metadata);
        queryResponse.setRows(rows);
        return new ResultSetImpl(this.connection, this.connection.getPrivateStatement(), queryResponse);
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("executeUpdate");
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        log.info("StatementImpl.execute, sql:{}, autoGeneratedKeys:{}.", sql, autoGeneratedKeys);
        return false;
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        log.info("StatementImpl.execute, sql:{}, columnIndexes:{}.", sql, columnIndexes);
        return false;
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        log.info("StatementImpl.execute, sql:{}, columnNames:{}.", sql, columnNames);
        return false;
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        return java.sql.ResultSet.HOLD_CURSORS_OVER_COMMIT;
    }

    @Override
    public boolean isClosed() throws SQLException {
        return this.isClosed;
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {

    }

    @Override
    public boolean isPoolable() throws SQLException {
        return false;
    }

    @Override
    public void closeOnCompletion() throws SQLException {

    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        return false;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw SqlError.createSQLFeatureNotSupportedException("unwrap");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return false;
    }
}
