package com.eniot.data.query.util;

import com.eniot.data.query.entity.DisallowedSqlActions;
import com.eniot.data.query.exception.SqlError;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.sql.Types;
import java.util.Locale;

/**
 * @author jinghui.zhao
 * @date 2020/1/9
 */
public class JdbcUtils {

    private static final Logger log = LoggerFactory.getLogger(JdbcUtils.class);

    /**
     * Method checkSql.
     *
     * @param sql the SQL to check
     * @throws SQLException if query is illegal.
     */
    public static void checkSql(String sql) throws SQLException {
        if (StringUtils.isBlank(sql)) {
            log.info("checkSql, sql is null");
            throw SqlError.createSQLException("sql is null!", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
        for (DisallowedSqlActions action : DisallowedSqlActions.values()) {
            if (sql.trim().toLowerCase(Locale.ENGLISH).startsWith(action.toString().toLowerCase(Locale.ENGLISH))) {
                log.info("checkSql, sql is illegal! Only for select sql:{}.", sql);
                throw SqlError.createSQLException("sql is illegal! Only for select sql.", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
            }
        }
        String illegalCharacter = ";";
        if (sql.contains(illegalCharacter)) {
            log.info("checkSql, `;` is illegal char in sql:{}.", sql);
            throw SqlError.createSQLException("`;` is illegal char in sql.", SqlError.SQL_STATE_ILLEGAL_ARGUMENT);
        }
    }

    /**
     * Adds '+' to decimal numbers that are positive (MySQL doesn't understand
     * them otherwise
     *
     * @param dString The value as a string
     * @return String the string with a '+' added (if needed)
     */
    public static String fixDecimalExponent(String dString) {
        int ePos = dString.indexOf('E');

        if (ePos == -1) {
            ePos = dString.indexOf('e');
        }

        if (ePos != -1) {
            if (dString.length() > (ePos + 1)) {
                char maybeMinusChar = dString.charAt(ePos + 1);

                char negativeChar = '-';
                char positiveChar = '+';
                if (maybeMinusChar != negativeChar && maybeMinusChar != positiveChar) {
                    StringBuilder strBuilder = new StringBuilder(dString.length() + 1);
                    strBuilder.append(dString.substring(0, ePos + 1));
                    strBuilder.append('+');
                    strBuilder.append(dString.substring(ePos + 1, dString.length()));
                    dString = strBuilder.toString();
                }
            }
        }

        return dString;
    }

    public static int getPrecisionFromTypes(int type) {
        switch (type) {
            case Types.BIT:
            case Types.BOOLEAN:
                return 1;
            case Types.TINYINT:
                return 4;
            case Types.SMALLINT:
                return 6;
            case Types.INTEGER:
                return 11;
            case Types.BIGINT:
                return 20;
            case Types.FLOAT:
            case Types.REAL:
                return 12;
            case Types.DOUBLE:
                return 22;
            case Types.NUMERIC:
            case Types.DECIMAL:
                return 38;
            case Types.DATE:
                return 10;
            case Types.TIME:
                return 10;
            case Types.TIMESTAMP:
                return 19;
            case Types.CHAR:
                return 1;
            case Types.VARCHAR:
                return 1000;
            case Types.LONGVARCHAR:
                return 715827882;
            default:
                return 1000;
        }
    }

    public static int getScaleFromTypes(int type) {
        switch (type) {
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
                return 31;
            case Types.BIT:
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.NUMERIC:
            case Types.DECIMAL:
                return 0;
            default:
                return 0;
        }
    }

    public static String getTypeNameFromTypes(int type) {
        switch (type) {
            case Types.BIT:
                return "BIT";
            case Types.TINYINT:
                return "TINYINT";
            case Types.SMALLINT:
                return "SMALLINT";
            case Types.INTEGER:
                return "INTEGER";
            case Types.BIGINT:
                return "BIGINT";
            case Types.FLOAT:
                return "FLOAT";
            case Types.REAL:
                return "REAL";
            case Types.DOUBLE:
                return "DOUBLE";
            case Types.NUMERIC:
                return "NUMERIC";
            case Types.DECIMAL:
                return "DECIMAL";
            case Types.DATE:
                return "DATE";
            case Types.TIME:
                return "TIME";
            case Types.TIMESTAMP:
                return "TIMESTAMP";
            case Types.VARBINARY:
                return "VARBINARY";
            default:
                return "UNKNOWN";
        }
    }

    public static String getClassNameFromTypes(int type) {
        switch (type) {
            case Types.BIT:
            case Types.BOOLEAN:
                return "java.lang.Boolean";
            case Types.TINYINT:
            case Types.SMALLINT:
                return "java.lang.Integer";
            case Types.INTEGER:
                return "java.lang.Integer";
            case Types.BIGINT:
                return "java.lang.Long";
            case Types.DECIMAL:
            case Types.NUMERIC:
                return "java.math.BigDecimal";
            case Types.REAL:
                return "java.lang.Float";
            case Types.FLOAT:
            case Types.DOUBLE:
                return "java.lang.Double";
            case Types.CHAR:
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
                return "java.lang.String";
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY:
                return "java.lang.String";
            case Types.DATE:
                return "java.sql.Date";
            case Types.TIME:
                return "java.sql.Time";
            case Types.TIMESTAMP:
                return "java.sql.Timestamp";
            default:
                return "java.lang.Object";
        }
    }

    public static boolean isNumberType(int type) {
        switch (type) {
            case Types.BIT:
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
            case Types.NUMERIC:
            case Types.DECIMAL:
                return true;
            default:
                return false;
        }
    }
}
