/**
 * Project: sparkDemo
 * <p/>
 * Copyright http://www.envisioncn.com/
 * All rights reserved.
 *
 * @author xiaomin.zhou
 */
package com.eniot.data.query.util;

import com.google.gson.internal.bind.util.ISO8601Utils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

/**
 * Time Utility
 *
 * @author xiaomin.zhou
 */
public class TimeUtil {
    private static final Log logger = LogFactory.getLog(TimeUtil.class);

    public static final String DEFAULT_TIMEZONE = "GMT+08:00";

    public static final String STANDARD_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static final String MS_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";

    /**
     * Dummy Constructor
     */
    private TimeUtil() {

    }

    public static String transform(Date date) {
        if (date == null) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(STANDARD_FORMAT);
        return sdf.format(date);
    }

    public static String transform(Date date, TimeZone tz) {
        if (date == null) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(STANDARD_FORMAT);
        sdf.setTimeZone(tz);
        return sdf.format(date);
    }

    public static String transform(Date date, TimeZone tz, String format) {
        if (date == null) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        sdf.setTimeZone(tz);
        return sdf.format(date);
    }

    public static long getMorningMillis(TimeZone tz) {
        Calendar cal = Calendar.getInstance(tz);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }


    public static TimeZone getDefaultTimezone() {
        return TimeZone.getTimeZone(DEFAULT_TIMEZONE);
    }

    public static Long getUnixTimestampFromStandardFormat(String localTime, String timeZone) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(STANDARD_FORMAT);
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timeZone));
        Date date = null;
        try {
            date = simpleDateFormat.parse(localTime);
        } catch (ParseException e) {
            logger.error("TimeUtil.getUnixTimestampFromStandardFormat ParseException",e);
            throw new IllegalArgumentException(String.format("Invalid Arguments: %s, date format should be %s", localTime, STANDARD_FORMAT));
        }
        return date.getTime();
    }

    /**
     * 将ISO8601时间字符串转换为unix时间
     *
     * @param timeInISO8601 ISO8601表示的时间字符串
     * @return unix时间
     */
    public static long parseISO8601ToUnixTime(String timeInISO8601) {
        try {
            Date date = ISO8601Utils.parse(timeInISO8601, new ParsePosition(0));
            return date.getTime();
        }catch (ParseException e){
            throw new IllegalArgumentException(String.format("Invalid Arguments: %s date format should be yyyy-MM-ddTHH:mm:ssZ or yyyy-MM-ddTHH:mm:ss+08:00", timeInISO8601));
        }

    }

    public static String getStandardFormatFromUnixTimestamp(Long timestamp, String timeZone) {
        return getFormatFromUnixTimestamp(timestamp, timeZone, STANDARD_FORMAT);
    }

    public static String getMsFormatFromUnixTimestamp(Long timestamp, String timeZone) {
        return getFormatFromUnixTimestamp(timestamp, timeZone, MS_FORMAT);
    }

    private static String getFormatFromUnixTimestamp(Long timestamp, String timeZone, String format){
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timeZone));
        Date date = new Date(timestamp);
        return simpleDateFormat.format(date);
    }

    public static String getStandardFormatFromUnixTimestamp(Long timestamp) {
        return getFormatFromUnixTimestamp(timestamp, "GMT+08:00", STANDARD_FORMAT);
    }
}
