package com.envisioniot.sub.client.internal;

import com.envisioniot.sub.common.generated.SubProto;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author by jie.jin on 2018/8/3.
 */
public class RequestFuture {

    private final AtomicReference<Object> result = new AtomicReference<>(INCOMPLETE_SENTINEL);
    private final CountDownLatch completedLatch = new CountDownLatch(1);
    private final long id;
    private final long timestamp;

    private static final long REQUEST_EXPIRATION_TIME = 5 * 60 * 1000;

    public RequestFuture(long id) {
        this.id = id;
        this.timestamp = System.currentTimeMillis();
    }

    private static final Object INCOMPLETE_SENTINEL = new Object();

    public boolean isDone() {
        return result.get() != INCOMPLETE_SENTINEL;
    }

    public boolean awaitDone(long timeout, TimeUnit unit) throws InterruptedException {
        return completedLatch.await(timeout, unit);
    }

    public SubProto.PullRsp value() {
        if (!succeeded()) {
            throw new IllegalStateException("Attempt to retrieve value from future which hasn't successfully completed");
        }
        return (SubProto.PullRsp) result.get();
    }

    public void complete(SubProto.PullRsp value) {
        try {
            if (!result.compareAndSet(INCOMPLETE_SENTINEL, value)) {
                throw new IllegalStateException("Invalid attempt to complete a request future which is already complete");
            }
        } finally {
            completedLatch.countDown();
        }
    }

    public boolean expired() {
        return System.currentTimeMillis() - timestamp > REQUEST_EXPIRATION_TIME;
    }

    public long getId() {
        return id;
    }

    public boolean failed() {
        return result.get() instanceof RuntimeException;
    }

    public boolean succeeded() {
        return isDone() && !failed();
    }
}
