package com.envisioniot.sub.client.internal.netty;

import com.envisioniot.sub.client.internal.ConnectionStateListener;
import com.envisioniot.sub.client.internal.MessageListener;
import com.envisioniot.sub.client.internal.RequestFuture;
import com.envisioniot.sub.client.internal.SubThread;
import com.envisioniot.sub.client.internal.netty.processor.AuthRspProcessor;
import com.envisioniot.sub.client.internal.netty.processor.PullRspProcessor;
import com.envisioniot.sub.client.internal.netty.processor.SubRspProcessor;
import com.envisioniot.sub.common.constants.MessageConstant;
import com.envisioniot.sub.common.generated.SubProto;
import com.envisioniot.sub.common.model.SubCategory;
import com.envisioniot.sub.common.model.TPartition;
import com.envisioniot.sub.common.netty.ChannelWriter;
import com.envisioniot.sub.common.netty.RegClientManager;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelOption;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * created by jie.jin on 2018/12/24.
 */
public class SubClient {

    private static final Logger LOG = LoggerFactory.getLogger(SubClient.class);

    private String host = "localhost";
    private int port = 9003;
    private String accessKey;
    private String secret;
    private SubCategory subCategory;
    private String subId;
    private String consumerGroup;
    private volatile boolean autoCommit = true;
    private volatile int commitInterval = 3000;
    private MessageListener msgListener;
    private ConnectionStateListener connectionStateListener;

    private int requestTimeout = 30000;
    private int requestQueueCap = 2;
    private BlockingQueue<RequestFuture> firedRequests;
    private long lastCommit = 0L;
    private Map<TPartition, Long> consumedOffsets;
    private AtomicLong idGen = new AtomicLong(0L);

    private AtomicBoolean threadStarted = new AtomicBoolean(false);

    private SendThread sendThread;
    private UserProcessThread userProcessThread;

    private Bootstrap bootstrap;
    private NioEventLoopGroup clientEventLoopGroup = new NioEventLoopGroup();

    private boolean isBatch = false;

    private void register(SubCategory subCategory) {
        RegClientManager.register(subCategory, SubProto.CmdId.idle_req_VALUE, SubProto.IdleReq.class);
        RegClientManager.register(subCategory, SubProto.CmdId.auth_req_VALUE, SubProto.AuthReq.class);
        RegClientManager
                .register(subCategory, SubProto.CmdId.auth_rsp_VALUE, SubProto.AuthRsp.class, new AuthRspProcessor(subCategory));
        RegClientManager.register(subCategory, SubProto.CmdId.sub_req_VALUE, SubProto.SubReq.class);
        RegClientManager
                .register(subCategory, SubProto.CmdId.sub_rsp_VALUE, SubProto.SubRsp.class, new SubRspProcessor(subCategory));
        RegClientManager.register(subCategory, SubProto.CmdId.pull_req_VALUE, SubProto.PullReq.class);
        RegClientManager
                .register(subCategory, SubProto.CmdId.pull_rsp_VALUE, SubProto.PullRsp.class, new PullRspProcessor(subCategory));
        RegClientManager.register(subCategory, SubProto.CmdId.commit_req_VALUE, SubProto.CommitDTO.class);
    }

    public SubClient(final SubCategory subCategory) {
        // attach shutdown handler to catch control-c or SIGTERM
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
            @Override
            public void run() {
                SendThread sendThread = SubClientCache.get(subCategory).getSendThread();
                if (sendThread != null) {
                    sendThread.disable();
                }

                UserProcessThread userProcessThread = SubClientCache.get(subCategory).getUserProcessThread();
                if (userProcessThread != null) {
                    userProcessThread.disable();
                }

                LOG.info(subCategory + " process has exited.");

            }
        }));

        register(subCategory);
        this.consumedOffsets = new ConcurrentHashMap<>();
        this.bootstrap = new Bootstrap();
        this.bootstrap.group(clientEventLoopGroup);
        this.bootstrap.channel(NioSocketChannel.class);
        this.bootstrap.remoteAddress(this.host, this.port);
        this.bootstrap.option(ChannelOption.SO_KEEPALIVE, true);
        this.bootstrap.option(ChannelOption.TCP_NODELAY, true);
        this.bootstrap.handler(new SubClientChannelInitializer(subCategory, false));
    }

    public void pauseSub() {
        if (sendThread != null) {
            sendThread.disable();
        }

        if (userProcessThread != null) {
            userProcessThread.disable();
        }

        shutdownGracefully();
    }

    public void connect() {
        if (null == this.firedRequests) {
            this.firedRequests = new LinkedBlockingQueue<>(requestQueueCap);
        }
        this.firedRequests.clear();
        this.bootstrap.connect().addListener(
                new ChannelFutureListener() {
                    @Override
                    public void operationComplete(ChannelFuture future) throws Exception {
                        if (!future.isSuccess()) {
                            future.channel().eventLoop().schedule(new Runnable() {
                                @Override
                                public void run() {
                                    connect();
                                }
                            }, 5, TimeUnit.SECONDS);
                            LOG.info("connect failed, try again later");
                        } else {
                            LOG.info("connect success");
                            if (null != connectionStateListener) {
                                connectionStateListener.connected();
                            }
                        }
                    }
                }
        );
    }

    public void startPull(ChannelHandlerContext ctx) {
        if (threadStarted.compareAndSet(false, true)) {
            this.sendThread = new SendThread(ctx);
            this.sendThread.setDaemon(true);
            this.sendThread.start();
            this.userProcessThread = new UserProcessThread(ctx);
            this.userProcessThread.setDaemon(true);
            this.userProcessThread.start();
        } else {
            this.sendThread.resetCtx(ctx);
            this.userProcessThread.resetCtx(ctx);
        }
    }

    public SendThread getSendThread() {
        return sendThread;
    }

    public UserProcessThread getUserProcessThread() {
        return userProcessThread;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getAccessKey() {
        return accessKey;
    }

    public String getSecret() {
        return secret;
    }

    public SubCategory getSubCategory() {
        return subCategory;
    }

    public String getSubId() {
        return subId;
    }

    public String getConsumerGroup() {
        return consumerGroup;
    }

    public BlockingQueue<RequestFuture> getFiredRequests() {
        return firedRequests;
    }

    public SubClient setHost(String host) {
        this.host = host;
        this.bootstrap.remoteAddress(this.host, this.port);
        return this;
    }

    public SubClient setPort(int port) {
        this.port = port;
        this.bootstrap.remoteAddress(this.host, this.port);
        return this;
    }

    public SubClient setAccessKey(String accessKey) {
        this.accessKey = accessKey;
        return this;
    }

    public SubClient setSecret(String secret) {
        this.secret = secret;
        return this;
    }

    public SubClient setSubCategory(SubCategory subCategory) {
        this.subCategory = subCategory;
        return this;
    }

    public SubClient setSubId(String subId) {
        this.subId = subId;
        return this;
    }

    public SubClient setConsumerGroup(String consumerGroup) {
        this.consumerGroup = consumerGroup;
        return this;
    }

    public SubClient setMessageListener(MessageListener listener) {
        this.msgListener = listener;
        return this;
    }

    public SubClient setConnectionStateListener(ConnectionStateListener connectionStateListener) {
        this.connectionStateListener = connectionStateListener;
        return this;
    }

    public ConnectionStateListener getConnectionStateListener() {
        return connectionStateListener;
    }

    public SubClient enableAutoCommit() {
        this.autoCommit = true;
        return this;
    }

    public SubClient disableAutoCommit() {
        this.autoCommit = false;
        return this;
    }


    public SubClient setAutoCommit(boolean autoCommit) {
        this.autoCommit = autoCommit;
        return this;
    }

    public boolean isAutoCommit() {
        return autoCommit;
    }

    public SubClient setAutoCommitInterval(int interval) {
        this.commitInterval = interval;
        return this;
    }

    public SubClient enablePreFetch() {
        this.requestQueueCap = 2;
        return this;
    }

    public SubClient disablePreFetch() {
        this.requestQueueCap = 1;
        return this;
    }

    public SubClient setRequestTimeout(int timeout) {
        this.requestTimeout = timeout;
        return this;
    }

    public SubClient setBatch(boolean batch) {
        isBatch = batch;
        return this;
    }
    public boolean isBatch() {
        return isBatch;
    }

    private SubProto.CommitDTO takeOutCommits() {
        SubProto.CommitDTO.Builder builder = SubProto.CommitDTO.newBuilder();
        for (Map.Entry<TPartition, Long> entry : this.consumedOffsets.entrySet()) {
            TPartition tp = entry.getKey();
            long offset = entry.getValue() + 1;
            builder.addCommits(SubProto.Commit.newBuilder().setTopic(tp.topic()).setPartition(tp.partition()).setOffset(offset).build());
        }
        this.consumedOffsets.clear();
        return builder.build();
    }

    public void shutdownGracefully() {
        clientEventLoopGroup.shutdownGracefully();
    }

    /**
     * simple flow control, i.e., we have limited requests on flight
     */
    private class SendThread extends SubThread {
        private long idleCnt = 0L;
        private AtomicReference<ChannelHandlerContext> ctx = new AtomicReference<>();

        private SendThread(ChannelHandlerContext ctx) {
            super("send-thread", false);
            this.ctx.set(ctx);
        }

        @Override
        public void run() {
            while (!closed.get()) {
                if (firePullRequest()) {
                    idleCnt = 0;
                } else {
                    ++idleCnt;
                }
                doBackoffInner(idleCnt);
            }
        }

        public void resetCtx(ChannelHandlerContext ctx) {
            this.ctx.set(ctx);
        }

        private boolean firePullRequest() {
            if (null == this.ctx.get() || !this.ctx.get().channel().isWritable()) {
                return false;
            }

            SubProto.PullReq pull = SubProto.PullReq.newBuilder().setId(idGen.getAndIncrement()).build();
            RequestFuture future = new RequestFuture(pull.getId());
            try {
                firedRequests.put(future);
            } catch (InterruptedException e) {
                LOG.error("put firedRequests queue exception. ", e);
                return false;
            }
            ChannelWriter.writeToChannel(this.ctx.get(), pull, true);
            return true;
        }
    }

    private class UserProcessThread extends SubThread {
        private long idleCnt = 0L;
        private final AtomicReference<ChannelHandlerContext> ctx = new AtomicReference<>();

        private UserProcessThread(ChannelHandlerContext ctx) {
            super("user-process-thread", true);
            this.ctx.set(ctx);
        }

        @Override
        public void run() {
            while (!closed.get()) {
                RequestFuture rf = firedRequests.peek();
                if (null == rf) {
                    ++idleCnt;
                    doBackoffInner(idleCnt);
                    continue;
                } else {
                    idleCnt = 0L;
                }
                SubProto.PullRsp response = null;
                try {
                    if (rf.awaitDone(requestTimeout, TimeUnit.MILLISECONDS)) {
                        response = rf.value();
                    }
                } catch (InterruptedException e) {
                    LOG.error("future await interrupted.", e);
                }
                synchronized (getFiredRequests()) {
                    if (rf == firedRequests.peek()) {
                        firedRequests.poll();
                    }
                }
                if (null == response) {
                    LOG.warn(String.format("wait for response of future (%d) timeout.", rf.getId()));
                    continue;
                }

                //TODO: check response code to decide if we need to force close channel
                if (response.getCode() != 0) {
                    LOG.warn("receive rsp code: " + response.getCode());
                }

                if (LOG.isDebugEnabled()) {
                    LOG.debug("receive message batch size: " + response.getMsgDTO().getMessagesCount());
                }


                List<String> patchList = new ArrayList<>();
                for (SubProto.Message message : response.getMsgDTO().getMessagesList()) {
                    if (null == msgListener) {
                        LOG.error("msgListener is null");
                        break;
                    }
                    if (!MessageConstant.NOT_MATCHED_MESSAGE.equals(message.getValue())) {
                        try {
                            if (isBatch) {
                                patchList.add(message.getValue());
                            } else {
                                msgListener.onMessage(message);
                            }
                        } catch (Exception e) {
                            LOG.error("handler message exception, sub client will do not auto commit!", e);
                            throw e;
                        }
                    }
                    consumedOffsets.put(new TPartition(message.getTopic(), message.getPartition()), message.getOffset());
                }

                if (isBatch && !patchList.isEmpty()) {
                    try {
                        msgListener.onMessages(patchList);
                    } catch (Exception e) {
                        LOG.error("handler message list exception, sub client will do not auto commit!", e);
                        throw e;
                    }
                }

                doAutoCommit();
            }
        }

        public void resetCtx(ChannelHandlerContext ctx) {
            this.ctx.set(ctx);
        }

        private void doCommit(long now) {
            SubProto.CommitDTO commitDTO = takeOutCommits();
            if (commitDTO.getCommitsCount() > 0) {
                lastCommit = now;
                ChannelWriter.writeToChannel(this.ctx.get(), commitDTO, true, 3000L);
            }
        }

        private void doAutoCommit() {
            if (!autoCommit) {
                return;
            }

            // 注视掉限流的数量
            long now = System.currentTimeMillis();
            if (now - lastCommit <= commitInterval) {
                return;
            }

            doCommit(now);
        }
    }

    /**
     * 开放自主提交功能, 须同时满足autoCommit为false, userProcessThread不为空
     */
    public void selfDoCommit() {
        if (!autoCommit && this.userProcessThread != null) {
            this.userProcessThread.doCommit(System.currentTimeMillis());
        }
    }

    public static void doBackoffInner(long idleCnt) {
        if (idleCnt > 0L) {
            long sleepTime = Math.min((long) Math.pow(2.0D, (double) (idleCnt - 1L)), 1000);

            try {
                Thread.sleep(sleepTime);
            } catch (InterruptedException ignored) {
            }

        }
    }
}
