/*
 * Decompiled with CFR 0.152.
 */
package com.epam.reportportal.service;

import com.epam.reportportal.exception.InternalReportPortalClientException;
import com.epam.reportportal.listeners.ListenerParameters;
import com.epam.reportportal.service.LaunchIdLock;
import com.epam.reportportal.utils.Waiter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.validation.constraints.NotNull;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LockFile
implements LaunchIdLock {
    private static final Logger LOGGER = LoggerFactory.getLogger(LockFile.class);
    public static final String LOCK_FILE_CHARSET = "US-ASCII";
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final float MAX_WAIT_TIME_DISCREPANCY = 0.1f;
    private final File lockFile;
    private final File syncFile;
    private final long fileWaitTimeout;
    private volatile String lockUuid;
    private volatile Pair<RandomAccessFile, FileLock> mainLock;

    public LockFile(ListenerParameters parameters) {
        this.lockFile = new File(parameters.getLockFileName());
        this.syncFile = new File(parameters.getSyncFileName());
        this.fileWaitTimeout = parameters.getFileWaitTimeout();
    }

    private Pair<RandomAccessFile, FileLock> obtainLock(File file) {
        RandomAccessFile lockAccess;
        String filePath = file.getPath();
        try {
            lockAccess = new RandomAccessFile(file, "rwd");
        }
        catch (FileNotFoundException e) {
            LOGGER.debug("Unable to open '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
            return null;
        }
        try {
            FileLock lock = lockAccess.getChannel().tryLock();
            if (lock == null) {
                LockFile.closeAccess(lockAccess);
                return null;
            }
            return Pair.of((Object)lockAccess, (Object)lock);
        }
        catch (OverlappingFileLockException e) {
            LOGGER.debug("Lock already acquired on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        catch (ClosedChannelException e) {
            LOGGER.warn("Channel was already closed on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while obtaining mainLock on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        LockFile.closeAccess(lockAccess);
        return null;
    }

    private Pair<RandomAccessFile, FileLock> waitForLock(File file) {
        return new Waiter("Wait for file '" + file.getPath() + "' lock").duration(this.fileWaitTimeout, TimeUnit.MILLISECONDS).applyRandomDiscrepancy(0.1f).till(() -> this.obtainLock(file));
    }

    private static void releaseLock(FileLock lock) {
        try {
            lock.release();
        }
        catch (ClosedChannelException e) {
            LOGGER.warn("Channel was already closed for file mainLock: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while releasing file mainLock: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
    }

    private static void closeAccess(RandomAccessFile access) {
        try {
            access.close();
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while closing file: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
    }

    private static String readLaunchUuid(RandomAccessFile access) throws IOException {
        return access.readLine();
    }

    private static void writeString(RandomAccessFile access, String text) throws IOException {
        access.write(text.getBytes(LOCK_FILE_CHARSET));
    }

    private static void writeLine(RandomAccessFile access, String text) throws IOException {
        LockFile.writeString(access, text + LINE_SEPARATOR);
    }

    private static void closeIo(Pair<RandomAccessFile, FileLock> io) {
        LockFile.releaseLock((FileLock)io.getRight());
        LockFile.closeAccess((RandomAccessFile)io.getLeft());
    }

    private <T> T executeBlockingOperation(IoOperation<T> operation, File file) {
        return (T)new Waiter("Wait for a blocking operation on file '" + file.getPath() + "'").duration(this.fileWaitTimeout, TimeUnit.MILLISECONDS).applyRandomDiscrepancy(0.1f).till(() -> {
            Pair<RandomAccessFile, FileLock> fileIo = this.obtainLock(file);
            if (fileIo != null) {
                try {
                    Object t = operation.execute(fileIo);
                    return t;
                }
                catch (IOException e) {
                    LOGGER.error("Unable to read/write a file after obtaining mainLock: " + e.getMessage(), (Throwable)e);
                }
                finally {
                    LockFile.closeIo(fileIo);
                }
            }
            return null;
        });
    }

    private void rewriteWith(RandomAccessFile access, String content) throws IOException {
        access.setLength(content.length());
        LockFile.writeLine(access, content);
    }

    void reset() {
        if (this.mainLock != null) {
            LockFile.closeIo(this.mainLock);
            this.mainLock = null;
        }
        this.lockUuid = null;
    }

    private void writeLaunchUuid(Pair<RandomAccessFile, FileLock> syncIo) {
        try {
            this.rewriteWith((RandomAccessFile)syncIo.getLeft(), this.lockUuid);
            this.rewriteWith((RandomAccessFile)this.mainLock.getLeft(), this.lockUuid);
        }
        catch (IOException e) {
            String error = "Unable to read/write a file after obtaining lock: " + e.getMessage();
            LOGGER.warn(error, (Throwable)e);
            this.reset();
            throw new InternalReportPortalClientException(error, e);
        }
    }

    private static void appendUuid(RandomAccessFile access, String uuid) throws IOException {
        access.seek(access.length());
        LockFile.writeLine(access, uuid);
    }

    private void writeInstanceUuid(String instanceUuid) {
        IoOperation<Boolean> uuidRead = fileIo -> {
            LockFile.appendUuid((RandomAccessFile)fileIo.getKey(), instanceUuid);
            return Boolean.TRUE;
        };
        this.executeBlockingOperation(uuidRead, this.syncFile);
    }

    private String obtainLaunch(String instanceUuid) {
        IoOperation<String> uuidRead = fileIo -> {
            RandomAccessFile access = (RandomAccessFile)fileIo.getKey();
            String uuid = LockFile.readLaunchUuid(access);
            LockFile.appendUuid(access, instanceUuid);
            return Optional.ofNullable(uuid).orElse(instanceUuid);
        };
        return this.executeBlockingOperation(uuidRead, this.syncFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public String obtainLaunchUuid(@NotNull String uuid) {
        Objects.requireNonNull(uuid);
        if (this.mainLock != null) {
            if (!uuid.equals(this.lockUuid)) {
                this.writeInstanceUuid(uuid);
            }
            return this.lockUuid;
        }
        Pair<RandomAccessFile, FileLock> syncLock = this.obtainLock(this.syncFile);
        if (syncLock != null) {
            try {
                Pair<RandomAccessFile, FileLock> lock;
                if (this.mainLock == null && (lock = this.obtainLock(this.lockFile)) != null) {
                    this.mainLock = lock;
                    this.lockUuid = uuid;
                    this.writeLaunchUuid(syncLock);
                    String string = uuid;
                    return string;
                }
            }
            finally {
                LockFile.closeIo(syncLock);
            }
        }
        return this.obtainLaunch(uuid);
    }

    @Override
    public void finishInstanceUuid(String uuid) {
        if (uuid == null) {
            return;
        }
        IoOperation<Boolean> uuidRemove = fileIo -> {
            String line;
            ArrayList<String> uuidList = new ArrayList<String>();
            RandomAccessFile fileAccess = (RandomAccessFile)fileIo.getKey();
            boolean needUpdate = false;
            while ((line = fileAccess.readLine()) != null) {
                String trimmedLine = line.trim();
                if (uuid.equals(trimmedLine)) {
                    needUpdate = true;
                    continue;
                }
                uuidList.add(trimmedLine);
            }
            if (!needUpdate) {
                return false;
            }
            String uuidNl = uuid + LINE_SEPARATOR;
            long newLength = fileAccess.length() - (long)uuidNl.length();
            if (newLength > 0L) {
                fileAccess.setLength(newLength);
                fileAccess.seek(0L);
                for (String uuid1 : uuidList) {
                    LockFile.writeLine(fileAccess, uuid1);
                }
                return false;
            }
            ((RandomAccessFile)fileIo.getKey()).setLength(0L);
            return true;
        };
        Boolean isLast = this.executeBlockingOperation(uuidRemove, this.syncFile);
        if (isLast != null && isLast.booleanValue()) {
            this.syncFile.delete();
        }
        if (this.mainLock != null && this.lockUuid.equals(uuid)) {
            this.reset();
            this.lockFile.delete();
        }
    }

    private static interface IoOperation<T> {
        public T execute(Pair<RandomAccessFile, FileLock> var1) throws IOException;
    }
}

