/*
 * Decompiled with CFR 0.152.
 */
package com.epam.reportportal.service;

import com.epam.reportportal.exception.InternalReportPortalClientException;
import com.epam.reportportal.listeners.ListenerParameters;
import com.epam.reportportal.utils.concurrency.LockCloseable;
import com.epam.reportportal.utils.http.ClientUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.annotation.Nonnull;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import okhttp3.FormBody;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OAuth2PasswordGrantAuthInterceptor
implements Interceptor {
    private static final Logger LOGGER = LoggerFactory.getLogger(OAuth2PasswordGrantAuthInterceptor.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final long TOKEN_EXPIRY_BUFFER_MILLIS = TimeUnit.MINUTES.toMillis(1L);
    private final ListenerParameters parameters;
    private volatile String accessToken;
    private volatile String refreshToken;
    private volatile long tokenExpiryTime;
    private volatile long lastAttemptTime;
    private volatile long lastAuthFailureTime;
    private final LockCloseable tokenLock = new LockCloseable(new ReentrantLock());
    private final OkHttpClient client;

    private static URL parseTokenUri(@Nonnull ListenerParameters parameters) {
        URL tokenUrl;
        String tokenUrlStr = parameters.getOauthTokenUri();
        if (StringUtils.isBlank((CharSequence)tokenUrlStr)) {
            LOGGER.error("URL for oAuth token is not set!");
            throw new InternalReportPortalClientException("URL for oAuth token is not set!");
        }
        try {
            tokenUrl = new URL(tokenUrlStr);
        }
        catch (MalformedURLException e) {
            LOGGER.error("Unable to parse oAuth token URL", (Throwable)e);
            throw new InternalReportPortalClientException("Unable to parse oAuth token URL", e);
        }
        return tokenUrl;
    }

    public OAuth2PasswordGrantAuthInterceptor(@Nonnull OkHttpClient client, @Nonnull ListenerParameters parameters) {
        this.client = client;
        this.parameters = parameters;
        OAuth2PasswordGrantAuthInterceptor.parseTokenUri(parameters);
    }

    public OAuth2PasswordGrantAuthInterceptor(@Nonnull ListenerParameters parameters) {
        this.parameters = parameters;
        URL tokenUrl = OAuth2PasswordGrantAuthInterceptor.parseTokenUri(parameters);
        OkHttpClient.Builder clientBuilder = ClientUtils.setupSsl(new OkHttpClient.Builder(), tokenUrl, parameters);
        if (parameters.isOauthUseProxy()) {
            ClientUtils.setupProxy(clientBuilder, parameters);
        }
        Optional.ofNullable(parameters.getHttpConnectTimeout()).ifPresent(d -> clientBuilder.connectTimeout(d.toMillis(), TimeUnit.MILLISECONDS));
        Optional.ofNullable(parameters.getHttpReadTimeout()).ifPresent(d -> clientBuilder.readTimeout(d.toMillis(), TimeUnit.MILLISECONDS));
        Optional.ofNullable(parameters.getHttpWriteTimeout()).ifPresent(d -> clientBuilder.writeTimeout(d.toMillis(), TimeUnit.MILLISECONDS));
        Optional.ofNullable(parameters.getHttpCallTimeout()).ifPresent(d -> clientBuilder.callTimeout(d.toMillis(), TimeUnit.MILLISECONDS));
        this.client = clientBuilder.build();
    }

    private boolean parseTokenResponse(@Nonnull String responseBody) {
        Map tokenResponse;
        try {
            tokenResponse = (Map)MAPPER.readValue(responseBody, Map.class);
        }
        catch (JsonProcessingException e) {
            LOGGER.error("Failed to parse OAuth 2.0 token response", (Throwable)e);
            return false;
        }
        this.accessToken = (String)tokenResponse.get("access_token");
        if (this.accessToken == null) {
            LOGGER.error("OAuth 2.0 token response does not contain 'access_token' field");
            return false;
        }
        this.refreshToken = (String)tokenResponse.get("refresh_token");
        Object expiresIn = tokenResponse.get("expires_in");
        if (expiresIn != null) {
            long expiresInSeconds = expiresIn instanceof Number ? ((Number)expiresIn).longValue() : Long.parseLong(expiresIn.toString());
            this.tokenExpiryTime = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(expiresInSeconds);
        } else {
            this.tokenExpiryTime = System.currentTimeMillis() + TimeUnit.HOURS.toMillis(1L);
        }
        LOGGER.debug("Successfully obtained OAuth 2.0 access token, expires in {} seconds", (Object)TimeUnit.MILLISECONDS.toSeconds(this.tokenExpiryTime - System.currentTimeMillis()));
        return true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean executeTokenRequest(@Nonnull Request tokenRequest) {
        try (Response response = this.client.newCall(tokenRequest).execute();){
            if (!response.isSuccessful()) {
                LOGGER.error("Failed to obtain OAuth 2.0 token. Status code: {}", (Object)response.code());
                boolean bl2 = false;
                return bl2;
            }
            String responseBody = response.body() != null ? response.body().string() : "{}";
            boolean bl = this.parseTokenResponse(responseBody);
            return bl;
        }
        catch (IOException e) {
            LOGGER.error("Failed to obtain OAuth 2.0 token", (Throwable)e);
            return false;
        }
    }

    private boolean isTokenValid() {
        return this.accessToken != null && System.currentTimeMillis() < this.tokenExpiryTime - TOKEN_EXPIRY_BUFFER_MILLIS;
    }

    private boolean refreshAccessToken() {
        this.lastAttemptTime = System.currentTimeMillis();
        FormBody.Builder formBuilder = new FormBody.Builder().add("grant_type", "refresh_token").add("refresh_token", this.refreshToken).add("client_id", Objects.requireNonNull(this.parameters.getOauthClientId()));
        Optional.ofNullable(this.parameters.getOauthClientSecret()).ifPresent(s -> formBuilder.add("client_secret", s));
        FormBody formBody = formBuilder.build();
        Request tokenRequest = new Request.Builder().url(Objects.requireNonNull(this.parameters.getOauthTokenUri())).post((RequestBody)formBody).build();
        return this.executeTokenRequest(tokenRequest);
    }

    private void obtainAccessToken() {
        this.lastAttemptTime = System.currentTimeMillis();
        FormBody.Builder formBuilder = new FormBody.Builder().add("grant_type", "password").add("username", Objects.requireNonNull(this.parameters.getOauthUsername())).add("password", Objects.requireNonNull(this.parameters.getOauthPassword())).add("client_id", Objects.requireNonNull(this.parameters.getOauthClientId()));
        Optional.ofNullable(this.parameters.getOauthClientSecret()).ifPresent(s -> formBuilder.add("client_secret", s));
        Optional.ofNullable(this.parameters.getOauthScope()).ifPresent(s -> formBuilder.add("scope", s));
        FormBody formBody = formBuilder.build();
        Request tokenRequest = new Request.Builder().url(Objects.requireNonNull(this.parameters.getOauthTokenUri())).post((RequestBody)formBody).build();
        this.executeTokenRequest(tokenRequest);
    }

    private void invalidateToken() {
        try (LockCloseable ignored = this.tokenLock.lock();){
            this.accessToken = null;
            this.tokenExpiryTime = 0L;
            this.lastAttemptTime = 0L;
        }
    }

    private void ensureValidToken() {
        if (this.isTokenValid()) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        if (this.lastAttemptTime > 0L && TimeUnit.MILLISECONDS.toSeconds(currentTime) == TimeUnit.MILLISECONDS.toSeconds(this.lastAttemptTime)) {
            return;
        }
        try (LockCloseable ignored = this.tokenLock.lock();){
            boolean refreshed;
            if (this.isTokenValid()) {
                return;
            }
            if (this.lastAttemptTime > 0L && TimeUnit.MILLISECONDS.toSeconds(currentTime) == TimeUnit.MILLISECONDS.toSeconds(this.lastAttemptTime)) {
                return;
            }
            if (this.refreshToken != null && (refreshed = this.refreshAccessToken())) {
                return;
            }
            this.obtainAccessToken();
        }
    }

    @Nonnull
    public Response intercept(@Nonnull Interceptor.Chain chain) throws IOException {
        this.ensureValidToken();
        if (this.accessToken == null) {
            return chain.proceed(chain.request());
        }
        Request request = chain.request().newBuilder().header("Authorization", "Bearer " + this.accessToken).build();
        Response response = chain.proceed(request);
        if (response.code() != 403 && response.code() != 401) {
            return response;
        }
        long currentTime = System.currentTimeMillis();
        if (this.lastAuthFailureTime > 0L && TimeUnit.MILLISECONDS.toSeconds(currentTime) == TimeUnit.MILLISECONDS.toSeconds(this.lastAuthFailureTime)) {
            return response;
        }
        this.lastAuthFailureTime = currentTime;
        LOGGER.debug("Received {} response, attempting to refresh token and retry request", (Object)response.code());
        this.invalidateToken();
        this.ensureValidToken();
        if (this.accessToken == null) {
            return response;
        }
        response.close();
        Request retryRequest = chain.request().newBuilder().header("Authorization", "Bearer " + this.accessToken).build();
        response = chain.proceed(retryRequest);
        if (response.code() == 403 || response.code() == 401) {
            LOGGER.warn("Still received {} after token refresh, request may be forbidden", (Object)response.code());
        }
        return response;
    }
}

