/*
 * Decompiled with CFR 0.152.
 */
package com.evanlennick.retry4j;

import com.evanlennick.retry4j.AttemptStatus;
import com.evanlennick.retry4j.RetryExecutor;
import com.evanlennick.retry4j.Status;
import com.evanlennick.retry4j.config.RetryConfig;
import com.evanlennick.retry4j.config.RetryConfigBuilder;
import com.evanlennick.retry4j.exception.RetriesExhaustedException;
import com.evanlennick.retry4j.exception.UnexpectedException;
import com.evanlennick.retry4j.listener.RetryListener;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CallExecutor<T>
implements RetryExecutor<T, Status<T>> {
    private Logger logger = LoggerFactory.getLogger(CallExecutor.class);
    private RetryConfig config;
    private RetryListener afterFailedTryListener;
    private RetryListener beforeNextTryListener;
    private RetryListener onFailureListener;
    private RetryListener onSuccessListener;
    private RetryListener onCompletionListener;
    private Exception lastKnownExceptionThatCausedRetry;
    private Status<T> status = new Status();

    public CallExecutor() {
        this(new RetryConfigBuilder().fixedBackoff5Tries10Sec().build());
    }

    public CallExecutor(RetryConfig config) {
        this.config = config;
        this.status.setId(UUID.randomUUID().toString());
    }

    @Override
    public Status<T> execute(Callable<T> callable) {
        return this.execute((Callable)callable, (String)null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Status<T> execute(Callable<T> callable, String callName) {
        this.logger.trace("Starting retry4j execution with callable {}", (Object)this.config, callable);
        this.logger.debug("Starting retry4j execution with executor state {}", (Object)this);
        long start = System.currentTimeMillis();
        this.status.setStartTime(start);
        int maxTries = this.config.getMaxNumberOfTries();
        long millisBetweenTries = this.config.getDelayBetweenRetries().toMillis();
        this.status.setCallName(callName);
        AttemptStatus attemptStatus = new AttemptStatus();
        attemptStatus.setSuccessful(false);
        try {
            int tries;
            for (tries = 0; tries < maxTries && !attemptStatus.wasSuccessful(); ++tries) {
                this.logger.trace("Retry4j executing callable {}", callable);
                attemptStatus = this.tryCall(callable);
                if (!attemptStatus.wasSuccessful()) {
                    this.handleRetry(millisBetweenTries, tries + 1);
                }
                this.logger.trace("Retry4j retrying for time number {}", (Object)tries);
            }
            this.refreshRetryStatus(attemptStatus.wasSuccessful(), tries);
            this.status.setEndTime(System.currentTimeMillis());
            this.postExecutionCleanup(callable, maxTries, attemptStatus);
            this.logger.debug("Finished retry4j execution in {} ms", (Object)this.status.getTotalElapsedDuration().toMillis());
            this.logger.trace("Finished retry4j execution with executor state {}", (Object)this);
        }
        finally {
            if (null != this.onCompletionListener) {
                this.onCompletionListener.onEvent(this.status);
            }
        }
        return this.status;
    }

    /*
     * Enabled aggressive block sorting
     */
    private void postExecutionCleanup(Callable<T> callable, int maxTries, AttemptStatus<T> attemptStatus) {
        if (attemptStatus.wasSuccessful()) {
            this.status.setResult(attemptStatus.getResult());
            if (null == this.onSuccessListener) return;
            this.onSuccessListener.onEvent(this.status);
            return;
        }
        String failureMsg = String.format("Call '%s' failed after %d tries!", callable.toString(), maxTries);
        if (null != this.onFailureListener) {
            this.onFailureListener.onEvent(this.status);
            return;
        }
        this.logger.trace("Throwing retries exhausted exception");
        throw new RetriesExhaustedException(failureMsg, this.lastKnownExceptionThatCausedRetry, this.status);
    }

    private AttemptStatus<T> tryCall(Callable<T> callable) throws UnexpectedException {
        AttemptStatus<T> attemptStatus = new AttemptStatus<T>();
        try {
            boolean shouldRetryOnThisResult;
            T callResult = callable.call();
            boolean bl = shouldRetryOnThisResult = this.config.shouldRetryOnValue() != false && callResult.equals(this.config.getValueToRetryOn());
            if (shouldRetryOnThisResult) {
                attemptStatus.setSuccessful(false);
            } else {
                attemptStatus.setResult(callResult);
                attemptStatus.setSuccessful(true);
            }
        }
        catch (Exception e) {
            if (this.shouldThrowException(e)) {
                this.logger.trace("Throwing expected exception {}", (Throwable)e);
                throw new UnexpectedException("Unexpected exception thrown during retry execution!", e);
            }
            this.lastKnownExceptionThatCausedRetry = e;
            attemptStatus.setSuccessful(false);
        }
        return attemptStatus;
    }

    private void handleRetry(long millisBetweenTries, int tries) {
        this.refreshRetryStatus(false, tries);
        if (null != this.afterFailedTryListener) {
            this.afterFailedTryListener.onEvent(this.status);
        }
        this.sleep(millisBetweenTries, tries);
        if (null != this.beforeNextTryListener) {
            this.beforeNextTryListener.onEvent(this.status);
        }
    }

    private void refreshRetryStatus(boolean success, int tries) {
        long currentTime = System.currentTimeMillis();
        long elapsed = currentTime - this.status.getStartTime();
        this.status.setTotalTries(tries);
        this.status.setTotalElapsedDuration(Duration.of(elapsed, ChronoUnit.MILLIS));
        this.status.setSuccessful(success);
        this.status.setLastExceptionThatCausedRetry(this.lastKnownExceptionThatCausedRetry);
    }

    private void sleep(long millis, int tries) {
        Duration duration = Duration.of(millis, ChronoUnit.MILLIS);
        long millisToSleep = this.config.getBackoffStrategy().getDurationToWait(tries, duration).toMillis();
        this.logger.trace("Retry4j executor sleeping for {} ms", (Object)millisToSleep);
        try {
            TimeUnit.MILLISECONDS.sleep(millisToSleep);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private boolean shouldThrowException(Exception e) {
        if (this.config.isRetryOnAnyException().booleanValue()) {
            return false;
        }
        for (Class<? extends Exception> exceptionInSet : this.config.getRetryOnSpecificExceptions()) {
            if (!exceptionInSet.isAssignableFrom(e.getClass())) continue;
            return false;
        }
        for (Class<? extends Exception> exceptionInSet : this.config.getRetryOnAnyExceptionExcluding()) {
            if (exceptionInSet.isAssignableFrom(e.getClass())) continue;
            return false;
        }
        return true;
    }

    @Override
    public void setConfig(RetryConfig config) {
        this.logger.trace("Set config on retry4j executor {}", (Object)config);
        this.config = config;
    }

    public CallExecutor<T> afterFailedTry(RetryListener listener) {
        this.afterFailedTryListener = listener;
        return this;
    }

    public CallExecutor<T> beforeNextTry(RetryListener listener) {
        this.beforeNextTryListener = listener;
        return this;
    }

    public CallExecutor<T> onCompletion(RetryListener listener) {
        this.onCompletionListener = listener;
        return this;
    }

    public CallExecutor<T> onSuccess(RetryListener listener) {
        this.onSuccessListener = listener;
        return this;
    }

    public CallExecutor<T> onFailure(RetryListener listener) {
        this.onFailureListener = listener;
        return this;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("CallExecutor{");
        sb.append("config=").append(this.config);
        sb.append(", afterFailedTryListener=").append(this.afterFailedTryListener);
        sb.append(", beforeNextTryListener=").append(this.beforeNextTryListener);
        sb.append(", onFailureListener=").append(this.onFailureListener);
        sb.append(", onSuccessListener=").append(this.onSuccessListener);
        sb.append(", lastKnownExceptionThatCausedRetry=").append(this.lastKnownExceptionThatCausedRetry);
        sb.append(", status=").append(this.status);
        sb.append('}');
        return sb.toString();
    }
}

