/*
 * Ext GWT - Ext for GWT
 * Copyright(c) 2007, 2008, Ext JS, LLC.
 * licensing@extjs.com
 * 
 * http://extjs.com/license
 */
package com.extjs.gxt.ui.client.widget.treegrid;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.extjs.gxt.ui.client.core.XDOM;
import com.extjs.gxt.ui.client.data.ModelData;
import com.extjs.gxt.ui.client.data.ModelIconProvider;
import com.extjs.gxt.ui.client.data.TreeLoader;
import com.extjs.gxt.ui.client.event.Events;
import com.extjs.gxt.ui.client.event.GridEvent;
import com.extjs.gxt.ui.client.event.TreeGridEvent;
import com.extjs.gxt.ui.client.store.ListStore;
import com.extjs.gxt.ui.client.store.StoreEvent;
import com.extjs.gxt.ui.client.store.StoreListener;
import com.extjs.gxt.ui.client.store.TreeStore;
import com.extjs.gxt.ui.client.store.TreeStoreEvent;
import com.extjs.gxt.ui.client.widget.grid.ColumnData;
import com.extjs.gxt.ui.client.widget.grid.ColumnModel;
import com.extjs.gxt.ui.client.widget.grid.Grid;
import com.extjs.gxt.ui.client.widget.grid.GridCellRenderer;
import com.extjs.gxt.ui.client.widget.tree.TreeStyle;
import com.extjs.gxt.ui.client.widget.tree.Tree.Joint;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;

public class TreeGrid<M extends ModelData> extends Grid<M> {

  public class TreeGridCellRenderer implements GridCellRenderer<M> {
    public Object render(M model, String property, ColumnData config, int rowIndex,
        int colIndex, ListStore<M> store, Grid<M> grid) {
      Joint j = calcualteJoint(model);
      String iconStyle = calculateIconStyle(model);
      int level = treeStore.getDepth(model);
      String text = model.get(property);
      return treeGridView.getTemplate("1", text, iconStyle, false, j.value(), level - 1);
    }
  }

  public class TreeNode {

    protected M m;
    protected String id;
    protected Element joint, check, text;
    private boolean expanded = false;
    private boolean expand;
    private boolean leaf = true;
    private boolean childrenRendered;
    private boolean loaded;

    public TreeNode(String id, M m) {
      this.id = id;
      this.m = m;
      m.set("gxt-id", id);
      if (loader != null && !loaded) {
        leaf = !loader.hasChildren(m);
      }
    }

    public int getItemCount() {
      return treeStore.getChildCount(m);
    }

    public M getModel() {
      return m;
    }

    public TreeNode getParent() {
      M p = treeStore.getParent(m);
      return findNode(p);
    }

    public int indexOf(TreeNode child) {
      M c = child.getModel();
      return store.indexOf(c);
    }

    public boolean isExpanded() {
      return expanded;
    }

    public boolean isLeaf() {
      return !hasChildren(m);
    }

    public void setExpanded(boolean expand) {
      TreeGrid.this.setExpanded(m, expand);
    }
  }

  protected Map<String, TreeNode> nodes = new HashMap<String, TreeNode>();
  protected TreeStore<M> treeStore;
  protected TreeLoader<M> loader;
  protected TreeGridView treeGridView;

  private ModelIconProvider<M> iconProvider;
  private TreeStyle style = new TreeStyle();
  private boolean autoLoad, filtering;
  private boolean caching = true;
  private StoreListener<M> storeListener = new StoreListener<M>() {
    @Override
    public void storeAdd(StoreEvent<M> se) {
      onAdd((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeClear(StoreEvent<M> se) {
      onDataChanged((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeDataChanged(StoreEvent<M> se) {
      onDataChanged((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeFilter(StoreEvent<M> se) {
      onFilter((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeRemove(StoreEvent<M> se) {
      onRemove((TreeStoreEvent<M>) se);
    }
  };

  @SuppressWarnings("unchecked")
  public TreeGrid(TreeStore store, ColumnModel cm) {
    super(new ListStore<M>(), cm);
    this.treeStore = store;
    this.loader = treeStore.getLoader();

    addStyleName("x-treegrid");

    treeStore.addStoreListener(storeListener);

    treeGridView = new TreeGridView();
    setView(treeGridView);

    setSelectionModel(new TreeGridSelectionModel<M>());
  }

  /**
   * Returns the tree style.
   * 
   * @return the tree style
   */
  public TreeStyle getStyle() {
    return style;
  }

  /**
   * Returns the tree's tree store.
   * 
   * @return the tree store
   */
  public TreeStore<M> getTreeStore() {
    return treeStore;
  }

  /**
   * Returns true if the model is expanded.
   * 
   * @param model the model
   * @return true if expanded
   */
  public boolean isExpanded(M model) {
    TreeNode node = findNode(model);
    return node.expanded;
  }

  /**
   * Returns true if the model is a leaf node. The leaf state allows a tree item
   * to specify if it has children before the children have been realized.
   * 
   * @param model the model
   * @return the leaf state
   */
  public boolean isLeaf(M model) {
    TreeNode node = findNode(model);
    return node.isLeaf();
  }

  /**
   * Sets the item's expand state.
   * 
   * @param model the model
   * @param expand true to expand
   */
  public void setExpanded(M model, boolean expand) {
    setExpanded(model, expand, false);
  }

  /**
   * Sets the item's expand state.
   * 
   * @param model the model
   * @param expand true to expand
   * @param deep true to expand all children recursively
   */
  public void setExpanded(M model, boolean expand, boolean deep) {
    TreeNode node = findNode(model);
    if (node != null) {
      TreeGridEvent<M> tge = new TreeGridEvent<M>(this);
      tge.setModel(model);

      if (expand && !node.expanded) {
        // if we have a loader and node is not loaded make
        // load request and exit method
        if (loader != null && (!node.loaded || !caching) && !filtering) {
          treeStore.removeAll(model);
          node.expand = true;
          loader.loadChildren(model);
          return;
        }

        if (!node.isLeaf()) {
          if (fireEvent(Events.BeforeExpand, tge)) {
            node.expanded = true;
            if (!node.childrenRendered) {
              renderChildren(model);
              node.childrenRendered = true;
            }
            // expand
            treeGridView.expand(node);

            M parent = treeStore.getParent(model);
            while (parent != null) {
              TreeNode pnode = findNode(parent);
              if (!pnode.expanded) {
                setExpanded(pnode.m, true);
              }
              parent = treeStore.getParent(parent);
            }

          }
        }
        if (deep) {
          expandChildren(model, true);
        }

      } else if (!expand && node.expanded) {
        if (fireEvent(Events.BeforeCollapse, tge)) {
          node.expanded = false;
          // collapse
          treeGridView.collapse(node);
        }
        if (deep) {
          for (M child : treeStore.getChildren(model)) {
            setExpanded(child, false, true);
          }
        }
      }
    }
  }

  /**
   * Toggles the model's expand state.
   * 
   * @param model the model
   */
  public void toggle(M model) {
    TreeNode node = findNode(model);
    if (node != null) {
      setExpanded(model, !node.expanded);
    }
  }

  protected TreeNode findNode(ModelData model) {
    if (model == null) return null;
    return nodes.get((String) model.get("gxt-id"));
  }

  protected boolean hasChildren(M model) {
    TreeNode node = findNode(model);
    if (loader != null && !node.loaded) {
      return loader.hasChildren(model);
    }
    if (!node.leaf || treeStore.getChildCount(model) > 0) {
      return true;
    }
    return false;
  }

  protected void onAdd(TreeStoreEvent<M> se) {

  }

  @Override
  protected void onClick(GridEvent<M> e) {
    super.onClick(e);
    M m = e.getModel();
    if (m != null) {
      TreeNode node = findNode(m);
      if (node != null) {
        if (e.within(treeGridView.getJointElement(node))) {
          toggle(m);
        }
      }
    }
  }

  protected void onDataChanged(TreeStoreEvent<M> se) {
    if (!isRendered()) {
      return;
    }

    M p = se.getParent();
    if (p == null) {
      el().setInnerHtml("");
      nodes.clear();
      renderChildren(null);
    } else {
      TreeNode n = findNode(p);
      n.loaded = true;

      if (n.childrenRendered) {
        // n.container.setInnerHTML("");
      }

      renderChildren(p);

      if (n.expand && !n.isLeaf()) {
        n.expand = false;
        setExpanded(p, true);
      }
    }
  }

  @Override
  protected void onDoubleClick(GridEvent<M> e) {
    super.onDoubleClick(e);
    M m = e.getModel();
    if (m != null) {
      TreeNode node = findNode(m);
      setExpanded(node.m, !node.expanded);
    }
  }

  protected void onFilter(TreeStoreEvent<M> se) {

  }

  protected void onRemove(TreeStoreEvent<M> se) {

  }

  @Override
  protected void onRender(Element target, int index) {
    super.onRender(target, index);

    cm.getColumn(0).setRenderer(new TreeGridCellRenderer());

    el().setTabIndex(0);

    if (treeStore.getRootItems().size() == 0 && loader != null) {
      loader.load();
    } else {
      renderChildren(null);
    }

    sinkEvents(Event.ONCLICK | Event.ONDBLCLICK | Event.MOUSEEVENTS | Event.KEYEVENTS);
  }

  protected void refresh(M model) {
    TreeNode node = findNode(model);
    if (node != null) {
      String style = calculateIconStyle(model);
      treeGridView.onIconStyleChange(findNode(model), style);
      Joint j = calcualteJoint(model);
      treeGridView.onJointChange(node, j);
    }
  }

  protected void register(M child) {
    String id = XDOM.getUniqueId();
    nodes.put(id, new TreeNode(id, child));
  }

  protected void renderChildren(M parent) {
    List<M> children = parent == null ? treeStore.getRootItems()
        : treeStore.getChildren(parent);

    for (M child : children) {
      register(child);
    }

    if (parent == null) {
      store.add(children);
    }

    for (M child : children) {
      if (loader != null) {
        if (autoLoad) {
          if (store.isFiltered()) {
            renderChildren(child);
          } else {
            loader.loadChildren(child);
          }
        }
      }
    }
  }

  protected void unregister(M item) {
    nodes.remove(item);
  }

  private Joint calcualteJoint(M model) {
    if (model == null) {
      return Joint.NONE;
    }
    TreeNode node = findNode(model);
    Joint joint = Joint.NONE;

    if (!node.isLeaf()) {
      boolean children = true;

      if (node.isExpanded()) {
        joint = children ? Joint.EXPANDED : Joint.NONE;
      } else {
        joint = children ? Joint.COLLAPSED : Joint.NONE;
      }
    }
    return joint;
  }

  private String calculateIconStyle(M model) {
    String style = null;
    if (iconProvider != null) {
      String iconStyle = iconProvider.getIcon(model);
      if (iconStyle != null) {
        return iconStyle;
      }
    }
    TreeNode node = findNode(model);
    TreeStyle ts = getStyle();
    if (!node.isLeaf()) {
      if (isExpanded(model) && ts.getNodeOpenIconStyle() != null) {
        style = ts.getNodeOpenIconStyle();
      } else if (isExpanded(model) && ts.getNodeOpenIconStyle() != null) {
        style = ts.getNodeCloseIconStyle();
      } else if (!isExpanded(model)) {
        style = ts.getNodeCloseIconStyle();
      }
    } else {
      style = ts.getLeafIconStyle();
    }
    return style;
  }

  private void expandChildren(M m, boolean deep) {
    for (M child : treeStore.getChildren(m)) {
      setExpanded(child, true);
    }
  }
}
