/*
 * Ext GWT - Ext for GWT
 * Copyright(c) 2007, 2008, Ext JS, LLC.
 * licensing@extjs.com
 * 
 * http://extjs.com/license
 */
package com.extjs.gxt.ui.client.widget.treepanel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.extjs.gxt.ui.client.core.El;
import com.extjs.gxt.ui.client.core.XDOM;
import com.extjs.gxt.ui.client.data.ModelData;
import com.extjs.gxt.ui.client.data.ModelIconProvider;
import com.extjs.gxt.ui.client.data.TreeLoadEvent;
import com.extjs.gxt.ui.client.data.TreeLoader;
import com.extjs.gxt.ui.client.event.CheckChangedListener;
import com.extjs.gxt.ui.client.event.CheckProvider;
import com.extjs.gxt.ui.client.event.ComponentEvent;
import com.extjs.gxt.ui.client.event.Events;
import com.extjs.gxt.ui.client.event.TreePanelEvent;
import com.extjs.gxt.ui.client.store.StoreEvent;
import com.extjs.gxt.ui.client.store.StoreListener;
import com.extjs.gxt.ui.client.store.TreeStore;
import com.extjs.gxt.ui.client.store.TreeStoreEvent;
import com.extjs.gxt.ui.client.widget.BoxComponent;
import com.extjs.gxt.ui.client.widget.menu.Menu;
import com.extjs.gxt.ui.client.widget.tree.TreeStyle;
import com.extjs.gxt.ui.client.widget.tree.Tree.CheckCascade;
import com.extjs.gxt.ui.client.widget.tree.Tree.CheckNodes;
import com.extjs.gxt.ui.client.widget.tree.Tree.Joint;
import com.google.gwt.dom.client.NodeList;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;

/**
 * A standard hierarchical tree widget.
 * 
 * @param <M> the model type
 */
public class TreePanel<M extends ModelData> extends BoxComponent implements
    CheckProvider<M> {

  public class TreeNode {

    protected M m;
    protected String id;
    protected Element element, container, joint, check, text;
    private boolean expanded = false;
    private boolean expand, checked;
    private boolean leaf = true;
    private boolean childrenRendered;
    private boolean loaded;

    TreeNode(String id, M m) {
      this.id = id;
      this.m = m;
      m.set("gxt-id", id);
      if (loader != null && !loaded) {
        leaf = !loader.hasChildren(m);
      }
    }

    public Element getElement() {
      return element;
    }

    public int getItemCount() {
      return store.getChildCount(m);
    }

    public M getModel() {
      return m;
    }

    public TreeNode getParent() {
      M p = store.getParent(m);
      return findNode(p);
    }

    public int indexOf(TreeNode child) {
      M c = child.getModel();
      return store.indexOf(c);
    }

    public boolean isExpanded() {
      return expanded;
    }

    public boolean isLeaf() {
      return !hasChildren(m);
    }

    public void setExpanded(boolean expand) {
      TreePanel.this.setExpanded(m, expand);
    }
  }

  protected TreePanelSelectionModel<M> sm;
  protected TreePanelView view = new TreePanelView();
  protected String displayProperty;
  protected TreeStore<M> store;
  protected TreeLoader<M> loader;
  protected String itemSelector = ".x-tree-item";
  protected Map<String, TreeNode> nodes = new HashMap<String, TreeNode>();

  private TreeStyle style = new TreeStyle();
  private ModelIconProvider<M> iconProvider;
  private boolean caching = true;
  private boolean autoLoad, filtering;
  private boolean expandOnFilter = true;
  private boolean checkable;
  private CheckNodes checkNodes = CheckNodes.BOTH;
  private CheckCascade checkStyle = CheckCascade.PARENTS;

  private StoreListener<M> storeListener = new StoreListener<M>() {
    @Override
    public void storeAdd(StoreEvent<M> se) {
      onAdd((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeClear(StoreEvent<M> se) {
      onDataChanged((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeDataChanged(StoreEvent<M> se) {
      onDataChanged((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeFilter(StoreEvent<M> se) {
      onFilter((TreeStoreEvent<M>) se);
    }

    @Override
    public void storeRemove(StoreEvent<M> se) {
      onRemove((TreeStoreEvent<M>) se);
    }
  };

  /**
   * Creates a new tree panel.
   * 
   * @param store the tree store
   */
  public TreePanel(TreeStore<M> store) {
    this.store = store;
    this.loader = store.getLoader();

    store.addStoreListener(storeListener);

    baseStyle = "my-tree x-ftree2-arrows";
    setSelectionModel(new TreePanelSelectionModel<M>());
    view.bind(this, store);
  }

  public void addCheckListener(CheckChangedListener<M> listener) {
    addListener(Events.CheckChange, listener);
  }

  /**
   * Collapses all nodes.
   */
  public void collapseAll() {
    for (M child : store.getRootItems()) {
      setExpanded(child, false);
    }
  }

  /**
   * Expands all nodes.
   */
  public void expandAll() {
    for (M child : store.getRootItems()) {
      setExpanded(child, true, true);
    }
  }

  public TreeNode findNode(Element target) {
    Element item = fly(target).findParentElement(itemSelector, 20);
    if (item != null) {
      String id = item.getId();
      TreeNode node = nodes.get(id);
      if (node != null) {
        return node;
      }
    }

    return null;
  }

  public List<M> getCheckedSelection() {
    List<M> checked = new ArrayList<M>();
    for (M m : store.getAllItems()) {
      if (isChecked(m)) {
        checked.add(m);
      }
    }
    return checked;
  }

  /**
   * Returns the child nodes value.
   * 
   * @return the child nodes value
   */
  public CheckNodes getCheckNodes() {
    return checkNodes;
  }

  /**
   * The check style value.
   * 
   * @return the check style
   */
  public CheckCascade getCheckStyle() {
    return checkStyle;
  }

  /**
   * Returns the display property.
   * 
   * @return the display property
   */
  public String getDisplayProperty() {
    return displayProperty;
  }

  /**
   * Returns the model icon provider.
   * 
   * @return the icon provider
   */
  public ModelIconProvider<M> getIconProvider() {
    return iconProvider;
  }

  /**
   * Returns the tree's selection model.
   * 
   * @return the selection model
   */
  public TreePanelSelectionModel<M> getSelectionModel() {
    return sm;
  }

  /**
   * Returns the tree's store.
   * 
   * @return the store
   */
  public TreeStore<M> getStore() {
    return store;
  }

  /**
   * Returns the tree style.
   * 
   * @return the tree style
   */
  public TreeStyle getStyle() {
    return style;
  }

  /**
   * Returns the tree's view.
   * 
   * @return the view
   */
  public TreePanelView getView() {
    return view;
  }

  /**
   * Returns true if auto load is enabled.
   * 
   * @return the auto load state
   */
  public boolean isAutoLoad() {
    return autoLoad;
  }

  /**
   * Returns true when a loader is queried for it's children each time a node is
   * expanded. Only applies when using a loader with the tree store.
   * 
   * @return true if caching
   */
  public boolean isCaching() {
    return caching;
  }

  /**
   * Returns true if check boxes are enabled.
   * 
   * @return the check box state
   */
  public boolean isCheckable() {
    return checkable;
  }

  public boolean isChecked(M model) {
    TreeNode node = findNode(model);
    if (node != null) {
      return node.checked;
    }
    return false;
  }

  /**
   * Returns true if the model is expanded.
   * 
   * @param model the model
   * @return true if expanded
   */
  public boolean isExpanded(M model) {
    TreeNode node = findNode(model);
    return node.expanded;
  }

  /**
   * Returns the if expand all and collapse all is enabled on filter changes.
   * 
   * @return the expand all collapse all state
   */
  public boolean isExpandOnFilter() {
    return expandOnFilter;
  }

  /**
   * Returns true if the model is a leaf node. The leaf state allows a tree item
   * to specify if it has children before the children have been realized.
   * 
   * @param model the model
   * @return the leaf state
   */
  public boolean isLeaf(M model) {
    TreeNode node = findNode(model);
    return node.isLeaf();
  }

  @Override
  @SuppressWarnings("unchecked")
  public void onComponentEvent(ComponentEvent ce) {
    super.onComponentEvent(ce);
    TreePanelEvent<M> tpe = (TreePanelEvent) ce;
    if (tpe.getItem() != null) {
      int type = ce.getEventTypeInt();
      switch (type) {
        case Event.ONCLICK:
          onClick(tpe);
          break;
        case Event.ONDBLCLICK:
          onDoubleClick(tpe);
          break;
      }
    }

    view.onEvent(tpe);
  }

  public void removeCheckListener(CheckChangedListener<M> listener) {
    removeListener(Events.CheckChange, listener);
  }

  /**
   * Sets whether all children should automatically be loaded recursively.
   * Useful when using filters. Only applies when using a loader.
   * 
   * @param autoLoad true to auto load
   */
  public void setAutoLoad(boolean autoLoad) {
    this.autoLoad = autoLoad;
  }

  /**
   * Sets whether the children should be cached after first being retrieved from
   * the store (defaults to true). When <code>false</code>, a load request will
   * be made each time a node is expanded.
   * 
   * @param caching the caching state
   */
  public void setCaching(boolean caching) {
    this.caching = caching;
  }

  /**
   * Sets whether check boxes are used in the tree.
   * 
   * @param checkable true for check boxes
   */
  public void setCheckable(boolean checkable) {
    this.checkable = checkable;
  }

  /**
   * Sets the check state of the item.
   * 
   * @param item the item
   * @param checked true for checked
   */
  public void setChecked(M item, boolean checked) {
    TreeNode node = findNode(item);
    if (node != null && node.check != null) {
      if (node.checked == checked) {
        return;
      }
      if (fireEvent(Events.BeforeCheckChange, new TreePanelEvent<M>(this, item))) {
        node.checked = checked;
        view.onCheckChange(node, checked);

        switch (getCheckStyle()) {
          case PARENTS:
            if (checked) {
              M p = store.getParent(item);
              while (p != null) {
                setChecked(p, true);
                p = store.getParent(p);
              }
            } else {
              for (M child : store.getChildren(item)) {
                setChecked(child, false);
              }
            }
            break;
          case CHILDREN:
            for (M child : store.getChildren(item)) {
              setChecked(child, checked);
            }
        }
      }
    }
  }

  public void setCheckedSelection(List<M> selection) {

  }

  /**
   * Sets which tree items will display a check box (defaults to BOTH).
   * <p>
   * Valid values are:
   * <ul>
   * <li>BOTH - both nodes and leafs</li>
   * <li>PARENT - only nodes with children</li>
   * <li>LEAF - only leafs</li>
   * </ul>
   * 
   * @param checkNodes the child nodes value
   */
  public void setCheckNodes(CheckNodes checkNodes) {
    this.checkNodes = checkNodes;
  }

  /**
   * Sets the cascading behavior for check tree (defaults to PARENTS).
   * <p>
   * Valid values are:
   * <ul>
   * <li>NONE - no cascading</li>
   * <li>PARENTS - cascade to parents</li>
   * <li>CHILDREN - cascade to children</li>
   * </ul>
   * 
   * @param checkStyle the child style
   */
  public void setCheckStyle(CheckCascade checkStyle) {
    this.checkStyle = checkStyle;
  }

  @Override
  public void setContextMenu(Menu menu) {
    super.setContextMenu(menu);
  }

  /**
   * Sets the display property name used to the item's text.
   * 
   * @param displayProperty the property
   */
  public void setDisplayProperty(String displayProperty) {
    this.displayProperty = displayProperty;
  }

  /**
   * Sets the item's expand state.
   * 
   * @param model the model
   * @param expand true to expand
   */
  public void setExpanded(M model, boolean expand) {
    setExpanded(model, expand, false);
  }

  /**
   * Sets the item's expand state.
   * 
   * @param model the model
   * @param expand true to expand
   * @param deep true to expand all children recursively
   */
  public void setExpanded(M model, boolean expand, boolean deep) {
    TreeNode node = findNode(model);
    if (node != null) {
      TreePanelEvent<M> tpe = new TreePanelEvent<M>(this);
      tpe.setItem(model);

      if (expand && !node.expanded) {
        // if we have a loader and node is not loaded make
        // load request and exit method
        if (loader != null && (!node.loaded || !caching) && !filtering) {
          store.removeAll(model);
          node.expand = true;
          loader.loadChildren(model);
          return;
        }

        if (!node.isLeaf()) {
          if (fireEvent(Events.BeforeExpand, tpe)) {
            node.expanded = true;
            if (!node.childrenRendered) {
              renderChildren(model);
              node.childrenRendered = true;
            }
            // expand
            view.expand(node);

            M parent = store.getParent(model);
            while (parent != null) {
              TreeNode pnode = findNode(parent);
              if (!pnode.expanded) {
                setExpanded(pnode.m, true);
              }
              parent = store.getParent(parent);
            }
          }
        }
        if (deep) {
          expandChildren(model, true);
        }
      } else if (!expand && node.expanded) {
        if (fireEvent(Events.BeforeCollapse, tpe)) {
          node.expanded = false;
          // collapse
          view.collapse(node);
        }
        if (deep) {
          for (M child : store.getChildren(model)) {
            setExpanded(child, false, true);
          }
        }
      }
    }
  }

  /**
   * Sets whether the tree should expand all and collapse all when filters are
   * applied (defaults to true).
   * 
   * @param expandOnFilter true to expand and collapse on filter changes
   */
  public void setExpandOnFilter(boolean expandOnFilter) {
    this.expandOnFilter = expandOnFilter;
  }

  /**
   * Sets the tree's model icon provider which provides the icon style for each
   * model.
   * 
   * @param iconProvider the icon provider
   */
  public void setIconProvider(ModelIconProvider<M> iconProvider) {
    this.iconProvider = iconProvider;
  }

  /**
   * Sets the tree's selection model.
   * 
   * @param sm the selection model
   */
  public void setSelectionModel(TreePanelSelectionModel<M> sm) {
    if (this.sm != null) {
      this.sm.bindTree(null);
    }
    this.sm = sm;
    if (sm != null) {
      sm.bindTree(this);
    }
  }

  /**
   * Sets the tree style.
   * 
   * @param style the tree style
   */
  public void setStyle(TreeStyle style) {
    this.style = style;
  }

  /**
   * Sets the tree's view.
   * 
   * @param view the view
   */
  public void setView(TreePanelView view) {
    this.view = view;
    view.bind(this, store);
  }

  /**
   * Toggles the model's expand state.
   * 
   * @param model the model
   */
  public void toggle(M model) {
    TreeNode node = findNode(model);
    if (node != null) {
      setExpanded(model, !node.expanded);
    }
  }

  @Override
  protected ComponentEvent createComponentEvent(Event event) {
    TreePanelEvent<M> tpe = new TreePanelEvent<M>(this, event);
    if (event != null) {
      TreeNode node = findNode((Element) event.getEventTarget().cast());
      if (node != null) {
        tpe.setItem(node.m);
        tpe.setNode(node);
      }
    }
    return tpe;
  }

  protected TreeNode findNode(ModelData model) {
    if (model == null) return null;
    return nodes.get((String) model.get("gxt-id"));
  }

  protected Element getContainer(M model) {
    if (model == null) {
      return getElement();
    }
    TreeNode node = findNode(model);
    if (node != null) {
      return node.container;
    }
    return null;
  }

  protected String getText(M model) {
    if (displayProperty != null) {
      return (String) model.get(displayProperty);
    }
    return "";
  }

  protected boolean hasChildren(M model) {
    TreeNode node = findNode(model);
    if (loader != null && !node.loaded) {
      return loader.hasChildren(model);
    }
    if (!node.leaf || store.getChildCount(model) > 0) {
      return true;
    }
    return false;
  }

  protected void onAdd(TreeStoreEvent<M> se) {
    M parent = se.getParent();
    TreeNode pn = findNode(parent);
    if (parent == null || (pn != null && pn.childrenRendered)) {
      StringBuilder sb = new StringBuilder();
      for (M child : se.getChildren()) {
        sb.append(renderChild(parent, child, store.getDepth(parent)));
      }
      Element div = DOM.createDiv();
      div.setInnerHTML(sb.toString());
      fly(getContainer(parent)).insertChild((Element) div.getFirstChildElement(),
          se.getIndex());
      NodeList<Element> items = getContainer(parent).getChildNodes().cast();

      for (int i = 0, len = items.getLength(); i < len; i++) {
        M child = parent == null ? store.getChild(i) : store.getChild(parent, i);
        TreeNode node = findNode(child);
        node.element = items.getItem(i);
        node.container = view.getContainer(node);
      }
    }

    refresh(parent);
  }

  protected void onBeforeLoad(TreeLoadEvent le) {

  }

  @SuppressWarnings("unchecked")
  protected void onCheckClick(TreePanelEvent tpe, TreeNode node) {
    tpe.stopEvent();
    setChecked((M) tpe.getItem(), !node.checked);
  }

  protected void onClear(TreeStoreEvent<M> se) {
    el().setInnerHtml("");
    nodes.clear();
  }

  @SuppressWarnings("unchecked")
  protected void onClick(TreePanelEvent tpe) {
    TreeNode node = tpe.getNode();
    if (node != null) {
      if (tpe.within(view.getJointElement(node))) {
        toggle((M) tpe.getItem());
      }
      if (checkable && tpe.within(view.getCheckElement(node))) {
        onCheckClick(tpe, node);
      }
    }
  }

  protected void onDataChanged(TreeStoreEvent<M> se) {
    if (!isRendered()) {
      return;
    }

    M p = se.getParent();
    if (p == null) {
      el().setInnerHtml("");
      nodes.clear();
      renderChildren(null);
    } else {
      TreeNode n = findNode(p);
      n.loaded = true;

      if (n.childrenRendered) {
        n.container.setInnerHTML("");
      }

      renderChildren(p);

      if (n.expand && !n.isLeaf()) {
        n.expand = false;
        setExpanded(p, true);
      }
    }
  }

  @SuppressWarnings("unchecked")
  protected void onDoubleClick(TreePanelEvent tpe) {
    TreeNode node = findNode((M) tpe.getItem());
    setExpanded(node.m, !node.expanded);
  }

  protected void onFilter(TreeStoreEvent<M> se) {
    filtering = store.isFiltered();
    el().setInnerHtml("");
    nodes.clear();
    renderChildren(null);

    if (expandOnFilter && store.isFiltered()) {
      expandAll();
    }
  }

  protected void onRemove(TreeStoreEvent<M> se) {
    TreeNode node = findNode(se.getChild());
    if (node != null && node.element != null) {
      El.fly(node.element).removeFromParent();
      nodes.remove(node);
      TreeNode p = findNode(se.getParent());
      if (p != null && p.expanded && p.getItemCount() == 0) {
        setExpanded(p.m, false);
      }
    }
  }

  @Override
  protected void onRender(Element target, int index) {
    super.onRender(target, index);
    setElement(DOM.createDiv(), target, index);
    disableTextSelection(true);

    el().setTabIndex(0);
    el().setElementAttribute("hideFocus", "true");
    
    if (store.getRootItems().size() == 0 && loader != null) {
      loader.load();
    } else {
      renderChildren(null);
    }

    sinkEvents(Event.ONCLICK | Event.ONDBLCLICK | Event.MOUSEEVENTS | Event.KEYEVENTS);
  }

  protected void refresh(M model) {
    TreeNode node = findNode(model);
    if (node != null) {
      String style = calculateIconStyle(model);
      view.onIconStyleChange(findNode(model), style);
      Joint j = calcualteJoint(model);
      view.onJointChange(node, j);
    }
  }

  protected String renderChild(M parent, M child, int depth) {
    String id = XDOM.getUniqueId();
    nodes.put(id, new TreeNode(id, child));
    Joint j = calcualteJoint(child);
    return view.getTemplate(child, id, getText(child), calculateIconStyle(child), checkable,
        j.value(), depth);
  }

  protected void renderChildren(M parent) {
    Element container = getContainer(parent);

    StringBuilder markup = new StringBuilder();
    int depth = store.getDepth(parent);
    List<M> children = parent == null ? store.getRootItems() : store.getChildren(parent);

    for (int i = 0; i < children.size(); i++) {
      markup.append(renderChild(parent, children.get(i), depth));
    }

    container.setInnerHTML(markup.toString());
    NodeList<Element> items = container.getChildNodes().cast();

    for (int i = 0, len = items.getLength(); i < len; i++) {
      M child = parent == null ? store.getChild(i) : store.getChild(parent, i);
      TreeNode node = findNode(child);
      node.element = items.getItem(i);
      node.container = node.element.getFirstChildElement().getNextSiblingElement().cast();

      if (loader != null) {
        if (autoLoad) {
          if (store.isFiltered()) {
            renderChildren(child);
          } else {
            loader.loadChildren(child);
          }
        }
      }
    }

    TreeNode n = findNode(parent);
    if (n != null) {
      n.childrenRendered = true;
    }

  }

  private Joint calcualteJoint(M model) {
    if (model == null) {
      return Joint.NONE;
    }
    TreeNode node = findNode(model);
    Joint joint = Joint.NONE;

    if (!node.isLeaf()) {
      boolean children = true;

      if (node.expanded) {
        joint = children ? Joint.EXPANDED : Joint.NONE;
      } else {
        joint = children ? Joint.COLLAPSED : Joint.NONE;
      }
    }
    return joint;
  }

  private String calculateIconStyle(M model) {
    String style = null;
    if (iconProvider != null) {
      String iconStyle = iconProvider.getIcon(model);
      if (iconStyle != null) {
        return iconStyle;
      }
    }
    TreeNode node = findNode(model);
    TreeStyle ts = getStyle();
    if (!node.isLeaf()) {
      if (isExpanded(model) && ts.getNodeOpenIconStyle() != null) {
        style = ts.getNodeOpenIconStyle();
      } else if (isExpanded(model) && ts.getNodeOpenIconStyle() != null) {
        style = ts.getNodeCloseIconStyle();
      } else if (!isExpanded(model)) {
        style = ts.getNodeCloseIconStyle();
      }
    } else {
      style = ts.getLeafIconStyle();
    }
    return style;
  }

  private void expandChildren(M m, boolean deep) {
    for (M child : store.getChildren(m)) {
      setExpanded(child, true);
    }
  }

}
