/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.airlift.discovery.store;

import com.facebook.airlift.concurrent.Threads;
import com.facebook.airlift.discovery.store.Entry;
import com.facebook.airlift.discovery.store.LocalStore;
import com.facebook.airlift.discovery.store.RemoteStore;
import com.facebook.airlift.discovery.store.StoreConfig;
import com.facebook.airlift.discovery.store.Version;
import com.facebook.airlift.units.Duration;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Streams;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import jakarta.inject.Inject;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.weakref.jmx.Managed;

public class DistributedStore {
    private final String name;
    private final LocalStore localStore;
    private final RemoteStore remoteStore;
    private final Supplier<ZonedDateTime> timeSupplier;
    private final Duration tombstoneMaxAge;
    private final Duration garbageCollectionInterval;
    private final ScheduledExecutorService garbageCollector;
    private final AtomicLong lastGcTimestamp = new AtomicLong();

    @Inject
    public DistributedStore(String name, LocalStore localStore, RemoteStore remoteStore, StoreConfig config, Supplier<ZonedDateTime> timeSupplier) {
        this.name = Objects.requireNonNull(name, "name is null");
        this.localStore = Objects.requireNonNull(localStore, "localStore is null");
        this.remoteStore = Objects.requireNonNull(remoteStore, "remoteStore is null");
        this.timeSupplier = Objects.requireNonNull(timeSupplier, "timeSupplier is null");
        Objects.requireNonNull(config, "config is null");
        this.tombstoneMaxAge = config.getTombstoneMaxAge();
        this.garbageCollectionInterval = config.getGarbageCollectionInterval();
        this.garbageCollector = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)("distributed-store-gc-" + name)));
    }

    @PostConstruct
    public void start() {
        this.garbageCollector.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                DistributedStore.this.removeExpiredEntries();
            }
        }, 0L, this.garbageCollectionInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Managed
    public String getName() {
        return this.name;
    }

    @Managed
    public long getLastGcTimestamp() {
        return this.lastGcTimestamp.get();
    }

    @Managed
    public void removeExpiredEntries() {
        for (Entry entry : this.localStore.getAll()) {
            if (!this.isExpired(entry)) continue;
            this.localStore.delete(entry.getKey(), entry.getVersion());
        }
        this.lastGcTimestamp.set(System.currentTimeMillis());
    }

    private boolean isExpired(Entry entry) {
        long ageInMs = this.timeSupplier.get().toInstant().toEpochMilli() - entry.getTimestamp();
        return entry.getValue() == null && ageInMs > this.tombstoneMaxAge.toMillis() || entry.getMaxAgeInMs() != null && ageInMs > entry.getMaxAgeInMs();
    }

    @PreDestroy
    public void shutdown() {
        this.garbageCollector.shutdownNow();
    }

    public void put(byte[] key, byte[] value) {
        Objects.requireNonNull(key, "key is null");
        Objects.requireNonNull(value, "value is null");
        long now = this.timeSupplier.get().toInstant().toEpochMilli();
        Entry entry = new Entry(key, value, new Version(now), now, null);
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public void put(byte[] key, byte[] value, Duration maxAge) {
        Objects.requireNonNull(key, "key is null");
        Objects.requireNonNull(value, "value is null");
        Objects.requireNonNull(maxAge, "maxAge is null");
        long now = this.timeSupplier.get().toInstant().toEpochMilli();
        Entry entry = new Entry(key, value, new Version(now), now, maxAge.toMillis());
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public byte[] get(byte[] key) {
        Objects.requireNonNull(key, "key is null");
        Entry entry = this.localStore.get(key);
        byte[] result = null;
        if (entry != null && entry.getValue() != null && !this.isExpired(entry)) {
            result = Arrays.copyOf(entry.getValue(), entry.getValue().length);
        }
        return result;
    }

    public void delete(byte[] key) {
        Objects.requireNonNull(key, "key is null");
        long now = this.timeSupplier.get().toInstant().toEpochMilli();
        Entry entry = new Entry(key, null, new Version(now), now, null);
        this.localStore.put(entry);
        this.remoteStore.put(entry);
    }

    public Iterable<Entry> getAll() {
        return (Iterable)Streams.stream(this.localStore.getAll()).filter(entry -> !this.expired().test((Entry)entry) && !DistributedStore.tombstone().test((Entry)entry)).collect(ImmutableList.toImmutableList());
    }

    private Predicate<? super Entry> expired() {
        return this::isExpired;
    }

    private static Predicate<? super Entry> tombstone() {
        return entry -> entry.getValue() == null;
    }
}

