/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.airlift.stats.cardinality;

import com.facebook.airlift.stats.cardinality.Bitmap;
import com.facebook.airlift.stats.cardinality.Format;
import com.facebook.airlift.stats.cardinality.RandomizationStrategy;
import com.facebook.airlift.stats.cardinality.SecureRandomizationStrategy;
import com.facebook.airlift.stats.cardinality.Utils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Murmur3Hash128;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import javax.annotation.concurrent.NotThreadSafe;
import org.openjdk.jol.info.ClassLayout;

@NotThreadSafe
public class SfmSketch {
    public static final double NON_PRIVATE_EPSILON = Double.POSITIVE_INFINITY;
    private static final int MAX_ESTIMATION_ITERATIONS = 1000;
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(SfmSketch.class).instanceSize();
    private final int indexBitLength;
    private final int precision;
    private final RandomizationStrategy randomizationStrategy;
    private double randomizedResponseProbability;
    private Bitmap bitmap;

    private SfmSketch(Bitmap bitmap, int indexBitLength, int precision, double randomizedResponseProbability, RandomizationStrategy randomizationStrategy) {
        SfmSketch.validatePrefixLength(indexBitLength);
        SfmSketch.validatePrecision(precision, indexBitLength);
        SfmSketch.validateRandomizedResponseProbability(randomizedResponseProbability);
        this.bitmap = bitmap;
        this.indexBitLength = indexBitLength;
        this.precision = precision;
        this.randomizedResponseProbability = randomizedResponseProbability;
        this.randomizationStrategy = randomizationStrategy;
    }

    public static SfmSketch create(int numberOfBuckets, int precision) {
        return SfmSketch.create(numberOfBuckets, precision, new SecureRandomizationStrategy());
    }

    public static SfmSketch create(int numberOfBuckets, int precision, RandomizationStrategy randomizationStrategy) {
        double randomizedResponseProbability = SfmSketch.getRandomizedResponseProbability(Double.POSITIVE_INFINITY);
        int indexBitLength = Utils.indexBitLength(numberOfBuckets);
        Bitmap bitmap = new Bitmap(numberOfBuckets * precision);
        return new SfmSketch(bitmap, indexBitLength, precision, randomizedResponseProbability, randomizationStrategy);
    }

    public static SfmSketch deserialize(Slice serialized) {
        return SfmSketch.deserialize(serialized, new SecureRandomizationStrategy());
    }

    public static SfmSketch deserialize(Slice serialized, RandomizationStrategy randomizationStrategy) {
        BasicSliceInput input = serialized.getInput();
        byte format = input.readByte();
        Preconditions.checkArgument((format == Format.SFM_V1.getTag() ? 1 : 0) != 0, (Object)"Wrong format tag");
        int indexBitLength = input.readInt();
        int precision = input.readInt();
        double randomizedResponseProbability = input.readDouble();
        Bitmap bitmap = Bitmap.fromSliceInput((SliceInput)input, Utils.numberOfBuckets(indexBitLength) * precision);
        return new SfmSketch(bitmap, indexBitLength, precision, randomizedResponseProbability, randomizationStrategy);
    }

    public void add(long value) {
        this.addHash(Murmur3Hash128.hash64((long)value));
    }

    public void add(Slice value) {
        this.addHash(Murmur3Hash128.hash64((Slice)value));
    }

    public void addHash(long hash) {
        int index = Utils.computeIndex(hash, this.indexBitLength);
        int zeros = Math.min(this.precision - 1, Utils.numberOfTrailingZeros(hash, this.indexBitLength));
        this.flipBitOn(index, zeros);
    }

    public long cardinality() {
        double guess = 1.0;
        double changeInGuess = Double.POSITIVE_INFINITY;
        for (int iterations = 0; Math.abs(changeInGuess) > 0.1 && iterations < 1000; ++iterations) {
            changeInGuess = -this.logLikelihoodFirstDerivative(guess) / this.logLikelihoodSecondDerivative(guess);
            guess += changeInGuess;
        }
        return Math.max(0L, Math.round(guess));
    }

    public void enablePrivacy(double epsilon) {
        Preconditions.checkArgument((!this.isPrivacyEnabled() ? 1 : 0) != 0, (Object)"sketch is already privacy-enabled");
        SfmSketch.validateEpsilon(epsilon);
        this.randomizedResponseProbability = SfmSketch.getRandomizedResponseProbability(epsilon);
        for (int i = 0; i < this.bitmap.length(); ++i) {
            this.bitmap.flipBit(i, this.randomizedResponseProbability, this.randomizationStrategy);
        }
    }

    public int estimatedSerializedSize() {
        return 17 + this.bitmap.byteLength() * 1;
    }

    private void flipBitOn(int bucket, int level) {
        Preconditions.checkArgument((!this.isPrivacyEnabled() ? 1 : 0) != 0, (Object)"privacy-enabled SfmSketch is immutable");
        int i = this.getBitLocation(bucket, level);
        this.bitmap.setBit(i, true);
    }

    @VisibleForTesting
    int getBitLocation(int bucket, int level) {
        return level * Utils.numberOfBuckets(this.indexBitLength) + bucket;
    }

    public Bitmap getBitmap() {
        return this.bitmap;
    }

    @VisibleForTesting
    double getOnProbability() {
        return 1.0 - this.randomizedResponseProbability;
    }

    static double getRandomizedResponseProbability(double epsilon) {
        if (epsilon == Double.POSITIVE_INFINITY) {
            return 0.0;
        }
        return 1.0 / (Math.exp(epsilon) + 1.0);
    }

    @VisibleForTesting
    double getRandomizedResponseProbability() {
        return this.randomizedResponseProbability;
    }

    public long getRetainedSizeInBytes() {
        return (long)INSTANCE_SIZE + this.bitmap.getRetainedSizeInBytes() + this.randomizationStrategy.getRetainedSizeInBytes();
    }

    public boolean isPrivacyEnabled() {
        return this.getRandomizedResponseProbability() > 0.0;
    }

    private double logLikelihoodFirstDerivative(double n) {
        double result = 0.0;
        for (int level = 0; level < this.precision; ++level) {
            double termOn = this.logLikelihoodTermFirstDerivative(level, true, n);
            double termOff = this.logLikelihoodTermFirstDerivative(level, false, n);
            for (int bucket = 0; bucket < Utils.numberOfBuckets(this.indexBitLength); ++bucket) {
                result += this.bitmap.getBit(this.getBitLocation(bucket, level)) ? termOn : termOff;
            }
        }
        return result;
    }

    private double logLikelihoodTermFirstDerivative(int level, boolean on, double n) {
        double p = this.observationProbability(level);
        int sign = on ? -1 : 1;
        double c1 = on ? this.getOnProbability() : 1.0 - this.getOnProbability();
        double c2 = this.getOnProbability() - this.getRandomizedResponseProbability();
        return Math.log1p(-p) * (1.0 - c1 / (c1 + (double)sign * c2 * Math.pow(1.0 - p, n)));
    }

    private double logLikelihoodSecondDerivative(double n) {
        double result = 0.0;
        for (int level = 0; level < this.precision; ++level) {
            double termOn = this.logLikelihoodTermSecondDerivative(level, true, n);
            double termOff = this.logLikelihoodTermSecondDerivative(level, false, n);
            for (int bucket = 0; bucket < Utils.numberOfBuckets(this.indexBitLength); ++bucket) {
                result += this.bitmap.getBit(this.getBitLocation(bucket, level)) ? termOn : termOff;
            }
        }
        return result;
    }

    private double logLikelihoodTermSecondDerivative(int level, boolean on, double n) {
        double p = this.observationProbability(level);
        int sign = on ? -1 : 1;
        double c1 = on ? this.getOnProbability() : 1.0 - this.getOnProbability();
        double c2 = this.getOnProbability() - this.getRandomizedResponseProbability();
        return (double)sign * c1 * c2 * Math.pow(Math.log1p(-p), 2.0) * Math.pow(1.0 - p, n) * Math.pow(c1 + (double)sign * c2 * Math.pow(1.0 - p, n), -2.0);
    }

    @VisibleForTesting
    static double mergeRandomizedResponseProbabilities(double p1, double p2) {
        return (p1 + p2 - 3.0 * p1 * p2) / (1.0 - 2.0 * p1 * p2);
    }

    public void mergeWith(SfmSketch other) {
        Preconditions.checkArgument((this.precision == other.precision ? 1 : 0) != 0, (String)"cannot merge two SFM sketches with different precision: %s vs. %s", (int)this.precision, (int)other.precision);
        Preconditions.checkArgument((this.indexBitLength == other.indexBitLength ? 1 : 0) != 0, (String)"cannot merge two SFM sketches with different indexBitLength: %s vs. %s", (int)this.indexBitLength, (int)other.indexBitLength);
        if (!this.isPrivacyEnabled() && !other.isPrivacyEnabled()) {
            this.setBitmap(this.bitmap.or(other.getBitmap()));
        } else {
            double p1 = this.randomizedResponseProbability;
            double p2 = other.randomizedResponseProbability;
            double p = SfmSketch.mergeRandomizedResponseProbabilities(p1, p2);
            double normalizer = (1.0 - 2.0 * p) / ((1.0 - 2.0 * p1) * (1.0 - 2.0 * p2));
            for (int i = 0; i < this.bitmap.length(); ++i) {
                double bit1 = this.bitmap.getBit(i) ? 1.0 : 0.0;
                double bit2 = other.bitmap.getBit(i) ? 1.0 : 0.0;
                double x = 1.0 - 2.0 * p - normalizer * (1.0 - p1 - bit1) * (1.0 - p2 - bit2);
                double probability = p + normalizer * x;
                probability = Math.min(1.0, Math.max(0.0, probability));
                this.bitmap.setBit(i, this.randomizationStrategy.nextBoolean(probability));
            }
        }
        this.randomizedResponseProbability = SfmSketch.mergeRandomizedResponseProbabilities(this.randomizedResponseProbability, other.randomizedResponseProbability);
    }

    private double observationProbability(int level) {
        return Math.pow(2.0, -(level + 1)) / (double)Utils.numberOfBuckets(this.indexBitLength);
    }

    public Slice serialize() {
        int size = this.estimatedSerializedSize();
        DynamicSliceOutput output = new DynamicSliceOutput(size).appendByte((int)Format.SFM_V1.getTag()).appendInt(this.indexBitLength).appendInt(this.precision).appendDouble(this.randomizedResponseProbability).appendBytes(this.bitmap.toBytes());
        return output.slice();
    }

    @VisibleForTesting
    void setBitmap(Bitmap bitmap) {
        this.bitmap = bitmap;
    }

    private static void validateEpsilon(double epsilon) {
        Preconditions.checkArgument((epsilon > 0.0 ? 1 : 0) != 0, (Object)"epsilon must be greater than zero or equal to NON_PRIVATE_EPSILON");
    }

    private static void validatePrecision(int precision, int indexBitLength) {
        Preconditions.checkArgument((precision > 0 && precision % 8 == 0 ? 1 : 0) != 0, (String)"precision must be a positive multiple of %s", (int)8);
        Preconditions.checkArgument((precision + indexBitLength <= 64 ? 1 : 0) != 0, (String)"precision + indexBitLength cannot exceed %s", (int)64);
    }

    private static void validatePrefixLength(int indexBitLength) {
        Preconditions.checkArgument((indexBitLength >= 1 && indexBitLength <= 32 ? 1 : 0) != 0, (Object)"indexBitLength is out of range");
    }

    private static void validateRandomizedResponseProbability(double p) {
        Preconditions.checkArgument((p >= 0.0 && p <= 0.5 ? 1 : 0) != 0, (Object)"randomizedResponseProbability should be in the interval [0, 0.5]");
    }
}

