/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.airlift.stats.cardinality;

import com.facebook.airlift.stats.cardinality.BiasCorrection;
import com.facebook.airlift.stats.cardinality.BucketListener;
import com.facebook.airlift.stats.cardinality.Format;
import com.facebook.airlift.stats.cardinality.HllInstance;
import com.facebook.airlift.stats.cardinality.SparseHll;
import com.facebook.airlift.stats.cardinality.Utils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Bytes;
import com.google.common.primitives.Ints;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import java.util.Arrays;
import java.util.HashSet;
import org.openjdk.jol.info.ClassLayout;

final class DenseHll
implements HllInstance {
    private static final double LINEAR_COUNTING_MIN_EMPTY_BUCKETS = 0.4;
    private static final int BITS_PER_BUCKET = 4;
    private static final int MAX_DELTA = 15;
    private static final int BUCKET_MASK = 15;
    private static final int DENSE_INSTANCE_SIZE = ClassLayout.parseClass(DenseHll.class).instanceSize();
    private static final int OVERFLOW_GROW_INCREMENT = 5;
    private final byte indexBitLength;
    private byte baseline;
    private int baselineCount;
    private final byte[] deltas;
    private int overflows;
    private int[] overflowBuckets;
    private byte[] overflowValues;

    public DenseHll(int indexBitLength) {
        DenseHll.validatePrefixLength(indexBitLength);
        int numberOfBuckets = Utils.numberOfBuckets(indexBitLength);
        this.indexBitLength = (byte)indexBitLength;
        this.baselineCount = numberOfBuckets;
        this.deltas = new byte[numberOfBuckets * 4 / 8];
        this.overflowBuckets = new int[0];
        this.overflowValues = new byte[0];
    }

    public DenseHll(Slice serialized) {
        int i;
        BasicSliceInput input = serialized.getInput();
        byte formatTag = input.readByte();
        Preconditions.checkArgument((formatTag == Format.DENSE_V1.getTag() || formatTag == Format.DENSE_V2.getTag() ? 1 : 0) != 0, (Object)"Invalid format tag");
        this.indexBitLength = input.readByte();
        DenseHll.validatePrefixLength(this.indexBitLength);
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        this.baseline = input.readByte();
        this.deltas = new byte[numberOfBuckets / 2];
        input.readBytes(this.deltas);
        if (formatTag == Format.DENSE_V1.getTag()) {
            short bucket = input.readShort();
            byte value = input.readByte();
            if (bucket >= 0 && value > 0) {
                Preconditions.checkArgument((bucket <= numberOfBuckets ? 1 : 0) != 0, (Object)"Overflow bucket index is out of range");
                this.overflows = 1;
                this.overflowBuckets = new int[]{bucket};
                this.overflowValues = new byte[]{value};
            } else {
                this.overflows = 0;
                this.overflowBuckets = new int[0];
                this.overflowValues = new byte[0];
            }
        } else if (formatTag == Format.DENSE_V2.getTag()) {
            this.overflows = input.readUnsignedShort();
            Preconditions.checkArgument((this.overflows <= numberOfBuckets ? 1 : 0) != 0, (Object)"Overflow entries is greater than actual number of buckets (possibly corrupt input)");
            this.overflowBuckets = new int[this.overflows];
            this.overflowValues = new byte[this.overflows];
            for (i = 0; i < this.overflows; ++i) {
                this.overflowBuckets[i] = input.readUnsignedShort();
                Preconditions.checkArgument((this.overflowBuckets[i] <= numberOfBuckets ? 1 : 0) != 0, (Object)"Overflow bucket index is out of range");
            }
            for (i = 0; i < this.overflows; ++i) {
                this.overflowValues[i] = input.readByte();
                Preconditions.checkArgument((this.overflowValues[i] > 0 ? 1 : 0) != 0, (Object)"Overflow bucket value must be > 0");
            }
        } else {
            throw new IllegalArgumentException(String.format("Invalid format tag: %d", formatTag));
        }
        this.baselineCount = 0;
        for (i = 0; i < numberOfBuckets; ++i) {
            if (this.getDelta(i) != 0) continue;
            ++this.baselineCount;
        }
        Preconditions.checkArgument((!input.isReadable() ? 1 : 0) != 0, (Object)"input is too big");
    }

    public static boolean canDeserialize(Slice serialized) {
        byte formatTag = serialized.getByte(0);
        return formatTag == Format.DENSE_V1.getTag() || formatTag == Format.DENSE_V2.getTag();
    }

    @Override
    public void insertHash(long hash) {
        int index = Utils.computeIndex(hash, this.indexBitLength);
        int value = Utils.computeValue(hash, this.indexBitLength);
        this.insert(index, value);
    }

    @Override
    public int estimatedInMemorySize() {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOf((byte[])this.deltas) + SizeOf.sizeOf((int[])this.overflowBuckets) + SizeOf.sizeOf((byte[])this.overflowValues));
    }

    @Override
    public int getIndexBitLength() {
        return this.indexBitLength;
    }

    @Override
    public long cardinality() {
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        if (this.baseline == 0 && (double)this.baselineCount > 0.4 * (double)numberOfBuckets) {
            return Math.round(Utils.linearCounting(this.baselineCount, numberOfBuckets));
        }
        double sum = 0.0;
        for (int i = 0; i < numberOfBuckets; ++i) {
            int value = this.getValue(i);
            sum += 1.0 / (double)(1L << value);
        }
        double estimate = Utils.alpha(this.indexBitLength) * (double)numberOfBuckets * (double)numberOfBuckets / sum;
        estimate = this.correctBias(estimate);
        return Math.round(estimate);
    }

    private double correctBias(double rawEstimate) {
        double bias;
        double[] estimates = BiasCorrection.RAW_ESTIMATES[this.indexBitLength - 4];
        if (rawEstimate < estimates[0] || rawEstimate > estimates[estimates.length - 1]) {
            return rawEstimate;
        }
        double[] biases = BiasCorrection.BIAS[this.indexBitLength - 4];
        int position = this.search(rawEstimate, estimates);
        if (position >= 0) {
            bias = biases[position];
        } else {
            int insertionPoint = -(position + 1);
            double x0 = estimates[insertionPoint - 1];
            double y0 = biases[insertionPoint - 1];
            double x1 = estimates[insertionPoint];
            double y1 = biases[insertionPoint];
            bias = (rawEstimate - x0) * (y1 - y0) / (x1 - x0) + y0;
        }
        return rawEstimate - bias;
    }

    private int search(double rawEstimate, double[] estimateCurve) {
        int low = 0;
        int high = estimateCurve.length - 1;
        while (low <= high) {
            int middle = low + high >>> 1;
            double middleValue = estimateCurve[middle];
            if (rawEstimate > middleValue) {
                low = middle + 1;
                continue;
            }
            if (rawEstimate < middleValue) {
                high = middle - 1;
                continue;
            }
            return middle;
        }
        return -(low + 1);
    }

    public void insert(int bucket, int value) {
        int delta = value - this.baseline;
        int oldDelta = this.getDelta(bucket);
        if (delta <= oldDelta || oldDelta == 15 && delta <= oldDelta + this.getOverflow(bucket)) {
            return;
        }
        if (delta > 15) {
            byte overflow = (byte)(delta - 15);
            int overflowEntry = this.findOverflowEntry(bucket);
            if (overflowEntry != -1) {
                this.setOverflow(overflowEntry, overflow);
            } else {
                this.addOverflow(bucket, overflow);
            }
            delta = 15;
        }
        this.setDelta(bucket, delta);
        if (oldDelta == 0) {
            --this.baselineCount;
            this.adjustBaselineIfNeeded();
        }
    }

    @Override
    public Slice serialize() {
        int i;
        int size = this.estimatedSerializedSize();
        DynamicSliceOutput output = new DynamicSliceOutput(size).appendByte((int)Format.DENSE_V2.getTag()).appendByte((int)this.indexBitLength).appendByte((int)this.baseline).appendBytes(this.deltas).appendShort(this.overflows);
        this.sortOverflows();
        for (i = 0; i < this.overflows; ++i) {
            output.appendShort(this.overflowBuckets[i]);
        }
        for (i = 0; i < this.overflows; ++i) {
            output.appendByte((int)this.overflowValues[i]);
        }
        return output.slice();
    }

    private void sortOverflows() {
        for (int i = 1; i < this.overflows; ++i) {
            for (int j = i; j > 0 && this.overflowBuckets[j - 1] > this.overflowBuckets[j]; --j) {
                int bucket = this.overflowBuckets[j];
                byte value = this.overflowValues[j];
                this.overflowBuckets[j] = this.overflowBuckets[j - 1];
                this.overflowValues[j] = this.overflowValues[j - 1];
                this.overflowBuckets[j - 1] = bucket;
                this.overflowValues[j - 1] = value;
            }
        }
    }

    @Override
    public DenseHll toDense() {
        return this;
    }

    @Override
    public void eachBucket(BucketListener listener) {
        for (int i = 0; i < Utils.numberOfBuckets(this.indexBitLength); ++i) {
            listener.visit(i, this.getValue(i));
        }
    }

    @Override
    public int estimatedSerializedSize() {
        return 3 + Utils.numberOfBuckets(this.indexBitLength) * 1 / 2 + 2 + 2 * this.overflows + 1 * this.overflows;
    }

    private void setDelta(int bucket, int value) {
        int slot = DenseHll.bucketToSlot(bucket);
        byte clearMask = (byte)(15 << DenseHll.shiftForBucket(bucket));
        int n = slot;
        this.deltas[n] = (byte)(this.deltas[n] & ~clearMask);
        byte setMask = (byte)(value << DenseHll.shiftForBucket(bucket));
        int n2 = slot;
        this.deltas[n2] = (byte)(this.deltas[n2] | setMask);
    }

    private int getDelta(int bucket) {
        int slot = DenseHll.bucketToSlot(bucket);
        return this.deltas[slot] >> DenseHll.shiftForBucket(bucket) & 0xF;
    }

    @VisibleForTesting
    int getValue(int bucket) {
        int delta = this.getDelta(bucket);
        if (delta == 15) {
            delta += this.getOverflow(bucket);
        }
        return this.baseline + delta;
    }

    private void adjustBaselineIfNeeded() {
        while (this.baselineCount == 0) {
            this.baseline = (byte)(this.baseline + 1);
            for (int bucket = 0; bucket < Utils.numberOfBuckets(this.indexBitLength); ++bucket) {
                int delta = this.getDelta(bucket);
                boolean hasOverflow = false;
                if (delta == 15) {
                    for (int i = 0; i < this.overflows; ++i) {
                        if (this.overflowBuckets[i] != bucket) continue;
                        hasOverflow = true;
                        int n = i;
                        this.overflowValues[n] = (byte)(this.overflowValues[n] - 1);
                        if (this.overflowValues[i] != 0) break;
                        int lastEntry = this.overflows - 1;
                        if (i < lastEntry) {
                            this.overflowBuckets[i] = this.overflowBuckets[lastEntry];
                            this.overflowValues[i] = this.overflowValues[lastEntry];
                            this.overflowBuckets[lastEntry] = -1;
                            this.overflowValues[lastEntry] = 0;
                        }
                        --this.overflows;
                        break;
                    }
                }
                if (!hasOverflow) {
                    this.setDelta(bucket, --delta);
                }
                if (delta != 0) continue;
                ++this.baselineCount;
            }
        }
    }

    public DenseHll mergeWith(DenseHll other) {
        if (this.indexBitLength != other.indexBitLength) {
            throw new IllegalArgumentException(String.format("Cannot merge HLLs with different number of buckets: %s vs %s", Utils.numberOfBuckets(this.indexBitLength), Utils.numberOfBuckets(other.indexBitLength)));
        }
        int newBaseline = Math.max(this.baseline, other.baseline);
        int baselineCount = 0;
        int bucket = 0;
        for (int i = 0; i < this.deltas.length; ++i) {
            int newSlot = 0;
            byte slot1 = this.deltas[i];
            byte slot2 = other.deltas[i];
            for (int shift = 4; shift >= 0; shift -= 4) {
                int newValue;
                int newDelta;
                int delta1 = slot1 >>> shift & 0xF;
                int delta2 = slot2 >>> shift & 0xF;
                int value1 = this.baseline + delta1;
                int value2 = other.baseline + delta2;
                int overflowEntry = -1;
                if (delta1 == 15 && (overflowEntry = this.findOverflowEntry(bucket)) != -1) {
                    value1 += this.overflowValues[overflowEntry];
                }
                if (delta2 == 15) {
                    value2 += other.getOverflow(bucket);
                }
                if ((newDelta = (newValue = Math.max(value1, value2)) - newBaseline) == 0) {
                    ++baselineCount;
                }
                newDelta = this.updateOverflow(bucket, overflowEntry, newDelta);
                newSlot <<= 4;
                newSlot |= newDelta;
                ++bucket;
            }
            this.deltas[i] = (byte)newSlot;
        }
        this.baseline = (byte)newBaseline;
        this.baselineCount = baselineCount;
        this.adjustBaselineIfNeeded();
        return this;
    }

    public DenseHll mergeWith(SparseHll other) {
        if (this.indexBitLength != other.getIndexBitLength()) {
            throw new IllegalArgumentException(String.format("Cannot merge HLLs with different number of buckets: %s vs %s", Utils.numberOfBuckets(this.indexBitLength), Utils.numberOfBuckets(other.getIndexBitLength())));
        }
        other.eachBucket(this::insert);
        return this;
    }

    private int findOverflowEntry(int bucket) {
        for (int i = 0; i < this.overflows; ++i) {
            if (this.overflowBuckets[i] != bucket) continue;
            return i;
        }
        return -1;
    }

    private int getOverflow(int bucket) {
        for (int i = 0; i < this.overflows; ++i) {
            if (this.overflowBuckets[i] != bucket) continue;
            return this.overflowValues[i];
        }
        return 0;
    }

    private int updateOverflow(int bucket, int overflowEntry, int delta) {
        if (delta > 15) {
            if (overflowEntry != -1) {
                this.setOverflow(overflowEntry, (byte)(delta - 15));
            } else {
                this.addOverflow(bucket, (byte)(delta - 15));
            }
            delta = 15;
        } else if (overflowEntry != -1) {
            this.removeOverflow(overflowEntry);
        }
        return delta;
    }

    private void setOverflow(int overflowEntry, byte overflow) {
        this.overflowValues[overflowEntry] = overflow;
    }

    private void removeOverflow(int overflowEntry) {
        this.overflowBuckets[overflowEntry] = this.overflowBuckets[this.overflows - 1];
        this.overflowValues[overflowEntry] = this.overflowValues[this.overflows - 1];
        --this.overflows;
    }

    private void addOverflow(int bucket, byte overflow) {
        this.overflowBuckets = Ints.ensureCapacity((int[])this.overflowBuckets, (int)(this.overflows + 1), (int)5);
        this.overflowValues = Bytes.ensureCapacity((byte[])this.overflowValues, (int)(this.overflows + 1), (int)5);
        this.overflowBuckets[this.overflows] = bucket;
        this.overflowValues[this.overflows] = overflow;
        ++this.overflows;
    }

    public static int estimatedInMemorySize(int indexBitLength) {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOfByteArray((int)(Utils.numberOfBuckets(indexBitLength) / 2)));
    }

    private static int bucketToSlot(int bucket) {
        return bucket >> 1;
    }

    private static int shiftForBucket(int bucket) {
        return (~bucket & 1) << 2;
    }

    private static void validatePrefixLength(int indexBitLength) {
        Preconditions.checkArgument((indexBitLength >= 1 && indexBitLength <= 16 ? 1 : 0) != 0, (Object)"indexBitLength is out of range");
    }

    @Override
    public void verify() {
        int zeroDeltas = 0;
        for (int i = 0; i < Utils.numberOfBuckets(this.indexBitLength); ++i) {
            if (this.getDelta(i) != 0) continue;
            ++zeroDeltas;
        }
        Preconditions.checkState((zeroDeltas == this.baselineCount ? 1 : 0) != 0, (String)"baselineCount (%s) doesn't match number of zero deltas (%s)", (int)this.baselineCount, (int)zeroDeltas);
        HashSet<Integer> overflows = new HashSet<Integer>();
        for (int i = 0; i < this.overflows; ++i) {
            int bucket = this.overflowBuckets[i];
            overflows.add(bucket);
            Preconditions.checkState((this.overflowValues[i] > 0 ? 1 : 0) != 0, (String)"Overflow at %s for bucket %s is 0", (int)i, (int)bucket);
            Preconditions.checkState((this.getDelta(bucket) == 15 ? 1 : 0) != 0, (String)"delta in bucket %s is less than MAX_DELTA (%s < %s) even though there's an associated overflow entry", (Object)bucket, (Object)this.getDelta(bucket), (Object)15);
        }
        Preconditions.checkState((overflows.size() == this.overflows ? 1 : 0) != 0, (String)"Duplicate overflow buckets: %s", (Object)Ints.asList((int[])Arrays.copyOf(this.overflowBuckets, this.overflows)));
    }
}

