/*
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to use,
 * copy, modify, and distribute this software in source code or binary form for use
 * in connection with the web services and APIs provided by Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use of
 * this software is subject to the Facebook Developer Principles and Policies
 * [http://developers.facebook.com/policy/]. This copyright notice shall be
 * included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.facebook.marketing.internal;

import android.content.Context;
import android.view.View;
import android.widget.AdapterView;
import androidx.annotation.Nullable;
import com.facebook.FacebookSdk;
import com.facebook.appevents.codeless.internal.ViewHierarchy;

public class ButtonIndexingEventListener {
  private static final String TAG = ButtonIndexingEventListener.class.getCanonicalName();

  public static ButtonIndexingOnClickListener getOnClickListener(View hostView, String mapKey) {
    return new ButtonIndexingOnClickListener(hostView, mapKey);
  }

  public static ButtonIndexingOnItemClickListener getOnItemClickListener(
      AdapterView hostView, String mapKey) {
    return new ButtonIndexingOnItemClickListener(hostView, mapKey);
  }

  public static class ButtonIndexingOnClickListener implements View.OnClickListener {

    public ButtonIndexingOnClickListener(final View hostView, final String mapKey) {
      if (null == hostView) {
        return;
      }
      this.existingOnClickListener = ViewHierarchy.getExistingOnClickListener(hostView);
      this.mapKey = mapKey;
      this.supportButtonIndexing = true;
    }

    @Override
    public void onClick(final View host) {
      if (null != this.existingOnClickListener) {
        this.existingOnClickListener.onClick(host);
      }
      logIndexing(host, this.mapKey);
    }

    public boolean getSupportButtonIndexing() {
      return supportButtonIndexing;
    }

    @Nullable private View.OnClickListener existingOnClickListener;
    private String mapKey;
    private boolean supportButtonIndexing = false;
  }

  public static class ButtonIndexingOnItemClickListener implements AdapterView.OnItemClickListener {

    public ButtonIndexingOnItemClickListener(final AdapterView hostView, final String mapKey) {
      if (null == hostView) {
        return;
      }
      this.existingOnClickListener = hostView.getOnItemClickListener();
      this.mapKey = mapKey;
      this.supportButtonIndexing = true;
    }

    @Override
    public void onItemClick(AdapterView<?> parent, final View host, int position, long id) {
      if (null != this.existingOnClickListener) {
        this.existingOnClickListener.onItemClick(parent, host, position, id);
      }
      logIndexing(host, this.mapKey);
    }

    public boolean getSupportButtonIndexing() {
      return supportButtonIndexing;
    }

    @Nullable private AdapterView.OnItemClickListener existingOnClickListener;
    private String mapKey;
    private boolean supportButtonIndexing = false;
  }

  private static void logIndexing(final View view, final String mapKey) {
    FacebookSdk.getExecutor()
        .execute(
            new Runnable() {
              @Override
              public void run() {
                final Context context = FacebookSdk.getApplicationContext();
                final String applicationId = FacebookSdk.getApplicationId();
                ButtonIndexingLogger.logIndexing(applicationId, view, mapKey, context);
              }
            });
  }
}
