/*
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to use,
 * copy, modify, and distribute this software in source code or binary form for use
 * in connection with the web services and APIs provided by Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use of
 * this software is subject to the Facebook Developer Principles and Policies
 * [http://developers.facebook.com/policy/]. This copyright notice shall be
 * included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.facebook.marketing.internal;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.os.Bundle;
import android.util.Base64;
import android.util.Log;
import android.view.View;
import android.widget.ImageView;
import com.facebook.AccessToken;
import com.facebook.FacebookSdk;
import com.facebook.GraphRequest;
import com.facebook.appevents.codeless.ViewIndexer;
import com.facebook.appevents.codeless.internal.Constants;
import com.facebook.appevents.codeless.internal.SensitiveUserDataUtils;
import com.facebook.appevents.codeless.internal.ViewHierarchy;
import com.facebook.appevents.internal.AppEventUtility;
import com.facebook.internal.AttributionIdentifiers;
import com.facebook.internal.Utility;
import java.io.ByteArrayOutputStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class ButtonIndexingLogger {
  private static final String TAG = ButtonIndexingLogger.class.getCanonicalName();
  private static final String API_ENDPOINT = "%s/button_indexing";
  private static volatile Set<String> clickedKeySet = new HashSet<>();
  private static volatile Set<String> loadedKeySet = new HashSet<>();

  private static JSONObject generateButtonDetail(View view, String mapKey, boolean isFromClick) {
    try {
      JSONObject result = new JSONObject();
      JSONArray path = new JSONArray();
      String[] indexList = mapKey.split("\\.", -1);
      View currentView = view;
      int i = indexList.length - 1;
      while (currentView != null) {
        JSONObject currentPath = new JSONObject();
        currentPath.put("classname", currentView.getClass().getCanonicalName());
        currentPath.put("index", indexList[i]);
        currentPath.put("id", currentView.getId());
        String text =
            SensitiveUserDataUtils.isSensitiveUserData(currentView)
                ? ""
                : ViewHierarchy.getTextOfView(currentView);
        currentPath.put("text", text);
        currentPath.put(
            "tag", currentView.getTag() == null ? "" : String.valueOf(currentView.getTag()));
        currentPath.put(
            "description",
            currentView.getContentDescription() == null
                ? ""
                : String.valueOf(currentView.getContentDescription()));
        path.put(currentPath);
        currentView = ViewHierarchy.getParentOfView(currentView);
        i--;
      }
      JSONArray revertedPath = new JSONArray();
      for (int j = path.length() - 1; j >= 0; j--) {
        revertedPath.put(path.get(j));
      }
      result.put("path", revertedPath);
      result.put("is_from_click", isFromClick);
      if (view instanceof ImageView) {
        Bitmap bitmap = ((BitmapDrawable) ((ImageView) view).getDrawable()).getBitmap();
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, byteArrayOutputStream);
        byte[] byteArray = byteArrayOutputStream.toByteArray();
        String encoded = Base64.encodeToString(byteArray, Base64.DEFAULT);
        result.put("image", encoded);
      }
      return result;
    } catch (Exception e) {
      Log.e(TAG, "Log button indexing error", e);
      return new JSONObject();
    }
  }

  private static void sendGraphAPIRequest(Context context, String paths, String applicationId) {
    AttributionIdentifiers identifiers = AttributionIdentifiers.getAttributionIdentifiers(context);
    if (identifiers != null && identifiers.getAndroidAdvertiserId() != null) {
      String version = AppEventUtility.getAppVersion();
      Bundle publishParams = new Bundle();
      try {
        publishParams.putString("app_version", version);
        publishParams.putString("indexed_button_list", paths);
        final GraphRequest postRequest =
            GraphRequest.newPostRequest(
                null, String.format(Locale.US, API_ENDPOINT, applicationId), null, null);
        postRequest.setParameters(publishParams);
        postRequest.executeAndWait();
      } catch (Exception e) {
        Log.e(TAG, "failed to send button indexing request", e);
      }
    }
  }

  public static void logIndexing(String applicationId, View view, String mapKey, Context context) {
    if (clickedKeySet.contains(mapKey)) {
      return;
    }
    clickedKeySet.add(mapKey);
    JSONObject detail = generateButtonDetail(view, mapKey, true);
    if (detail.length() > 0) {
      JSONArray paths = new JSONArray(Arrays.asList(detail.toString()));
      sendGraphAPIRequest(context, paths.toString(), applicationId);
    }
  }

  public static void logAllIndexing(final JSONObject tree, final String activityName) {
    FacebookSdk.getExecutor()
        .execute(
            new Runnable() {
              @Override
              public void run() {
                JSONObject viewTree = new JSONObject();
                try {
                  final String applicationId = FacebookSdk.getApplicationId();
                  final AccessToken accessToken = AccessToken.getCurrentAccessToken();
                  viewTree.put("screenname", activityName);
                  JSONArray viewArray = new JSONArray();
                  viewArray.put(tree);
                  viewTree.put("view", viewArray);
                  GraphRequest request =
                      ViewIndexer.buildAppIndexingRequest(
                          viewTree.toString(),
                          accessToken,
                          applicationId,
                          Constants.BUTTON_SAMPLING);
                  if (request != null) {
                    request.executeAndWait();
                  }
                } catch (JSONException e) {
                  Utility.logd(TAG, e);
                }
              }
            });
  }
}
