/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.stats;

import com.facebook.stats.Clock;
import com.facebook.stats.RealtimeClock;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.Ordering;
import com.google.common.collect.PeekingIterator;
import com.google.common.util.concurrent.AtomicDouble;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class QuantileDigest {
    private static final int MAX_BITS = 64;
    private static final double MAX_SIZE_FACTOR = 1.5;
    static final long RESCALE_THRESHOLD_SECONDS = 50L;
    static final double ZERO_WEIGHT_THRESHOLD = 1.0E-5;
    private final double maxError;
    private final Clock clock;
    private final double alpha;
    private final boolean compressAutomatically;
    private Node root;
    private double weightedCount;
    private long max;
    private long min = Long.MAX_VALUE;
    private long landmarkInSeconds;
    private int totalNodeCount = 0;
    private int nonZeroNodeCount = 0;
    private int compressions = 0;
    private int maxTotalNodeCount = 0;
    private int maxTotalNodesAfterCompress = 0;

    public QuantileDigest(double maxError) {
        this(maxError, 0.0);
    }

    public QuantileDigest(double maxError, double alpha) {
        this(maxError, alpha, new RealtimeClock(), true);
    }

    @VisibleForTesting
    QuantileDigest(double maxError, double alpha, Clock clock, boolean compressAutomatically) {
        Preconditions.checkArgument((maxError >= 0.0 && maxError <= 1.0 ? 1 : 0) != 0, (Object)"maxError must be in range [0, 1]");
        Preconditions.checkArgument((alpha >= 0.0 && alpha < 1.0 ? 1 : 0) != 0, (Object)"alpha must be in range [0, 1)");
        this.maxError = maxError;
        this.alpha = alpha;
        this.clock = clock;
        this.compressAutomatically = compressAutomatically;
        this.landmarkInSeconds = TimeUnit.MILLISECONDS.toSeconds(clock.getMillis());
    }

    public synchronized void add(long value) {
        Preconditions.checkArgument((value >= 0L ? 1 : 0) != 0, (Object)"value must be >= 0");
        long nowInSeconds = TimeUnit.MILLISECONDS.toSeconds(this.clock.getMillis());
        int maxExpectedNodeCount = 3 * this.calculateCompressionFactor();
        if (nowInSeconds - this.landmarkInSeconds >= 50L) {
            this.rescale(nowInSeconds);
            this.compress();
        } else if ((double)this.nonZeroNodeCount > 1.5 * (double)maxExpectedNodeCount && this.compressAutomatically) {
            this.compress();
        }
        double weight = this.weight(TimeUnit.MILLISECONDS.toSeconds(this.clock.getMillis()));
        this.weightedCount += weight;
        this.max = Math.max(this.max, value);
        this.min = Math.min(this.min, value);
        this.insert(value, weight);
    }

    public synchronized List<Long> getQuantiles(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(quantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum = 0.0;

            @Override
            public boolean process(Node node) {
                this.sum += node.weightedCount;
                while (iterator.hasNext() && this.sum > (Double)iterator.peek() * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.min(node.getUpperBound(), QuantileDigest.this.max);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            builder.add((Object)this.max);
            iterator.next();
        }
        return builder.build();
    }

    public synchronized long getQuantile(double quantile) {
        return this.getQuantiles((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public synchronized double getCount() {
        return this.weightedCount / this.weight(TimeUnit.MILLISECONDS.toSeconds(this.clock.getMillis()));
    }

    public synchronized List<Bucket> getHistogram(List<Long> bucketUpperBounds) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(bucketUpperBounds), (Object)"buckets must be sorted in increasing order");
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(bucketUpperBounds.iterator());
        final AtomicDouble sum = new AtomicDouble();
        final AtomicDouble lastSum = new AtomicDouble();
        final AtomicDouble bucketWeightedSum = new AtomicDouble();
        final double normalizationFactor = this.weight(TimeUnit.MILLISECONDS.toSeconds(this.clock.getMillis()));
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                while (iterator.hasNext() && (Long)iterator.peek() <= node.getUpperBound()) {
                    double bucketCount = sum.get() - lastSum.get();
                    Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
                    builder.add((Object)bucket);
                    lastSum.set(sum.get());
                    bucketWeightedSum.set(0.0);
                    iterator.next();
                }
                bucketWeightedSum.addAndGet((double)node.getMiddle() * node.weightedCount);
                sum.addAndGet(node.weightedCount);
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            double bucketCount = sum.get() - lastSum.get();
            Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
            builder.add((Object)bucket);
            iterator.next();
        }
        return builder.build();
    }

    public long getMin() {
        final AtomicLong chosen = new AtomicLong(this.min);
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node.weightedCount >= 1.0E-5) {
                    chosen.set(node.getLowerBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.FORWARD);
        return Math.max(this.min, chosen.get());
    }

    public long getMax() {
        final AtomicLong chosen = new AtomicLong(this.max);
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node.weightedCount >= 1.0E-5) {
                    chosen.set(node.getUpperBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.REVERSE);
        return Math.min(this.max, chosen.get());
    }

    @VisibleForTesting
    synchronized int getTotalNodeCount() {
        return this.totalNodeCount;
    }

    @VisibleForTesting
    synchronized int getNonZeroNodeCount() {
        return this.nonZeroNodeCount;
    }

    @VisibleForTesting
    synchronized int getCompressions() {
        return this.compressions;
    }

    @VisibleForTesting
    synchronized void compress() {
        ++this.compressions;
        final int compressionFactor = this.calculateCompressionFactor();
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                Node node2;
                if (node.isLeaf()) {
                    return true;
                }
                double leftWeight = 0.0;
                if (node.left != null) {
                    leftWeight = node.left.weightedCount;
                }
                double rightWeight = 0.0;
                if (node.right != null) {
                    rightWeight = node.right.weightedCount;
                }
                boolean shouldCompress = node.weightedCount + leftWeight + rightWeight < QuantileDigest.this.weightedCount / (double)compressionFactor;
                double oldNodeWeight = node.weightedCount;
                if (shouldCompress || leftWeight < 1.0E-5) {
                    node.left = QuantileDigest.this.tryRemove(node.left);
                    QuantileDigest.this.weightedCount = QuantileDigest.this.weightedCount + leftWeight;
                    node2 = node;
                    node2.weightedCount = node2.weightedCount + leftWeight;
                }
                if (shouldCompress || rightWeight < 1.0E-5) {
                    node.right = QuantileDigest.this.tryRemove(node.right);
                    QuantileDigest.this.weightedCount = QuantileDigest.this.weightedCount + rightWeight;
                    node2 = node;
                    node2.weightedCount = node2.weightedCount + rightWeight;
                }
                if (oldNodeWeight < 1.0E-5 && node.weightedCount >= 1.0E-5) {
                    ++QuantileDigest.this.nonZeroNodeCount;
                }
                return true;
            }
        });
        if (this.root != null && this.root.weightedCount < 1.0E-5) {
            this.root = this.tryRemove(this.root);
        }
        this.maxTotalNodesAfterCompress = Math.max(this.maxTotalNodesAfterCompress, this.totalNodeCount);
    }

    private double weight(long timestamp) {
        return Math.exp(this.alpha * (double)(timestamp - this.landmarkInSeconds));
    }

    private void rescale(long newLandmarkInSeconds) {
        final double factor = Math.exp(-this.alpha * (double)(newLandmarkInSeconds - this.landmarkInSeconds));
        this.weightedCount *= factor;
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                double oldWeight = node.weightedCount;
                Node node2 = node;
                node2.weightedCount = node2.weightedCount * factor;
                if (oldWeight >= 1.0E-5 && node.weightedCount < 1.0E-5) {
                    --QuantileDigest.this.nonZeroNodeCount;
                }
                return true;
            }
        });
        this.landmarkInSeconds = newLandmarkInSeconds;
    }

    private int calculateCompressionFactor() {
        if (this.root == null) {
            return 1;
        }
        return Math.max((int)((double)(this.root.level + 1) / this.maxError), 1);
    }

    private void insert(long value, double weight) {
        long lastBranch = 0L;
        Node parent = null;
        Node current = this.root;
        while (true) {
            if (current == null) {
                this.setChild(parent, lastBranch, this.createLeaf(value, weight));
                return;
            }
            if (value >>> current.level != current.value >>> current.level) {
                this.setChild(parent, lastBranch, this.makeSiblings(current, this.createLeaf(value, weight)));
                return;
            }
            if (current.level == 0 && current.value == value) {
                double oldWeight = current.weightedCount;
                Node node = current;
                node.weightedCount = node.weightedCount + weight;
                if (current.weightedCount >= 1.0E-5 && oldWeight < 1.0E-5) {
                    ++this.nonZeroNodeCount;
                }
                return;
            }
            long branch = value & current.getBranchMask();
            parent = current;
            lastBranch = branch;
            if (branch == 0L) {
                current = current.left;
                continue;
            }
            current = current.right;
        }
    }

    private void setChild(Node parent, long branch, Node child) {
        if (parent == null) {
            this.root = child;
        } else if (branch == 0L) {
            parent.left = child;
        } else {
            parent.right = child;
        }
    }

    private Node makeSiblings(Node node, Node sibling) {
        int parentLevel = 64 - Long.numberOfLeadingZeros(node.value ^ sibling.value);
        Node parent = new Node(node.value, parentLevel, 0.0);
        long branch = sibling.value & parent.getBranchMask();
        if (branch == 0L) {
            parent.left = sibling;
            parent.right = node;
        } else {
            parent.left = node;
            parent.right = sibling;
        }
        ++this.totalNodeCount;
        this.maxTotalNodeCount = Math.max(this.maxTotalNodeCount, this.totalNodeCount);
        return parent;
    }

    private Node createLeaf(long value, double weight) {
        ++this.totalNodeCount;
        this.maxTotalNodeCount = Math.max(this.maxTotalNodeCount, this.totalNodeCount);
        ++this.nonZeroNodeCount;
        return new Node(value, 0, weight);
    }

    private Node tryRemove(Node node) {
        if (node == null) {
            return null;
        }
        if (node.weightedCount >= 1.0E-5) {
            --this.nonZeroNodeCount;
        }
        this.weightedCount -= node.weightedCount;
        Node result = null;
        if (node.isLeaf()) {
            --this.totalNodeCount;
        } else if (node.hasSingleChild()) {
            result = node.getSingleChild();
            --this.totalNodeCount;
        } else {
            node.weightedCount = 0.0;
            result = node;
        }
        return result;
    }

    private boolean postOrderTraversal(Node node, Callback callback) {
        return this.postOrderTraversal(node, callback, TraversalOrder.FORWARD);
    }

    private boolean postOrderTraversal(Node node, Callback callback, TraversalOrder order) {
        Node second;
        Node first;
        if (node == null) {
            return false;
        }
        if (order == TraversalOrder.FORWARD) {
            first = node.left;
            second = node.right;
        } else {
            first = node.right;
            second = node.left;
        }
        if (first != null && !this.postOrderTraversal(first, callback, order)) {
            return false;
        }
        if (second != null && !this.postOrderTraversal(second, callback, order)) {
            return false;
        }
        return callback.process(node);
    }

    public synchronized double getConfidenceFactor() {
        return this.computeMaxPathWeight(this.root) * 1.0 / this.weightedCount;
    }

    private double computeMaxPathWeight(Node node) {
        if (node == null || node.level == 0) {
            return 0.0;
        }
        double leftMaxWeight = this.computeMaxPathWeight(node.left);
        double rightMaxWeight = this.computeMaxPathWeight(node.right);
        return Math.max(leftMaxWeight, rightMaxWeight) + node.weightedCount;
    }

    @VisibleForTesting
    synchronized void validate() {
        final AtomicDouble sumOfWeights = new AtomicDouble();
        final AtomicInteger actualNodeCount = new AtomicInteger();
        final AtomicInteger actualNonZeroNodeCount = new AtomicInteger();
        if (this.root != null) {
            this.validateStructure(this.root);
            this.postOrderTraversal(this.root, new Callback(){

                @Override
                public boolean process(Node node) {
                    sumOfWeights.addAndGet(node.weightedCount);
                    actualNodeCount.incrementAndGet();
                    if (node.weightedCount > 1.0E-5) {
                        actualNonZeroNodeCount.incrementAndGet();
                    }
                    return true;
                }
            });
        }
        Preconditions.checkState((Math.abs(sumOfWeights.get() - this.weightedCount) < 1.0E-5 ? 1 : 0) != 0, (String)"Computed weight (%s) doesn't match summary (%s)", (Object[])new Object[]{sumOfWeights.get(), this.weightedCount});
        Preconditions.checkState((actualNodeCount.get() == this.totalNodeCount ? 1 : 0) != 0, (String)"Actual node count (%s) doesn't match summary (%s)", (Object[])new Object[]{actualNodeCount.get(), this.totalNodeCount});
        Preconditions.checkState((actualNonZeroNodeCount.get() == this.nonZeroNodeCount ? 1 : 0) != 0, (String)"Actual non-zero node count (%s) doesn't match summary (%s)", (Object[])new Object[]{actualNonZeroNodeCount.get(), this.nonZeroNodeCount});
    }

    private void validateStructure(Node node) {
        Preconditions.checkState((node.level >= 0 ? 1 : 0) != 0);
        if (node.left != null) {
            this.validateBranchStructure(node, node.left, node.right, true);
            this.validateStructure(node.left);
        }
        if (node.right != null) {
            this.validateBranchStructure(node, node.right, node.left, false);
            this.validateStructure(node.right);
        }
    }

    private void validateBranchStructure(Node parent, Node child, Node otherChild, boolean isLeft) {
        Preconditions.checkState((child.level < parent.level ? 1 : 0) != 0, (String)"Child level (%s) should be smaller than parent level (%s)", (Object[])new Object[]{child.level, parent.level});
        long branch = child.value & 1L << parent.level - 1;
        Preconditions.checkState((branch == 0L && isLeft || branch != 0L && !isLeft ? 1 : 0) != 0, (Object)"Value of child node is inconsistent with its branch");
        Preconditions.checkState((parent.weightedCount >= 1.0E-5 || child.weightedCount >= 1.0E-5 || otherChild != null ? 1 : 0) != 0, (Object)"Found a linear chain of zero-weight nodes");
    }

    private static interface Callback {
        public boolean process(Node var1);
    }

    private static class Node {
        private double weightedCount;
        private int level;
        private long value;
        private Node left;
        private Node right;

        private Node(long value, int level, double weightedCount) {
            this.value = value;
            this.level = level;
            this.weightedCount = weightedCount;
        }

        public boolean isLeaf() {
            return this.left == null && this.right == null;
        }

        public boolean hasSingleChild() {
            return this.left == null && this.right != null || this.left != null && this.right == null;
        }

        public Node getSingleChild() {
            Preconditions.checkState((boolean)this.hasSingleChild(), (Object)"Node does not have a single child");
            return (Node)Objects.firstNonNull((Object)this.left, (Object)this.right);
        }

        public long getUpperBound() {
            long mask = (1L << this.level) - 1L;
            return this.value | mask;
        }

        public long getBranchMask() {
            return 1L << this.level - 1;
        }

        public long getLowerBound() {
            long mask = Long.MAX_VALUE << this.level;
            return this.value & mask;
        }

        public long getMiddle() {
            return this.getLowerBound() + (this.getUpperBound() - this.getLowerBound()) / 2L;
        }

        public String toString() {
            return String.format("%s (level = %d, count = %s, left = %s, right = %s)", this.value, this.level, this.weightedCount, this.left != null, this.right != null);
        }
    }

    public static class Bucket {
        private double count;
        private double mean;

        public Bucket(double count, double mean) {
            this.count = count;
            this.mean = mean;
        }

        public double getCount() {
            return this.count;
        }

        public double getMean() {
            return this.mean;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bucket bucket = (Bucket)o;
            if (Double.compare(bucket.count, this.count) != 0) {
                return false;
            }
            return Double.compare(bucket.mean, this.mean) == 0;
        }

        public int hashCode() {
            long temp = this.count != 0.0 ? Double.doubleToLongBits(this.count) : 0L;
            int result = (int)(temp ^ temp >>> 32);
            temp = this.mean != 0.0 ? Double.doubleToLongBits(this.mean) : 0L;
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public String toString() {
            return String.format("[count: %f, mean: %f]", this.count, this.mean);
        }
    }

    private static enum TraversalOrder {
        FORWARD,
        REVERSE;

    }
}

