/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.stats.cardinality;

import com.facebook.stats.cardinality.Estimator;
import com.facebook.stats.cardinality.Numbers;
import com.facebook.stats.cardinality.UnsafeUtil;
import com.google.common.base.Preconditions;
import java.util.Arrays;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
class SparseEstimator
implements Estimator {
    private static final int BITS_PER_BUCKET = 4;
    private static final int BUCKET_VALUE_MASK = 15;
    public static final int MAX_BUCKET_VALUE = 16;
    private static final int INSTANCE_SIZE = UnsafeUtil.sizeOf(SparseEstimator.class);
    private final byte indexBits;
    private short bucketCount = 0;
    private long[] slots;

    SparseEstimator(int numberOfBuckets) {
        this(numberOfBuckets, 1);
    }

    SparseEstimator(int[] buckets) {
        this(buckets.length, SparseEstimator.countNonZeroBuckets(buckets));
        for (int bucket = 0; bucket < buckets.length; ++bucket) {
            int value = buckets[bucket];
            if (value == 0) continue;
            this.setEntry(this.bucketCount, bucket, value);
            this.bucketCount = (short)(this.bucketCount + 1);
        }
    }

    SparseEstimator(int numberOfBuckets, int initialCapacity) {
        Preconditions.checkArgument((boolean)Numbers.isPowerOf2(numberOfBuckets), (Object)"numberOfBuckets must be a power of 2");
        this.indexBits = (byte)Integer.numberOfTrailingZeros(numberOfBuckets);
        this.slots = new long[(initialCapacity + this.getBucketsPerSlot()) / this.getBucketsPerSlot()];
    }

    @Override
    public boolean setIfGreater(int bucket, int highestBitPosition) {
        Preconditions.checkArgument((highestBitPosition < 16 ? 1 : 0) != 0, (String)"highestBitPosition %s is bigger than allowed by BITS_PER_BUCKET (%s)", (Object[])new Object[]{highestBitPosition, 4});
        if (highestBitPosition == 0) {
            return false;
        }
        int index = this.findBucket(bucket);
        if (index < 0) {
            this.insertAt(-(index + 1), bucket, highestBitPosition);
            return true;
        }
        if (this.getEntry(index).getValue() < highestBitPosition) {
            this.setEntry(index, bucket, highestBitPosition);
            return true;
        }
        return false;
    }

    @Override
    public int[] buckets() {
        int[] buckets = new int[this.getNumberOfBuckets()];
        for (int i = 0; i < this.bucketCount; ++i) {
            Entry entry = this.getEntry(i);
            buckets[entry.getBucket()] = entry.getValue();
        }
        return buckets;
    }

    @Override
    public int getNumberOfBuckets() {
        return 1 << this.indexBits;
    }

    @Override
    public int getMaxAllowedBucketValue() {
        return 16;
    }

    private Entry getEntry(int index) {
        int totalBitsPerBucket = this.getTotalBitsPerBucket();
        int bucketMask = (1 << totalBitsPerBucket) - 1;
        int bucketsPerSlot = this.getBucketsPerSlot();
        int slot = index / bucketsPerSlot;
        int offset = index % bucketsPerSlot;
        int bucketEntry = (int)(this.slots[slot] >>> offset * totalBitsPerBucket & (long)bucketMask);
        return new Entry(bucketEntry >> 4, bucketEntry & 0xF);
    }

    private int getBucketsPerSlot() {
        return 64 / this.getTotalBitsPerBucket();
    }

    private int getTotalBitsPerBucket() {
        return this.indexBits + 4;
    }

    private void setEntry(int index, int bucket, int value) {
        int totalBitsPerBucket = this.getTotalBitsPerBucket();
        long bucketMask = (1L << totalBitsPerBucket) - 1L;
        int bucketsPerSlot = this.getBucketsPerSlot();
        int slot = index / bucketsPerSlot;
        int offset = index % bucketsPerSlot;
        long bucketEntry = bucket << 4 | value;
        long bucketClearMask = bucketMask << offset * totalBitsPerBucket;
        long bucketSetMask = bucketEntry << offset * totalBitsPerBucket;
        this.slots[slot] = this.slots[slot] & (bucketClearMask ^ 0xFFFFFFFFFFFFFFFFL) | bucketSetMask;
    }

    @Override
    public int estimateSizeInBytes() {
        return SparseEstimator.estimateSizeInBytes(this.bucketCount, this.getNumberOfBuckets());
    }

    public static int estimateSizeInBytes(int nonZeroBuckets, int totalBuckets) {
        Preconditions.checkArgument((boolean)Numbers.isPowerOf2(totalBuckets), (Object)"totalBuckets must be a power of 2");
        int bits = Integer.numberOfTrailingZeros(totalBuckets);
        int bucketsPerSlot = 64 / (bits + 4);
        return (nonZeroBuckets + bucketsPerSlot) / bucketsPerSlot * 64 / 8 + INSTANCE_SIZE;
    }

    @Override
    public long estimate() {
        int totalBuckets = this.getNumberOfBuckets();
        int zeroBuckets = totalBuckets - this.bucketCount;
        return Math.round((double)totalBuckets * Math.log((double)totalBuckets * 1.0 / (double)zeroBuckets));
    }

    private void grow() {
        this.slots = Arrays.copyOf(this.slots, this.slots.length + 1);
    }

    private int findBucket(int bucket) {
        int low = 0;
        int high = this.bucketCount - 1;
        while (low <= high) {
            int middle = low + high >>> 1;
            Entry middleBucket = this.getEntry(middle);
            if (bucket > middleBucket.getBucket()) {
                low = middle + 1;
                continue;
            }
            if (bucket < middleBucket.getBucket()) {
                high = middle - 1;
                continue;
            }
            return middle;
        }
        return -(low + 1);
    }

    private void insertAt(int index, int bucket, int value) {
        int totalBitsPerBucket = this.getTotalBitsPerBucket();
        int bucketsPerSlot = this.getBucketsPerSlot();
        this.bucketCount = (short)(this.bucketCount + 1);
        if ((this.bucketCount + bucketsPerSlot - 1) / bucketsPerSlot > this.slots.length) {
            this.grow();
        }
        int lastUsedSlot = (this.bucketCount - 1) / bucketsPerSlot;
        int insertAtSlot = index / bucketsPerSlot;
        int insertOffset = index % bucketsPerSlot;
        long bucketMask = (1L << totalBitsPerBucket) - 1L;
        for (int i = lastUsedSlot; i > insertAtSlot; --i) {
            int overflow = (int)(this.slots[i - 1] >>> (bucketsPerSlot - 1) * totalBitsPerBucket & bucketMask);
            this.slots[i] = this.slots[i] << totalBitsPerBucket | (long)overflow;
        }
        long old = this.slots[insertAtSlot];
        long bottomMask = (1L << insertOffset * totalBitsPerBucket) - 1L;
        long topMask = 0L;
        if (insertOffset < this.getBucketsPerSlot() - 1) {
            topMask = -1L << (insertOffset + 1) * totalBitsPerBucket;
        }
        long bucketSetMask = ((long)bucket << 4 | (long)value) << insertOffset * totalBitsPerBucket;
        this.slots[insertAtSlot] = old << totalBitsPerBucket & topMask | bucketSetMask | old & bottomMask;
    }

    private static int countNonZeroBuckets(int[] buckets) {
        int count = 0;
        for (int bucket : buckets) {
            if (bucket <= 0) continue;
            ++count;
        }
        return count;
    }

    private static class Entry {
        private final int bucket;
        private final int value;

        private Entry(int bucket, int value) {
            this.bucket = bucket;
            this.value = value;
        }

        public int getBucket() {
            return this.bucket;
        }

        public int getValue() {
            return this.value;
        }
    }
}

