/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.stats.cardinality;

import com.facebook.stats.cardinality.Estimator;
import com.facebook.stats.cardinality.HyperLogLogUtil;
import com.facebook.stats.cardinality.Numbers;
import com.facebook.stats.cardinality.UnsafeUtil;
import com.google.common.base.Preconditions;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
class DenseEstimator
implements Estimator {
    private static final int BITS_PER_BUCKET = 4;
    private static final int BUCKET_MAX_VALUE = 15;
    private static final int BUCKETS_PER_SLOT = 16;
    private static final long BUCKET_MASK = 15L;
    private static final int INSTANCE_SIZE = UnsafeUtil.sizeOf(DenseEstimator.class);
    private final int numberOfBuckets;
    private final long[] slots;
    private double currentSum;
    private byte baseline;
    private short baselineCount;

    public DenseEstimator(int numberOfBuckets) {
        Preconditions.checkArgument((boolean)Numbers.isPowerOf2(numberOfBuckets), (Object)"numberOfBuckets must be a power of 2");
        this.numberOfBuckets = numberOfBuckets;
        this.baseline = 0;
        this.baselineCount = (short)numberOfBuckets;
        this.currentSum = numberOfBuckets;
        int slotCount = (numberOfBuckets + 16 - 1) / 16;
        this.slots = new long[slotCount];
    }

    public DenseEstimator(int[] bucketValues) {
        this(bucketValues.length);
        this.baseline = (byte)127;
        this.baselineCount = 0;
        for (int value : bucketValues) {
            Preconditions.checkArgument((value >= 0 && value <= 127 ? 1 : 0) != 0, (String)"values must be >= 0 and <= %s, found %s", (int)127, (int)value);
            if (value < this.baseline) {
                this.baselineCount = 1;
                this.baseline = (byte)value;
                continue;
            }
            if (value != this.baseline) continue;
            this.baselineCount = (short)(this.baselineCount + 1);
        }
        this.currentSum = 0.0;
        int bucket = 0;
        for (int value : bucketValues) {
            this.set(bucket, value - this.baseline);
            this.currentSum += 1.0 / (double)(1L << value);
            ++bucket;
        }
    }

    @Override
    public int getNumberOfBuckets() {
        return this.numberOfBuckets;
    }

    @Override
    public int getMaxAllowedBucketValue() {
        return 127;
    }

    @Override
    public boolean setIfGreater(int bucket, int highestBitPosition) {
        int oldValue;
        int relativeHighestBitPosition = highestBitPosition - this.baseline;
        if (relativeHighestBitPosition > 15) {
            relativeHighestBitPosition = 15;
        }
        if (relativeHighestBitPosition <= (oldValue = this.get(bucket))) {
            return false;
        }
        this.set(bucket, relativeHighestBitPosition);
        this.currentSum -= 1.0 / (double)(1L << oldValue + this.baseline);
        this.currentSum += 1.0 / (double)(1L << relativeHighestBitPosition + this.baseline);
        if (oldValue == 0) {
            this.baselineCount = (short)(this.baselineCount - 1);
            this.rescaleAndRecomputeBaseCountIfNeeded();
        }
        return true;
    }

    private void set(int bucket, int value) {
        int slot = bucket / 16;
        int offset = bucket % 16;
        long bucketClearMask = 15L << offset * 4;
        int n = slot;
        this.slots[n] = this.slots[n] & (bucketClearMask ^ 0xFFFFFFFFFFFFFFFFL);
        long bucketSetMask = (long)value << offset * 4;
        int n2 = slot;
        this.slots[n2] = this.slots[n2] | bucketSetMask;
    }

    private int get(int bucket) {
        int slot = bucket / 16;
        int offset = bucket % 16;
        return (int)(this.slots[slot] >> offset * 4 & 0xFL);
    }

    private void rescaleAndRecomputeBaseCountIfNeeded() {
        while (this.baselineCount == 0) {
            this.baseline = (byte)(this.baseline + 1);
            this.baselineCount = 0;
            for (int i = 0; i < this.numberOfBuckets; ++i) {
                int value = this.get(i);
                this.set(i, --value);
                if (value != 0) continue;
                this.baselineCount = (short)(this.baselineCount + 1);
            }
        }
    }

    @Override
    public long estimate() {
        double alpha = HyperLogLogUtil.computeAlpha(this.numberOfBuckets);
        double result = alpha * (double)this.numberOfBuckets * (double)this.numberOfBuckets / this.currentSum;
        if (result <= 2.5 * (double)this.numberOfBuckets && this.baseline == 0 && this.baselineCount > 0) {
            result = (double)this.numberOfBuckets * Math.log((double)this.numberOfBuckets * 1.0 / (double)this.baselineCount);
        }
        return Math.round(result);
    }

    @Override
    public int estimateSizeInBytes() {
        return DenseEstimator.estimateSizeInBytes(this.numberOfBuckets);
    }

    public static int estimateSizeInBytes(int numberOfBuckets) {
        return (numberOfBuckets + 16 - 1) / 16 * 64 / 8 + INSTANCE_SIZE;
    }

    @Override
    public int[] buckets() {
        int[] result = new int[this.numberOfBuckets];
        for (int i = 0; i < this.numberOfBuckets; ++i) {
            result[i] = this.get(i) + this.baseline;
        }
        return result;
    }
}

