/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.facebook.litho.sections.widget

import androidx.recyclerview.widget.RecyclerView.ItemAnimator
import androidx.recyclerview.widget.RecyclerView.ItemDecoration
import androidx.recyclerview.widget.RecyclerView.OnItemTouchListener
import com.facebook.litho.Component
import com.facebook.litho.Dimen
import com.facebook.litho.Handle
import com.facebook.litho.LithoStartupLogger
import com.facebook.litho.ResourcesScope
import com.facebook.litho.Style
import com.facebook.litho.kotlinStyle
import com.facebook.litho.sections.Section
import com.facebook.litho.widget.LithoRecyclerView
import com.facebook.litho.widget.SectionsRecyclerView.SectionsRecyclerViewLogger
import com.facebook.litho.widget.StickyHeaderControllerFactory

/**
 * Kotlin wrapper for the Component generated by RecyclerCollectionComponentSpec.
 *
 * Note: Since RecyclerCollectionComponent returns a container around a RecyclerView (for
 * pull-to-refresh), Style/common props apply to that container instead of the underlying
 * RecyclerView. The properties named "recyclerView*" allow you to apply some of these properties to
 * the underlying RecyclerView instead.
 */
@Suppress("FunctionName")
inline fun ResourcesScope.RecyclerCollectionComponent(
    section: Section,
    recyclerConfiguration: RecyclerConfiguration =
        RecyclerCollectionComponentSpec.recyclerConfiguration,
    itemAnimator: ItemAnimator? = RecyclerCollectionComponentSpec.itemAnimator,
    itemDecoration: ItemDecoration? = null,
    canMeasureRecycler: Boolean = false,
    loadingComponent: Component? = null,
    emptyComponent: Component? = null,
    errorComponent: Component? = null,
    recyclerViewClipToPadding: Boolean = RecyclerCollectionComponentSpec.clipToPadding,
    recyclerViewClipChildren: Boolean = RecyclerCollectionComponentSpec.clipChildren,
    recyclerViewStartPadding: Dimen? = null,
    recyclerViewEndPadding: Dimen? = null,
    recyclerViewTopPadding: Dimen? = null,
    recyclerViewBottomPadding: Dimen? = null,
    disablePTR: Boolean = false,
    nestedScrollingEnabled: Boolean = RecyclerCollectionComponentSpec.nestedScrollingEnabled,
    scrollBarStyle: Int = RecyclerCollectionComponentSpec.scrollBarStyle,
    recyclerViewId: Int = RecyclerCollectionComponentSpec.recyclerViewId,
    overScrollMode: Int = RecyclerCollectionComponentSpec.overScrollMode,
    refreshProgressBarColor: Int = RecyclerCollectionComponentSpec.refreshProgressBarColor,
    touchInterceptor: LithoRecyclerView.TouchInterceptor? = null,
    itemTouchListener: OnItemTouchListener? = null,
    eventsController: RecyclerCollectionEventsController? = null,
    sectionTreeTag: String? = null,
    startupLogger: LithoStartupLogger? = null,
    stickyHeaderControllerFactory: StickyHeaderControllerFactory? = null,
    sectionsViewLogger: SectionsRecyclerViewLogger? = null,
    handle: Handle? = null,
    style: Style? = null,
): RecyclerCollectionComponent =
    RecyclerCollectionComponent.create(context)
        .section(section)
        .recyclerConfiguration(recyclerConfiguration)
        .itemAnimator(itemAnimator)
        .itemDecoration(itemDecoration)
        .canMeasureRecycler(canMeasureRecycler)
        .loadingComponent(loadingComponent)
        .emptyComponent(emptyComponent)
        .errorComponent(errorComponent)
        .clipToPadding(recyclerViewClipToPadding)
        .clipChildren(recyclerViewClipChildren)
        .leftPaddingPx(recyclerViewStartPadding?.toPixels(resourceResolver) ?: 0)
        .rightPaddingPx(recyclerViewEndPadding?.toPixels(resourceResolver) ?: 0)
        .topPaddingPx(recyclerViewTopPadding?.toPixels(resourceResolver) ?: 0)
        .bottomPaddingPx(recyclerViewBottomPadding?.toPixels(resourceResolver) ?: 0)
        .disablePTR(disablePTR)
        .nestedScrollingEnabled(nestedScrollingEnabled)
        .scrollBarStyle(scrollBarStyle)
        .recyclerViewId(recyclerViewId)
        .overScrollMode(overScrollMode)
        .refreshProgressBarColor(refreshProgressBarColor)
        .touchInterceptor(touchInterceptor)
        .itemTouchListener(itemTouchListener)
        .eventsController(eventsController)
        .sectionTreeTag(sectionTreeTag)
        .startupLogger(startupLogger)
        .stickyHeaderControllerFactory(stickyHeaderControllerFactory)
        .sectionsViewLogger(sectionsViewLogger)
        .handle(handle)
        .kotlinStyle(style)
        .build()
