/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.hadoop.\$internal.org.apache.commons.text.similarity;

import com.facebook.presto.hadoop.\$internal.org.apache.commons.lang3.Validate;

/**
 * This stores a {@link SimilarityScore} implementation and a {@link CharSequence} "left" string.
 * The {@link #apply(CharSequence right)} method accepts the "right" string and invokes the
 * comparison function for the pair of strings.
 *
 * <p>
 * The following is an example which finds the most similar string:
 * </p>
 * <pre>
 * SimilarityScore&lt;Integer&gt; similarityScore = new LevenshteinDistance();
 * String target = "Apache";
 * SimilarityScoreFrom&lt;Integer&gt; similarityScoreFrom =
 *     new SimilarityScoreFrom&lt;Integer&gt;(similarityScore, target);
 * String mostSimilar = null;
 * Integer shortestDistance = null;
 *
 * for (String test : new String[] { "Appaloosa", "a patchy", "apple" }) {
 *     Integer distance = similarityScoreFrom.apply(test);
 *     if (shortestDistance == null || distance &lt; shortestDistance) {
 *         shortestDistance = distance;
 *         mostSimilar = test;
 *     }
 * }
 *
 * System.out.println("The string most similar to \"" + target + "\" "
 *     + "is \"" + mostSimilar + "\" because "
 *     + "its distance is only " + shortestDistance + ".");
 * </pre>
 *
 * @param <R> This is the type of similarity score used by the SimilarityScore function.
 * @since 1.0
 */
public class SimilarityScoreFrom<R> {

    /**
     * Similarity score.
     */
    private final SimilarityScore<R> similarityScore;

    /**
     * Left parameter used in distance function.
     */
    private final CharSequence left;

    /**
     * This accepts the similarity score implementation and the "left" string.
     *
     * @param similarityScore This may not be null.
     * @param left This may be null here,
     *             but the SimilarityScore#compare(CharSequence left, CharSequence right)
     *             implementation may not accept nulls.
     */
    public SimilarityScoreFrom(final SimilarityScore<R> similarityScore, final CharSequence left) {
        Validate.isTrue(similarityScore != null, "The edit distance may not be null.");

        this.similarityScore = similarityScore;
        this.left = left;
    }

    /**
     * This compares "left" field against the "right" parameter
     * using the "similarity score" implementation.
     *
     * @param right the second CharSequence
     * @return The similarity score between two CharSequences
     */
    public R apply(final CharSequence right) {
        return similarityScore.apply(left, right);
    }

    /**
     * Gets the left parameter.
     *
     * @return The left parameter
     */
    public CharSequence getLeft() {
        return left;
    }

    /**
     * Gets the edit distance.
     *
     * @return The edit distance
     */
    public SimilarityScore<R> getSimilarityScore() {
        return similarityScore;
    }

}
