/**
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package com.facebook.presto.hadoop.\$internal.org.apache.kerby.cms.type;

import com.facebook.presto.hadoop.\$internal.org.apache.kerby.asn1.Asn1FieldInfo;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.asn1.EnumType;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.asn1.type.Asn1BitString;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.asn1.type.Asn1Integer;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.asn1.type.Asn1SequenceType;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.AlgorithmIdentifier;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.AttCertIssuer;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.AttCertValidityPeriod;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.Attributes;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.CertificateSerialNumber;
import com.facebook.presto.hadoop.\$internal.org.apache.kerby.x509.type.Extensions;

/**
 * AttributeCertificateInfoV1 ::= SEQUENCE {
 *   version AttCertVersionV1 DEFAULT v1,
 *   subject Subject,
 *   issuer GeneralNames,
 *   signature AlgorithmIdentifier,
 *   serialNumber CertificateSerialNumber,
 *   attCertValidityPeriod AttCertValidityPeriod,
 *   attributes SEQUENCE OF Attribute,
 *   issuerUniqueID UniqueIdentifier OPTIONAL,
 *   extensions Extensions OPTIONAL
 * }
 *
 * AttCertVersionV1 ::= INTEGER { v1(0) }
 */
public class AttributeCertificateInfoV1 extends Asn1SequenceType {
    protected enum AttributeCertificateInfoV1Field implements EnumType {
        VERSION,
        SUBJECT,
        ISSUER,
        SIGNATURE,
        SERIAL_NUMBER,
        ATTR_CERT_VALIDITY_PERIOD,
        ATTRIBUTES,
        ISSUER_UNIQUE_ID,
        EXTENSIONS;

        @Override
        public int getValue() {
            return ordinal();
        }

        @Override
        public String getName() {
            return name();
        }
    }

    static Asn1FieldInfo[] fieldInfos = new Asn1FieldInfo[] {
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.VERSION, Asn1Integer.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.SUBJECT, Subject.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.ISSUER, AttCertIssuer.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.SIGNATURE, AlgorithmIdentifier.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.SERIAL_NUMBER, CertificateSerialNumber.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.ATTR_CERT_VALIDITY_PERIOD, AttCertValidityPeriod.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.ATTRIBUTES, Attributes.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.ISSUER_UNIQUE_ID, Asn1BitString.class),
            new Asn1FieldInfo(AttributeCertificateInfoV1Field.EXTENSIONS, Extensions.class)
    };

    public AttributeCertificateInfoV1() {
        super(fieldInfos);
    }

    public int getVersion() {
        return getFieldAsInteger(AttributeCertificateInfoV1Field.VERSION);
    }

    public void setVersion(int version) {
        setFieldAsInt(AttributeCertificateInfoV1Field.VERSION, version);
    }

    public Subject getSubject() {
        return getFieldAs(AttributeCertificateInfoV1Field.SUBJECT, Subject.class);
    }

    public void setSubject(Subject subject) {
        setFieldAs(AttributeCertificateInfoV1Field.SUBJECT, subject);
    }

    public AttCertIssuer getIssuer() {
        return getFieldAs(AttributeCertificateInfoV1Field.ISSUER, AttCertIssuer.class);
    }

    public void setIssuer(AttCertIssuer attCertIssuer) {
        setFieldAs(AttributeCertificateInfoV1Field.ISSUER, attCertIssuer);
    }

    public AlgorithmIdentifier getSignature() {
        return getFieldAs(AttributeCertificateInfoV1Field.SIGNATURE, AlgorithmIdentifier.class);
    }

    public void setSignature(AlgorithmIdentifier signature) {
        setFieldAs(AttributeCertificateInfoV1Field.SIGNATURE, signature);
    }

    public CertificateSerialNumber getSerialNumber() {
        return getFieldAs(AttributeCertificateInfoV1Field.SERIAL_NUMBER, CertificateSerialNumber.class);
    }

    public void setSerialNumber(CertificateSerialNumber certificateSerialNumber) {
        setFieldAs(AttributeCertificateInfoV1Field.SERIAL_NUMBER, certificateSerialNumber);
    }

    public AttCertValidityPeriod getAttrCertValidityPeriod() {
        return getFieldAs(AttributeCertificateInfoV1Field.ATTR_CERT_VALIDITY_PERIOD, AttCertValidityPeriod.class);
    }

    public void setAttrCertValidityPeriod(AttCertValidityPeriod attrCertValidityPeriod) {
        setFieldAs(AttributeCertificateInfoV1Field.ATTR_CERT_VALIDITY_PERIOD, attrCertValidityPeriod);
    }

    public Attributes getAttributes() {
        return getFieldAs(AttributeCertificateInfoV1Field.ATTRIBUTES, Attributes.class);
    }

    public void setAttributes(Attributes attributes) {
        setFieldAs(AttributeCertificateInfoV1Field.ATTRIBUTES, attributes);
    }

    public byte[] getIssuerUniqueID() {
        return getFieldAs(AttributeCertificateInfoV1Field.ISSUER_UNIQUE_ID, Asn1BitString.class).getValue();
    }

    public void setIssuerUniqueId(byte[] issuerUniqueId) {
        setFieldAs(AttributeCertificateInfoV1Field.ISSUER_UNIQUE_ID, new Asn1BitString(issuerUniqueId));
    }

    public Extensions getExtensions() {
        return getFieldAs(AttributeCertificateInfoV1Field.EXTENSIONS, Extensions.class);
    }

    public void setExtensions(Extensions extensions) {
        setFieldAs(AttributeCertificateInfoV1Field.EXTENSIONS, extensions);
    }
}
