package com.facebook.presto.hadoop.\$internal.org.bouncycastle.crypto.params;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.security.SecureRandom;

import com.facebook.presto.hadoop.\$internal.org.bouncycastle.math.ec.rfc8032.Ed448;
import com.facebook.presto.hadoop.\$internal.org.bouncycastle.util.Arrays;
import com.facebook.presto.hadoop.\$internal.org.bouncycastle.util.io.Streams;

public final class Ed448PrivateKeyParameters
    extends AsymmetricKeyParameter
{
    public static final int KEY_SIZE = Ed448.SECRET_KEY_SIZE;
    public static final int SIGNATURE_SIZE = Ed448.SIGNATURE_SIZE;

    private final byte[] data = new byte[KEY_SIZE];

    private Ed448PublicKeyParameters cachedPublicKey;

    public Ed448PrivateKeyParameters(SecureRandom random)
    {
        super(true);

        Ed448.generatePrivateKey(random, data);
    }

    public Ed448PrivateKeyParameters(byte[] buf)
    {
        this(validate(buf), 0);
    }

    public Ed448PrivateKeyParameters(byte[] buf, int off)
    {
        super(true);

        System.arraycopy(buf, off, data, 0, KEY_SIZE);
    }

    public Ed448PrivateKeyParameters(InputStream input) throws IOException
    {
        super(true);

        if (KEY_SIZE != Streams.readFully(input, data))
        {
            throw new EOFException("EOF encountered in middle of Ed448 private key");
        }
    }

    public void encode(byte[] buf, int off)
    {
        System.arraycopy(data, 0, buf, off, KEY_SIZE);
    }

    public byte[] getEncoded()
    {
        return Arrays.clone(data);
    }

    public Ed448PublicKeyParameters generatePublicKey()
    {
        synchronized (data)
        {
            if (null == cachedPublicKey)
            {
                cachedPublicKey = new Ed448PublicKeyParameters(Ed448.generatePublicKey(data, 0));
            }

            return cachedPublicKey;
        }
    }

    /**
     * @deprecated use overload that doesn't take a public key
     */
    public void sign(int algorithm, Ed448PublicKeyParameters publicKey, byte[] ctx, byte[] msg, int msgOff, int msgLen, byte[] sig, int sigOff)
    {
        sign(algorithm, ctx, msg, msgOff, msgLen, sig, sigOff);
    }

    public void sign(int algorithm, byte[] ctx, byte[] msg, int msgOff, int msgLen, byte[] sig, int sigOff)
    {
        Ed448PublicKeyParameters publicKey = generatePublicKey();

        byte[] pk = new byte[Ed448.PUBLIC_KEY_SIZE];
        publicKey.encode(pk, 0);

        switch (algorithm)
        {
        case Ed448.Algorithm.Ed448:
        {
            if (null == ctx)
            {
                throw new NullPointerException("'ctx' cannot be null");
            }
            if (ctx.length > 255)
            {
                throw new IllegalArgumentException("ctx");
            }

            Ed448.sign(data, 0, pk, 0, ctx, msg, msgOff, msgLen, sig, sigOff);
            break;
        }
        case Ed448.Algorithm.Ed448ph:
        {
            if (null == ctx)
            {
                throw new NullPointerException("'ctx' cannot be null");
            }
            if (ctx.length > 255)
            {
                throw new IllegalArgumentException("ctx");
            }
            if (Ed448.PREHASH_SIZE != msgLen)
            {
                throw new IllegalArgumentException("msgLen");
            }

            Ed448.signPrehash(data, 0, pk, 0, ctx, msg, msgOff, sig, sigOff);
            break;
        }
        default:
        {
            throw new IllegalArgumentException("algorithm");
        }
        }
    }

    private static byte[] validate(byte[] buf)
    {
        if (buf.length != KEY_SIZE)
        {
            throw new IllegalArgumentException("'buf' must have length " + KEY_SIZE);
        }
        return buf;
    }
}
