/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.datanode.web;

import com.facebook.presto.hadoop.\$internal.io.netty.bootstrap.Bootstrap;
import com.facebook.presto.hadoop.\$internal.io.netty.buffer.Unpooled;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.Channel;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelFuture;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelFutureListener;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelHandlerContext;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelInboundHandlerAdapter;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelInitializer;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.ChannelPipeline;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.SimpleChannelInboundHandler;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.socket.SocketChannel;
import com.facebook.presto.hadoop.\$internal.io.netty.channel.socket.nio.NioSocketChannel;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.DefaultFullHttpRequest;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.DefaultHttpResponse;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpRequest;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpRequestEncoder;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpResponseEncoder;
import com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpHeaderValues;
import com.facebook.presto.hadoop.\$internal.org.slf4j.Logger;

import java.net.InetSocketAddress;

import static com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpResponseStatus.INTERNAL_SERVER_ERROR;
import static com.facebook.presto.hadoop.\$internal.io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

/**
 * Dead simple session-layer HTTP proxy. It gets the HTTP responses
 * inside the context, assuming that the remote peer is reasonable fast and
 * the response is small. The upper layer should be filtering out malicious
 * inputs.
 */
class SimpleHttpProxyHandler extends SimpleChannelInboundHandler<HttpRequest> {
  private String uri;
  private Channel proxiedChannel;
  private final InetSocketAddress host;
  static final Logger LOG = DatanodeHttpServer.LOG;

  SimpleHttpProxyHandler(InetSocketAddress host) {
    this.host = host;
  }

  private static class Forwarder extends ChannelInboundHandlerAdapter {
    private final String uri;
    private final Channel client;

    private Forwarder(String uri, Channel client) {
      this.uri = uri;
      this.client = client;
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
      closeOnFlush(client);
    }

    @Override
    public void channelRead(final ChannelHandlerContext ctx, Object msg) {
      client.writeAndFlush(msg).addListener(new ChannelFutureListener() {
        @Override
        public void operationComplete(ChannelFuture future) {
          if (future.isSuccess()) {
            ctx.channel().read();
          } else {
            LOG.debug("Proxy failed. Cause: ", future.cause());
            future.channel().close();
          }
        }
      });
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
      LOG.debug("Proxy for " + uri + " failed. cause: ", cause);
      closeOnFlush(ctx.channel());
    }
  }

  @Override
  public void channelRead0
    (final ChannelHandlerContext ctx, final HttpRequest req) {
    uri = req.uri();
    final Channel client = ctx.channel();
    Bootstrap proxiedServer = new Bootstrap()
      .group(client.eventLoop())
      .channel(NioSocketChannel.class)
      .handler(new ChannelInitializer<SocketChannel>() {
        @Override
        protected void initChannel(SocketChannel ch) throws Exception {
          ChannelPipeline p = ch.pipeline();
          p.addLast(new HttpRequestEncoder(), new Forwarder(uri, client));
        }
      });
    ChannelFuture f = proxiedServer.connect(host);
    proxiedChannel = f.channel();
    f.addListener(new ChannelFutureListener() {
      @Override
      public void operationComplete(ChannelFuture future) throws Exception {
        if (future.isSuccess()) {
          ctx.channel().pipeline().remove(HttpResponseEncoder.class);
          HttpRequest newReq = new DefaultFullHttpRequest(HTTP_1_1, req.method(), req.uri());
          newReq.headers().add(req.headers());
          newReq.headers().set(CONNECTION, HttpHeaderValues.CLOSE);
          future.channel().writeAndFlush(newReq);
        } else {
          DefaultHttpResponse resp = new DefaultHttpResponse(HTTP_1_1,
            INTERNAL_SERVER_ERROR);
          resp.headers().set(CONNECTION, HttpHeaderValues.CLOSE);
          LOG.info("Proxy " + uri + " failed. Cause: ", future.cause());
          ctx.writeAndFlush(resp).addListener(ChannelFutureListener.CLOSE);
          client.close();
        }
      }
    });
  }

  @Override
  public void channelInactive(ChannelHandlerContext ctx) {
    if (proxiedChannel != null) {
      proxiedChannel.close();
      proxiedChannel = null;
    }
  }

  @Override
  public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
    if (LOG.isDebugEnabled()) {
      LOG.debug("Proxy for " + uri + " failed. cause: ", cause);
    }
    if (proxiedChannel != null) {
      proxiedChannel.close();
      proxiedChannel = null;
    }
    ctx.close();
  }

  private static void closeOnFlush(Channel ch) {
    if (ch.isActive()) {
      ch.writeAndFlush(Unpooled.EMPTY_BUFFER)
        .addListener(ChannelFutureListener.CLOSE);
    }
  }
}
