/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.facebook.presto.hadoop.$internal.io.netty.channel.epoll;

import com.facebook.presto.hadoop.$internal.io.netty.buffer.ByteBufAllocator;
import com.facebook.presto.hadoop.$internal.io.netty.channel.ChannelOption;
import com.facebook.presto.hadoop.$internal.io.netty.channel.DefaultChannelConfig;
import com.facebook.presto.hadoop.$internal.io.netty.channel.MessageSizeEstimator;
import com.facebook.presto.hadoop.$internal.io.netty.channel.RecvByteBufAllocator;
import com.facebook.presto.hadoop.$internal.io.netty.channel.socket.ServerSocketChannelConfig;
import com.facebook.presto.hadoop.$internal.io.netty.util.NetUtil;

import java.util.Map;

import static com.facebook.presto.hadoop.$internal.io.netty.channel.ChannelOption.SO_BACKLOG;
import static com.facebook.presto.hadoop.$internal.io.netty.channel.ChannelOption.SO_RCVBUF;
import static com.facebook.presto.hadoop.$internal.io.netty.channel.ChannelOption.SO_REUSEADDR;

public final class EpollServerSocketChannelConfig extends DefaultChannelConfig
        implements ServerSocketChannelConfig {

    private final EpollServerSocketChannel channel;
    private volatile int backlog = NetUtil.SOMAXCONN;

    EpollServerSocketChannelConfig(EpollServerSocketChannel channel) {
        super(channel);
        this.channel = channel;

        // Use SO_REUSEADDR by default as java.nio does the same.
        //
        // See https://github.com/netty/netty/issues/2605
        setReuseAddress(true);
    }

    @Override
    public Map<ChannelOption<?>, Object> getOptions() {
        return getOptions(super.getOptions(), SO_RCVBUF, SO_REUSEADDR, SO_BACKLOG, EpollChannelOption.SO_REUSEPORT);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T getOption(ChannelOption<T> option) {
        if (option == SO_RCVBUF) {
            return (T) Integer.valueOf(getReceiveBufferSize());
        }
        if (option == SO_REUSEADDR) {
            return (T) Boolean.valueOf(isReuseAddress());
        }
        if (option == SO_BACKLOG) {
            return (T) Integer.valueOf(getBacklog());
        }
        if (option == EpollChannelOption.SO_REUSEPORT) {
            return (T) Boolean.valueOf(isReusePort());
        }
        return super.getOption(option);
    }

    @Override
    public <T> boolean setOption(ChannelOption<T> option, T value) {
        validate(option, value);

        if (option == SO_RCVBUF) {
            setReceiveBufferSize((Integer) value);
        } else if (option == SO_REUSEADDR) {
            setReuseAddress((Boolean) value);
        } else if (option == SO_BACKLOG) {
            setBacklog((Integer) value);
        } else if (option == EpollChannelOption.SO_REUSEPORT) {
            setReusePort((Boolean) value);
        } else {
            return super.setOption(option, value);
        }

        return true;
    }

    @Override
    public boolean isReuseAddress() {
        return Native.isReuseAddress(channel.fd) == 1;
    }

    @Override
    public EpollServerSocketChannelConfig setReuseAddress(boolean reuseAddress) {
        Native.setReuseAddress(channel.fd, reuseAddress ? 1 : 0);
        return this;
    }

    @Override
    public int getReceiveBufferSize() {
        return Native.getReceiveBufferSize(channel.fd);
    }

    @Override
    public EpollServerSocketChannelConfig setReceiveBufferSize(int receiveBufferSize) {
        Native.setReceiveBufferSize(channel.fd, receiveBufferSize);

        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setPerformancePreferences(int connectionTime, int latency, int bandwidth) {
        return this;
    }

    @Override
    public int getBacklog() {
        return backlog;
    }

    @Override
    public EpollServerSocketChannelConfig setBacklog(int backlog) {
        if (backlog < 0) {
            throw new IllegalArgumentException("backlog: " + backlog);
        }
        this.backlog = backlog;
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setConnectTimeoutMillis(int connectTimeoutMillis) {
        super.setConnectTimeoutMillis(connectTimeoutMillis);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setMaxMessagesPerRead(int maxMessagesPerRead) {
        super.setMaxMessagesPerRead(maxMessagesPerRead);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setWriteSpinCount(int writeSpinCount) {
        super.setWriteSpinCount(writeSpinCount);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setAllocator(ByteBufAllocator allocator) {
        super.setAllocator(allocator);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setRecvByteBufAllocator(RecvByteBufAllocator allocator) {
        super.setRecvByteBufAllocator(allocator);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setAutoRead(boolean autoRead) {
        super.setAutoRead(autoRead);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setWriteBufferHighWaterMark(int writeBufferHighWaterMark) {
        super.setWriteBufferHighWaterMark(writeBufferHighWaterMark);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setWriteBufferLowWaterMark(int writeBufferLowWaterMark) {
        super.setWriteBufferLowWaterMark(writeBufferLowWaterMark);
        return this;
    }

    @Override
    public EpollServerSocketChannelConfig setMessageSizeEstimator(MessageSizeEstimator estimator) {
        super.setMessageSizeEstimator(estimator);
        return this;
    }

    /**
     * Returns {@code true} if the SO_REUSEPORT option is set.
     */
    public boolean isReusePort() {
        return Native.isReusePort(channel.fd) == 1;
    }

    /**
     * Set the SO_REUSEPORT option on the underlying Channel. This will allow to bind multiple
     * {@link EpollSocketChannel}s to the same port and so accept connections with multiple threads.
     *
     * Be aware this method needs be called before {@link EpollSocketChannel#bind(java.net.SocketAddress)} to have
     * any affect.
     */
    public EpollServerSocketChannelConfig setReusePort(boolean reusePort) {
        Native.setReusePort(channel.fd, reusePort ? 1 : 0);
        return this;
    }

    @Override
    protected void autoReadCleared() {
        channel.clearEpollIn();
    }
}
