/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.cassandra;

import com.datastax.driver.core.Host;
import com.datastax.driver.core.TokenRange;
import com.facebook.presto.cassandra.CassandraClientConfig;
import com.facebook.presto.cassandra.CassandraErrorCode;
import com.facebook.presto.cassandra.CassandraSession;
import com.facebook.presto.cassandra.SizeEstimate;
import com.facebook.presto.cassandra.TokenRing;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import javax.inject.Inject;

public class CassandraTokenSplitManager {
    private final CassandraSession session;
    private final int splitSize;
    private final Optional<Long> configSplitsPerNode;

    @Inject
    public CassandraTokenSplitManager(CassandraSession session, CassandraClientConfig config) {
        this(session, config.getSplitSize(), config.getSplitsPerNode());
    }

    public CassandraTokenSplitManager(CassandraSession session, int splitSize, Optional<Long> configSplitsPerNode) {
        this.session = Objects.requireNonNull(session, "session is null");
        this.splitSize = splitSize;
        this.configSplitsPerNode = configSplitsPerNode;
    }

    public List<TokenSplit> getSplits(String keyspace, String table, Optional<Long> sessionSplitsPerNode) {
        Set<TokenRange> tokenRanges = this.session.getTokenRanges();
        if (tokenRanges.isEmpty()) {
            throw new PrestoException((ErrorCodeSupplier)CassandraErrorCode.CASSANDRA_METADATA_ERROR, "The cluster metadata is not available. Please make sure that the Cassandra cluster is up and running, and that the contact points are specified correctly.");
        }
        if (tokenRanges.stream().anyMatch(TokenRange::isWrappedAround)) {
            tokenRanges = this.unwrap(tokenRanges);
        }
        Optional<TokenRing> tokenRing = TokenRing.createForPartitioner(this.session.getPartitioner());
        long totalPartitionsCount = this.getTotalPartitionsCount(keyspace, table, sessionSplitsPerNode);
        ArrayList<TokenSplit> splits = new ArrayList<TokenSplit>();
        for (TokenRange tokenRange : tokenRanges) {
            if (tokenRange.isEmpty()) continue;
            Preconditions.checkState((!tokenRange.isWrappedAround() ? 1 : 0) != 0, (Object)"all token ranges must be unwrapped at this step");
            List<String> endpoints = this.getEndpoints(keyspace, tokenRange);
            Preconditions.checkState((!endpoints.isEmpty() ? 1 : 0) != 0, (String)"endpoints is empty for token range: %s", (Object)tokenRange);
            if (!tokenRing.isPresent()) {
                Preconditions.checkState((!tokenRange.isWrappedAround() ? 1 : 0) != 0, (Object)"all token ranges must be unwrapped at this step");
                splits.add(CassandraTokenSplitManager.createSplit(tokenRange, endpoints));
                continue;
            }
            double tokenRangeRingFraction = tokenRing.get().getRingFraction(tokenRange.getStart().toString(), tokenRange.getEnd().toString());
            long partitionsCountEstimate = Math.round((double)totalPartitionsCount * tokenRangeRingFraction);
            Preconditions.checkState((partitionsCountEstimate >= 0L ? 1 : 0) != 0, (String)"unexpected partitions count estimate: %d", (long)partitionsCountEstimate);
            int subSplitCount = Math.max(StrictMath.toIntExact(partitionsCountEstimate / (long)this.splitSize), 1);
            List subRanges = tokenRange.splitEvenly(subSplitCount);
            for (TokenRange subRange : subRanges) {
                if (subRange.isEmpty()) continue;
                Preconditions.checkState((!subRange.isWrappedAround() ? 1 : 0) != 0, (Object)"all token ranges must be unwrapped at this step");
                splits.add(CassandraTokenSplitManager.createSplit(subRange, endpoints));
            }
        }
        Collections.shuffle(splits, ThreadLocalRandom.current());
        return Collections.unmodifiableList(splits);
    }

    private Set<TokenRange> unwrap(Set<TokenRange> tokenRanges) {
        ImmutableSet.Builder result = ImmutableSet.builder();
        for (TokenRange range : tokenRanges) {
            result.addAll((Iterable)range.unwrap());
        }
        return result.build();
    }

    public long getTotalPartitionsCount(String keyspace, String table, Optional<Long> sessionSplitsPerNode) {
        if (sessionSplitsPerNode.isPresent()) {
            return sessionSplitsPerNode.get();
        }
        if (this.configSplitsPerNode.isPresent()) {
            return this.configSplitsPerNode.get();
        }
        List<SizeEstimate> estimates = this.session.getSizeEstimates(keyspace, table);
        return estimates.stream().mapToLong(SizeEstimate::getPartitionsCount).sum();
    }

    private List<String> getEndpoints(String keyspace, TokenRange tokenRange) {
        Set<Host> endpoints = this.session.getReplicas(keyspace, tokenRange);
        return Collections.unmodifiableList(endpoints.stream().map(Host::toString).collect(Collectors.toList()));
    }

    private static TokenSplit createSplit(TokenRange range, List<String> endpoints) {
        Preconditions.checkArgument((!range.isEmpty() ? 1 : 0) != 0, (Object)"tokenRange must not be empty");
        String startToken = range.getStart().toString();
        String endToken = range.getEnd().toString();
        return new TokenSplit(startToken, endToken, endpoints);
    }

    public static class TokenSplit {
        private String startToken;
        private String endToken;
        private List<String> hosts;

        public TokenSplit(String startToken, String endToken, List<String> hosts) {
            this.startToken = Objects.requireNonNull(startToken, "startToken is null");
            this.endToken = Objects.requireNonNull(endToken, "endToken is null");
            this.hosts = ImmutableList.copyOf((Collection)Objects.requireNonNull(hosts, "hosts is null"));
        }

        public String getStartToken() {
            return this.startToken;
        }

        public String getEndToken() {
            return this.endToken;
        }

        public List<String> getHosts() {
            return this.hosts;
        }
    }
}

