/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.facebook.presto.common;

import com.facebook.drift.annotations.ThriftConstructor;
import com.facebook.drift.annotations.ThriftField;
import com.facebook.drift.annotations.ThriftStruct;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.Objects;

import static java.util.Objects.requireNonNull;

@ThriftStruct
public final class ErrorCode
{
    private final int code;
    private final String name;
    private final ErrorType type;
    private final boolean retriable;

    @JsonCreator
    @ThriftConstructor
    public ErrorCode(
            @JsonProperty("code") int code,
            @JsonProperty("name") String name,
            @JsonProperty("type") ErrorType type,
            @JsonProperty("retriable") boolean retriable)
    {
        if (code < 0) {
            throw new IllegalArgumentException("code is negative");
        }
        this.code = code;
        this.name = requireNonNull(name, "name is null");
        this.type = requireNonNull(type, "type is null");
        this.retriable = retriable;
    }

    public ErrorCode(int code, String name, ErrorType type)
    {
        this(code, name, type, false);
    }

    @JsonProperty
    @ThriftField(1)
    public int getCode()
    {
        return code;
    }

    @JsonProperty
    @ThriftField(2)
    public String getName()
    {
        return name;
    }

    @JsonProperty
    @ThriftField(3)
    public ErrorType getType()
    {
        return type;
    }

    @JsonProperty
    @ThriftField(4)
    public boolean isRetriable()
    {
        return retriable;
    }

    @Override
    public String toString()
    {
        return name + ":" + code;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj) {
            return true;
        }
        if (obj == null || getClass() != obj.getClass()) {
            return false;
        }

        ErrorCode that = (ErrorCode) obj;
        return Objects.equals(this.code, that.code);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(code);
    }
}
