/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.block;

import alluxio.client.WriteType;
import alluxio.client.block.BlockMasterClient;
import alluxio.client.block.BlockWorkerInfo;
import alluxio.client.block.policy.BlockLocationPolicy;
import alluxio.client.block.policy.options.GetWorkerOptions;
import alluxio.client.block.stream.BlockInStream;
import alluxio.client.block.stream.BlockOutStream;
import alluxio.client.block.stream.DataWriter;
import alluxio.client.block.util.BlockLocationUtils;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.URIStatus;
import alluxio.client.file.options.InStreamOptions;
import alluxio.client.file.options.OutStreamOptions;
import alluxio.collections.Pair;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.PreconditionMessage;
import alluxio.exception.status.ResourceExhaustedException;
import alluxio.exception.status.UnavailableException;
import alluxio.network.TieredIdentityFactory;
import alluxio.resource.CloseableResource;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.com.google.common.collect.ImmutableMap;
import alluxio.shaded.client.com.google.common.collect.Lists;
import alluxio.shaded.client.com.google.common.collect.Sets;
import alluxio.shaded.client.javax.annotation.concurrent.ThreadSafe;
import alluxio.wire.BlockInfo;
import alluxio.wire.BlockLocation;
import alluxio.wire.TieredIdentity;
import alluxio.wire.WorkerNetAddress;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class BlockStoreClient {
    private static final Logger LOG = LoggerFactory.getLogger(BlockStoreClient.class);
    private final FileSystemContext mContext;
    private final TieredIdentity mTieredIdentity;

    public static BlockStoreClient create(FileSystemContext context) {
        return new BlockStoreClient(context, TieredIdentityFactory.localIdentity(context.getClusterConf()));
    }

    @VisibleForTesting
    BlockStoreClient(FileSystemContext context, TieredIdentity tieredIdentity) {
        this.mContext = context;
        this.mTieredIdentity = tieredIdentity;
    }

    public BlockInfo getInfo(long blockId) throws IOException {
        try (CloseableResource<BlockMasterClient> masterClientResource = this.mContext.acquireBlockMasterClientResource();){
            BlockInfo blockInfo = masterClientResource.get().getBlockInfo(blockId);
            return blockInfo;
        }
    }

    public BlockInStream getInStream(long blockId, InStreamOptions options) throws IOException {
        return this.getInStream(blockId, options, ImmutableMap.of());
    }

    public BlockInStream getInStream(long blockId, InStreamOptions options, Map<WorkerNetAddress, Long> failedWorkers) throws IOException {
        BlockInfo info = this.getInfo(blockId);
        return this.getInStream(info, options, failedWorkers);
    }

    public BlockInStream getInStream(BlockInfo info, InStreamOptions options, Map<WorkerNetAddress, Long> failedWorkers) throws IOException {
        Pair<WorkerNetAddress, BlockInStream.BlockInStreamSource> dataSourceAndType = this.getDataSourceAndType(info, options.getStatus(), options.getUfsReadLocationPolicy(), failedWorkers);
        WorkerNetAddress dataSource = dataSourceAndType.getFirst();
        BlockInStream.BlockInStreamSource dataSourceType = dataSourceAndType.getSecond();
        try {
            return BlockInStream.create(this.mContext, info, dataSource, dataSourceType, options);
        }
        catch (UnavailableException e) {
            failedWorkers.put(dataSource, System.currentTimeMillis());
            throw e;
        }
    }

    public Pair<WorkerNetAddress, BlockInStream.BlockInStreamSource> getDataSourceAndType(BlockInfo info, URIStatus status, BlockLocationPolicy policy, Map<WorkerNetAddress, Long> failedWorkers) throws IOException {
        Set<WorkerNetAddress> workerPool;
        List<BlockLocation> locations = info.getLocations();
        List<BlockWorkerInfo> blockWorkerInfo = Collections.emptyList();
        if (status.isPersisted() || status.getPersistenceState().equals("TO_BE_PERSISTED")) {
            blockWorkerInfo = this.mContext.getCachedWorkers();
            if (blockWorkerInfo.isEmpty()) {
                throw new UnavailableException(ExceptionMessage.NO_WORKER_AVAILABLE.getMessage(new Object[0]));
            }
            workerPool = blockWorkerInfo.stream().map(BlockWorkerInfo::getNetAddress).collect(Collectors.toSet());
        } else {
            if (locations.isEmpty()) {
                blockWorkerInfo = this.mContext.getCachedWorkers();
                if (blockWorkerInfo.isEmpty()) {
                    throw new UnavailableException(ExceptionMessage.NO_WORKER_AVAILABLE.getMessage(new Object[0]));
                }
                throw new UnavailableException(MessageFormat.format("Block {0} is unavailable in both Alluxio and UFS.", info.getBlockId()));
            }
            workerPool = locations.stream().map(BlockLocation::getWorkerAddress).collect(Collectors.toSet());
        }
        Set<WorkerNetAddress> workers = this.handleFailedWorkers(workerPool, failedWorkers);
        BlockInStream.BlockInStreamSource dataSourceType = null;
        WorkerNetAddress dataSource = null;
        locations = locations.stream().filter(location -> workers.contains(location.getWorkerAddress())).collect(Collectors.toList());
        if (!locations.isEmpty()) {
            List<WorkerNetAddress> tieredLocations = locations.stream().map(BlockLocation::getWorkerAddress).collect(Collectors.toList());
            Collections.shuffle(tieredLocations);
            Optional<Pair<WorkerNetAddress, Boolean>> nearest = BlockLocationUtils.nearest(this.mTieredIdentity, tieredLocations, this.mContext.getClusterConf());
            if (nearest.isPresent()) {
                dataSource = nearest.get().getFirst();
                BlockInStream.BlockInStreamSource blockInStreamSource = nearest.get().getSecond().booleanValue() ? (this.mContext.hasProcessLocalWorker() ? BlockInStream.BlockInStreamSource.PROCESS_LOCAL : BlockInStream.BlockInStreamSource.NODE_LOCAL) : (dataSourceType = BlockInStream.BlockInStreamSource.REMOTE);
            }
        }
        if (dataSource == null) {
            dataSourceType = BlockInStream.BlockInStreamSource.UFS;
            Preconditions.checkNotNull(policy, "The UFS read location policy is not specified");
            blockWorkerInfo = blockWorkerInfo.stream().filter(workerInfo -> workers.contains(workerInfo.getNetAddress())).collect(Collectors.toList());
            GetWorkerOptions getWorkerOptions = GetWorkerOptions.defaults().setBlockInfo(new BlockInfo().setBlockId(info.getBlockId()).setLength(info.getLength()).setLocations(locations)).setBlockWorkerInfos(blockWorkerInfo);
            dataSource = policy.getWorker(getWorkerOptions).orElseThrow(() -> new UnavailableException(ExceptionMessage.NO_WORKER_AVAILABLE.getMessage(new Object[0])));
            if (this.mContext.hasProcessLocalWorker() && dataSource.equals(this.mContext.getNodeLocalWorker())) {
                dataSourceType = BlockInStream.BlockInStreamSource.PROCESS_LOCAL;
                LOG.debug("Create BlockInStream to read data from UFS through process local worker {}", (Object)dataSource);
            } else {
                LOG.debug("Create BlockInStream to read data from UFS through worker {} (client embedded in local worker process: {},client co-located with worker in different processes: {}, local worker address: {})", new Object[]{dataSource, this.mContext.hasProcessLocalWorker(), this.mContext.hasNodeLocalWorker(), this.mContext.hasNodeLocalWorker() ? this.mContext.getNodeLocalWorker() : "N/A"});
            }
        }
        return new Pair<Object, BlockInStream.BlockInStreamSource>(dataSource, dataSourceType);
    }

    private Set<WorkerNetAddress> handleFailedWorkers(Set<WorkerNetAddress> workers, Map<WorkerNetAddress, Long> failedWorkers) {
        if (workers.isEmpty()) {
            return Collections.emptySet();
        }
        Set<WorkerNetAddress> nonFailed = workers.stream().filter(worker -> !failedWorkers.containsKey(worker)).collect(Collectors.toSet());
        if (nonFailed.isEmpty()) {
            return Collections.singleton(workers.stream().min(Comparator.comparingLong(failedWorkers::get)).get());
        }
        return nonFailed;
    }

    public BlockOutStream getOutStream(long blockId, long blockSize, WorkerNetAddress address, OutStreamOptions options) throws IOException {
        Preconditions.checkNotNull(address, "address");
        LOG.debug("Create BlockOutStream for {} of block size {} at address {}, using options: {}", new Object[]{blockId, blockSize, address, options});
        DataWriter dataWriter = DataWriter.Factory.create(this.mContext, blockId, blockSize, address, options);
        return new BlockOutStream(dataWriter, blockSize, address);
    }

    public BlockOutStream getOutStream(long blockId, long blockSize, OutStreamOptions options) throws IOException {
        int initialReplicas;
        BlockLocationPolicy locationPolicy = Preconditions.checkNotNull(options.getLocationPolicy(), (Object)PreconditionMessage.BLOCK_WRITE_LOCATION_POLICY_UNSPECIFIED);
        GetWorkerOptions workerOptions = GetWorkerOptions.defaults().setBlockInfo(new BlockInfo().setBlockId(blockId).setLength(blockSize)).setBlockWorkerInfos(new ArrayList<BlockWorkerInfo>(this.mContext.getCachedWorkers()));
        int n = initialReplicas = options.getWriteType() == WriteType.ASYNC_THROUGH && options.getReplicationDurable() > options.getReplicationMin() ? options.getReplicationDurable() : options.getReplicationMin();
        if (initialReplicas <= 1) {
            WorkerNetAddress address = locationPolicy.getWorker(workerOptions).orElseThrow(() -> {
                try {
                    if (this.mContext.getCachedWorkers().isEmpty()) {
                        return new UnavailableException(ExceptionMessage.NO_WORKER_AVAILABLE.getMessage(new Object[0]));
                    }
                }
                catch (IOException e) {
                    return e;
                }
                return new UnavailableException(ExceptionMessage.NO_SPACE_FOR_BLOCK_ON_WORKER.getMessage(blockSize));
            });
            return this.getOutStream(blockId, blockSize, address, options);
        }
        HashMap<String, HashSet<BlockWorkerInfo>> blockWorkersByHost = new HashMap<String, HashSet<BlockWorkerInfo>>();
        for (BlockWorkerInfo blockWorker : workerOptions.getBlockWorkerInfos()) {
            String hostName = blockWorker.getNetAddress().getHost();
            if (blockWorkersByHost.containsKey(hostName)) {
                ((Set)blockWorkersByHost.get(hostName)).add(blockWorker);
                continue;
            }
            blockWorkersByHost.put(hostName, Sets.newHashSet(blockWorker));
        }
        ArrayList<WorkerNetAddress> workerAddressList = new ArrayList<WorkerNetAddress>();
        ArrayList<BlockWorkerInfo> updatedInfos = Lists.newArrayList(workerOptions.getBlockWorkerInfos());
        for (int i = 0; i < initialReplicas; ++i) {
            locationPolicy.getWorker(workerOptions).ifPresent(workerAddress -> {
                workerAddressList.add((WorkerNetAddress)workerAddress);
                updatedInfos.removeAll((Collection)blockWorkersByHost.get(workerAddress.getHost()));
                workerOptions.setBlockWorkerInfos(updatedInfos);
            });
        }
        if (workerAddressList.size() < initialReplicas) {
            throw new ResourceExhaustedException(String.format("Not enough workers for replications, %d workers selected but %d required", workerAddressList.size(), initialReplicas));
        }
        return BlockOutStream.createReplicatedBlockOutStream(this.mContext, blockId, blockSize, workerAddressList, options);
    }

    public long getCapacityBytes() throws IOException {
        try (CloseableResource<BlockMasterClient> blockMasterClientResource = this.mContext.acquireBlockMasterClientResource();){
            long l = blockMasterClientResource.get().getCapacityBytes();
            return l;
        }
    }

    public long getUsedBytes() throws IOException {
        try (CloseableResource<BlockMasterClient> blockMasterClientResource = this.mContext.acquireBlockMasterClientResource();){
            long l = blockMasterClientResource.get().getUsedBytes();
            return l;
        }
    }
}

