/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.PageStoreDir;
import alluxio.client.file.cache.store.RocksPageStoreOptions;
import alluxio.exception.PageNotFoundException;
import alluxio.proto.client.Cache;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.com.google.common.collect.ImmutableList;
import alluxio.shaded.client.com.google.protobuf.InvalidProtocolBufferException;
import alluxio.shaded.client.javax.annotation.Nullable;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class RocksPageStore
implements PageStore {
    private static final Logger LOG = LoggerFactory.getLogger(RocksPageStore.class);
    private static final String PAGE_COLUMN = "PAGE";
    private static final byte[] CONF_KEY = "CONF".getBytes();
    private static final int DEFAULT_COLUMN_INDEX = 0;
    private static final int PAGE_COLUMN_INDEX = 1;
    private final long mCapacity;
    private final RocksDB mDb;
    private final ColumnFamilyHandle mDefaultColumnHandle;
    private final ColumnFamilyHandle mPageColumnHandle;
    private final DBOptions mRocksOptions;

    public static RocksPageStore open(RocksPageStoreOptions pageStoreOptions) {
        Preconditions.checkArgument(pageStoreOptions.getMaxPageSize() > 0);
        RocksDB.loadLibrary();
        DBOptions rocksOptions = RocksPageStore.createDbOptions();
        RocksDB db = null;
        ArrayList<ColumnFamilyHandle> columnHandles = new ArrayList<ColumnFamilyHandle>();
        try {
            db = RocksPageStore.openDB(pageStoreOptions, rocksOptions, columnHandles);
        }
        catch (RocksDBException e) {
            try {
                PageStoreDir.clear(pageStoreOptions.mRootDir);
                rocksOptions = RocksPageStore.createDbOptions();
                columnHandles = new ArrayList();
                db = RocksPageStore.openDB(pageStoreOptions, rocksOptions, columnHandles);
            }
            catch (IOException | RocksDBException ex) {
                throw new RuntimeException("Couldn't open rocksDB database", e);
            }
        }
        catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Couldn't open rocksDB database", e);
        }
        return new RocksPageStore(pageStoreOptions, rocksOptions, db, (ColumnFamilyHandle)columnHandles.get(0), (ColumnFamilyHandle)columnHandles.get(1));
    }

    private static DBOptions createDbOptions() {
        DBOptions rocksOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true);
        return rocksOptions;
    }

    private static RocksDB openDB(RocksPageStoreOptions pageStoreOptions, DBOptions rocksOptions, List<ColumnFamilyHandle> columnHandles) throws RocksDBException, InvalidProtocolBufferException {
        Cache.PRocksPageStoreOptions persistedOptions;
        ImmutableList<ColumnFamilyDescriptor> columnDescriptors = ImmutableList.of(new ColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY), new ColumnFamilyDescriptor(PAGE_COLUMN.getBytes(), new ColumnFamilyOptions().setWriteBufferSize(pageStoreOptions.getWriteBufferSize()).setCompressionType(pageStoreOptions.getCompressionType())));
        RocksDB db = RocksDB.open(rocksOptions, pageStoreOptions.getRootDir().toString(), columnDescriptors, columnHandles);
        byte[] confData = db.get(columnHandles.get(0), CONF_KEY);
        Cache.PRocksPageStoreOptions pOptions = pageStoreOptions.toProto();
        if (confData != null && !(persistedOptions = Cache.PRocksPageStoreOptions.parseFrom(confData)).equals(pOptions)) {
            db.close();
            rocksOptions.close();
            throw new RocksDBException("Inconsistent configuration for RocksPageStore");
        }
        db.put(columnHandles.get(0), CONF_KEY, pOptions.toByteArray());
        return db;
    }

    private RocksPageStore(RocksPageStoreOptions pageStoreOptions, DBOptions rocksOptions, RocksDB rocksDB, ColumnFamilyHandle defaultColumnHandle, ColumnFamilyHandle pageColumnHandle) {
        this.mCapacity = (long)((double)pageStoreOptions.getCacheSize() / (1.0 + pageStoreOptions.getOverheadRatio()));
        this.mRocksOptions = rocksOptions;
        this.mDb = rocksDB;
        this.mDefaultColumnHandle = defaultColumnHandle;
        this.mPageColumnHandle = pageColumnHandle;
    }

    @Override
    public void put(PageId pageId, byte[] page, boolean isTemporary) throws IOException {
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId);
            this.mDb.put(this.mPageColumnHandle, key, page);
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to store page", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public int get(PageId pageId, int pageOffset, int bytesToRead, byte[] buffer, int bufferOffset, boolean isTemporary) throws IOException, PageNotFoundException {
        Preconditions.checkArgument(pageOffset >= 0, "page offset should be non-negative");
        try {
            byte[] page = this.mDb.get(this.mPageColumnHandle, RocksPageStore.getKeyFromPageId(pageId));
            if (page == null) {
                throw new PageNotFoundException(new String(RocksPageStore.getKeyFromPageId(pageId)));
            }
            Preconditions.checkArgument(pageOffset <= page.length, "page offset %s exceeded page size %s", pageOffset, page.length);
            try (ByteArrayInputStream bais = new ByteArrayInputStream(page);){
                int bytes;
                int bytesSkipped = (int)bais.skip(pageOffset);
                if (pageOffset != bytesSkipped) {
                    throw new IOException(String.format("Failed to read page %s from offset %s: %s bytes skipped", pageId, pageOffset, bytesSkipped));
                }
                int bytesRead = 0;
                int bytesLeft = Math.min(page.length - pageOffset, buffer.length - bufferOffset);
                for (bytesLeft = Math.min(bytesLeft, bytesToRead); bytesLeft >= 0 && (bytes = bais.read(buffer, bufferOffset + bytesRead, bytesLeft)) > 0; bytesRead += bytes, bytesLeft -= bytes) {
                }
                int n = bytesRead;
                return n;
            }
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to retrieve page", e);
        }
    }

    @Override
    public void delete(PageId pageId) throws PageNotFoundException {
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId);
            this.mDb.delete(this.mPageColumnHandle, key);
        }
        catch (RocksDBException e) {
            throw new PageNotFoundException("Failed to remove page", e);
        }
    }

    @Override
    public void close() {
        LOG.info("Closing RocksPageStore and recycling all RocksDB JNI objects");
        this.mDb.close();
        this.mRocksOptions.close();
        this.mDefaultColumnHandle.close();
        this.mPageColumnHandle.close();
        LOG.info("RocksPageStore closed");
    }

    static byte[] getKeyFromPageId(PageId pageId) {
        byte[] fileId = pageId.getFileId().getBytes();
        ByteBuffer buf = ByteBuffer.allocate(8 + fileId.length);
        buf.putLong(pageId.getPageIndex());
        buf.put(fileId);
        return buf.array();
    }

    @Nullable
    static PageId getPageIdFromKey(byte[] key) {
        if (key.length < 8) {
            return null;
        }
        ByteBuffer buf = ByteBuffer.wrap(key);
        long pageIndex = buf.getLong();
        String fileId = Charset.defaultCharset().decode(buf).toString();
        return new PageId(fileId, pageIndex);
    }

    public RocksIterator createNewInterator() {
        return this.mDb.newIterator(this.mPageColumnHandle);
    }
}

