/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.math.expr;

import com.google.common.collect.ImmutableSet;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.DoubleBinaryOperator;
import java.util.function.LongBinaryOperator;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.druid.common.config.NullHandling;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.RE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.math.expr.BinEqExpr;
import org.apache.druid.math.expr.Expr;
import org.apache.druid.math.expr.ExprEval;
import org.apache.druid.math.expr.ExprType;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;

public interface Function {
    public String name();

    public ExprEval apply(List<Expr> var1, Expr.ObjectBinding var2);

    default public Set<Expr> getScalarInputs(List<Expr> args) {
        return ImmutableSet.copyOf(args);
    }

    default public Set<Expr> getArrayInputs(List<Expr> args) {
        return Collections.emptySet();
    }

    default public boolean hasArrayInputs() {
        return false;
    }

    default public boolean hasArrayOutput() {
        return false;
    }

    public void validateArguments(List<Expr> var1);

    public static class ArrayPrependFunction
    implements Function {
        @Override
        public String name() {
            return "array_prepend";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(1));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval scalarExpr = args.get(0).eval(bindings);
            ExprEval arrayExpr = args.get(1).eval(bindings);
            if (arrayExpr.asArray() == null) {
                return ExprEval.of(null);
            }
            switch (arrayExpr.type()) {
                case STRING: 
                case STRING_ARRAY: {
                    return ExprEval.ofStringArray((String[])this.prepend(scalarExpr.asString(), arrayExpr.asStringArray()).toArray(String[]::new));
                }
                case LONG: 
                case LONG_ARRAY: {
                    return ExprEval.ofLongArray((Long[])this.prepend(scalarExpr.isNumericNull() ? null : Long.valueOf(scalarExpr.asLong()), arrayExpr.asLongArray()).toArray(Long[]::new));
                }
                case DOUBLE: 
                case DOUBLE_ARRAY: {
                    return ExprEval.ofDoubleArray((Double[])this.prepend(scalarExpr.isNumericNull() ? null : Double.valueOf(scalarExpr.asDouble()), arrayExpr.asDoubleArray()).toArray(Double[]::new));
                }
            }
            throw new RE("Unable to prepend to unknown type %s", new Object[]{arrayExpr.type()});
        }

        private <T> Stream<T> prepend(T val, T[] array) {
            ArrayList<T> l = new ArrayList<T>(Arrays.asList(array));
            l.add(0, val);
            return l.stream();
        }
    }

    public static class ArraySliceFunction
    implements Function {
        @Override
        public String name() {
            return "array_slice";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2 && args.size() != 3) {
                throw new IAE("Function[%s] needs 2 or 3 arguments", this.name());
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            if (args.size() == 3) {
                return ImmutableSet.of((Object)args.get(1), (Object)args.get(2));
            }
            return ImmutableSet.of((Object)args.get(1));
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            Object[] array = expr.asArray();
            if (array == null) {
                return ExprEval.of(null);
            }
            int start = args.get(1).eval(bindings).asInt();
            int end = array.length;
            if (args.size() == 3) {
                end = args.get(2).eval(bindings).asInt();
            }
            if (start < 0 || start > array.length || start > end) {
                return ExprEval.of(null);
            }
            switch (expr.type()) {
                case STRING: 
                case STRING_ARRAY: {
                    return ExprEval.ofStringArray(Arrays.copyOfRange(expr.asStringArray(), start, end));
                }
                case LONG: 
                case LONG_ARRAY: {
                    return ExprEval.ofLongArray(Arrays.copyOfRange(expr.asLongArray(), start, end));
                }
                case DOUBLE: 
                case DOUBLE_ARRAY: {
                    return ExprEval.ofDoubleArray(Arrays.copyOfRange(expr.asDoubleArray(), start, end));
                }
            }
            throw new RE("Unable to slice to unknown type %s", new Object[]{expr.type()});
        }
    }

    public static class ArrayOverlapFunction
    extends ArraysFunction {
        @Override
        public String name() {
            return "array_overlap";
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            List<Object> array2 = Arrays.asList(rhsExpr.asArray());
            boolean any = false;
            for (Object check : array1) {
                any |= array2.contains(check);
            }
            return ExprEval.of(any, ExprType.LONG);
        }
    }

    public static class ArrayContainsFunction
    extends ArraysFunction {
        @Override
        public String name() {
            return "array_contains";
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            Object[] array2 = rhsExpr.asArray();
            return ExprEval.of(Arrays.asList(array1).containsAll(Arrays.asList(array2)), ExprType.LONG);
        }
    }

    public static class ArrayConcatFunction
    extends ArraysFunction {
        @Override
        public String name() {
            return "array_concat";
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        ExprEval doApply(ExprEval lhsExpr, ExprEval rhsExpr) {
            Object[] array1 = lhsExpr.asArray();
            Object[] array2 = rhsExpr.asArray();
            if (array1 == null) {
                return ExprEval.of(null);
            }
            if (array2 == null) {
                return lhsExpr;
            }
            switch (lhsExpr.type()) {
                case STRING: 
                case STRING_ARRAY: {
                    return ExprEval.ofStringArray((String[])this.cat(lhsExpr.asStringArray(), rhsExpr.asStringArray()).toArray(String[]::new));
                }
                case LONG: 
                case LONG_ARRAY: {
                    return ExprEval.ofLongArray((Long[])this.cat(lhsExpr.asLongArray(), rhsExpr.asLongArray()).toArray(Long[]::new));
                }
                case DOUBLE: 
                case DOUBLE_ARRAY: {
                    return ExprEval.ofDoubleArray((Double[])this.cat(lhsExpr.asDoubleArray(), rhsExpr.asDoubleArray()).toArray(Double[]::new));
                }
            }
            throw new RE("Unable to concatenate to unknown type %s", new Object[]{lhsExpr.type()});
        }

        private <T> Stream<T> cat(T[] array1, T[] array2) {
            ArrayList<T> l = new ArrayList<T>(Arrays.asList(array1));
            l.addAll(Arrays.asList(array2));
            return l.stream();
        }
    }

    public static class ArrayAppendFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_append";
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            switch (arrayExpr.type()) {
                case STRING: 
                case STRING_ARRAY: {
                    return ExprEval.ofStringArray((String[])this.append(arrayExpr.asStringArray(), scalarExpr.asString()).toArray(String[]::new));
                }
                case LONG: 
                case LONG_ARRAY: {
                    return ExprEval.ofLongArray((Long[])this.append(arrayExpr.asLongArray(), scalarExpr.isNumericNull() ? null : Long.valueOf(scalarExpr.asLong())).toArray(Long[]::new));
                }
                case DOUBLE: 
                case DOUBLE_ARRAY: {
                    return ExprEval.ofDoubleArray((Double[])this.append(arrayExpr.asDoubleArray(), scalarExpr.isNumericNull() ? null : Double.valueOf(scalarExpr.asDouble())).toArray(Double[]::new));
                }
            }
            throw new RE("Unable to append to unknown type %s", new Object[]{arrayExpr.type()});
        }

        private <T> Stream<T> append(T[] array, T val) {
            ArrayList<T> l = new ArrayList<T>(Arrays.asList(array));
            l.add(val);
            return l.stream();
        }
    }

    public static class ArrayOrdinalOfFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_ordinal_of";
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            Object[] array = arrayExpr.asArray();
            switch (scalarExpr.type()) {
                case DOUBLE: 
                case LONG: 
                case STRING: {
                    int index = -1;
                    for (int i = 0; i < array.length; ++i) {
                        if (!Objects.equals(array[i], scalarExpr.value())) continue;
                        index = i;
                        break;
                    }
                    return index < 0 ? ExprEval.ofLong(NullHandling.replaceWithDefault() ? Integer.valueOf(-1) : null) : ExprEval.ofLong(index + 1);
                }
            }
            throw new IAE("Function[%s] 2nd argument must be a a scalar type", this.name());
        }
    }

    public static class ArrayOffsetOfFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_offset_of";
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            Object[] array = arrayExpr.asArray();
            switch (scalarExpr.type()) {
                case DOUBLE: 
                case LONG: 
                case STRING: {
                    int index = -1;
                    for (int i = 0; i < array.length; ++i) {
                        if (!Objects.equals(array[i], scalarExpr.value())) continue;
                        index = i;
                        break;
                    }
                    return index < 0 ? ExprEval.ofLong(NullHandling.replaceWithDefault() ? Integer.valueOf(-1) : null) : ExprEval.ofLong(index);
                }
            }
            throw new IAE("Function[%s] 2nd argument must be a a scalar type", this.name());
        }
    }

    public static class ArrayOrdinalFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_ordinal";
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            int position;
            Object[] array = arrayExpr.asArray();
            if (array.length > (position = scalarExpr.asInt() - 1)) {
                return ExprEval.bestEffortOf(array[position]);
            }
            return ExprEval.of(null);
        }
    }

    public static class ArrayOffsetFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_offset";
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            int position;
            Object[] array = arrayExpr.asArray();
            if (array.length > (position = scalarExpr.asInt())) {
                return ExprEval.bestEffortOf(array[position]);
            }
            return ExprEval.of(null);
        }
    }

    public static class ArrayToStringFunction
    extends ArrayScalarFunction {
        @Override
        public String name() {
            return "array_to_string";
        }

        @Override
        ExprEval doApply(ExprEval arrayExpr, ExprEval scalarExpr) {
            String join = scalarExpr.asString();
            Object[] raw = arrayExpr.asArray();
            if (raw == null || raw.length == 1 && raw[0] == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(Arrays.stream(raw).map(String::valueOf).collect(Collectors.joining(join != null ? join : "")));
        }
    }

    public static class StringToArrayFunction
    implements Function {
        @Override
        public String name() {
            return "string_to_array";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 argument", this.name());
            }
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            String arrayString = expr.asString();
            if (arrayString == null) {
                return ExprEval.of(null);
            }
            String split = args.get(1).eval(bindings).asString();
            return ExprEval.ofStringArray(arrayString.split(split != null ? split : ""));
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }
    }

    public static class ArrayLengthFunction
    implements Function {
        @Override
        public String name() {
            return "array_length";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            Object[] array = expr.asArray();
            if (array == null) {
                return ExprEval.of(null);
            }
            return ExprEval.ofLong(array.length);
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return Collections.emptySet();
        }
    }

    public static class ArrayConstructorFunction
    implements Function {
        @Override
        public String name() {
            return "array";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            int length = args.size();
            String[] stringsOut = null;
            Long[] longsOut = null;
            Double[] doublesOut = null;
            Enum elementType = null;
            for (int i = 0; i < length; ++i) {
                ExprEval evaluated = args.get(i).eval(bindings);
                if (elementType == null) {
                    elementType = evaluated.type();
                    switch (1.$SwitchMap$org$apache$druid$math$expr$ExprType[elementType.ordinal()]) {
                        case 3: {
                            stringsOut = new String[length];
                            break;
                        }
                        case 2: {
                            longsOut = new Long[length];
                            break;
                        }
                        case 1: {
                            doublesOut = new Double[length];
                            break;
                        }
                        default: {
                            throw new RE("Unhandled array constructor element type [%s]", elementType);
                        }
                    }
                }
                ArrayConstructorFunction.setArrayOutputElement(stringsOut, longsOut, doublesOut, (ExprType)elementType, i, evaluated);
            }
            switch (1.$SwitchMap$org$apache$druid$math$expr$ExprType[elementType.ordinal()]) {
                case 3: {
                    return ExprEval.ofStringArray(stringsOut);
                }
                case 2: {
                    return ExprEval.ofLongArray(longsOut);
                }
                case 1: {
                    return ExprEval.ofDoubleArray(doublesOut);
                }
            }
            throw new RE("Unhandled array constructor element type [%s]", elementType);
        }

        static void setArrayOutputElement(String[] stringsOut, Long[] longsOut, Double[] doublesOut, ExprType elementType, int i, ExprEval evaluated) {
            switch (elementType) {
                case STRING: {
                    stringsOut[i] = evaluated.asString();
                    break;
                }
                case LONG: {
                    longsOut[i] = evaluated.isNumericNull() ? null : Long.valueOf(evaluated.asLong());
                    break;
                }
                case DOUBLE: {
                    doublesOut[i] = evaluated.isNumericNull() ? null : Double.valueOf(evaluated.asDouble());
                }
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return Collections.emptySet();
        }

        @Override
        public boolean hasArrayOutput() {
            return true;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.isEmpty()) {
                throw new IAE("Function[%s] needs at least 1 argument", this.name());
            }
        }
    }

    public static class SubMonthFunc
    implements Function {
        @Override
        public String name() {
            return "subtract_months";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Long left = args.get(0).eval(bindings).asLong();
            Long right = args.get(1).eval(bindings).asLong();
            DateTimeZone timeZone = DateTimes.inferTzFromString(args.get(2).eval(bindings).asString());
            if (left == null || right == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(DateTimes.subMonths(right, left, timeZone));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class RpadFunc
    implements Function {
        @Override
        public String name() {
            return "rpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.rpad(base, len, pad));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class LpadFunc
    implements Function {
        @Override
        public String name() {
            return "lpad";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String base = args.get(0).eval(bindings).asString();
            int len = args.get(1).eval(bindings).asInt();
            String pad = args.get(2).eval(bindings).asString();
            if (base == null || pad == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(len == 0 ? NullHandling.defaultStringValue() : StringUtils.lpad(base, len, pad));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class IsNotNullFunc
    implements Function {
        @Override
        public String name() {
            return "notnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.of(expr.value() != null, ExprType.LONG);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }
    }

    public static class IsNullFunc
    implements Function {
        @Override
        public String name() {
            return "isnull";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval expr = args.get(0).eval(bindings);
            return ExprEval.of(expr.value() == null, ExprType.LONG);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }
    }

    public static class RepeatFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "repeat";
        }

        @Override
        protected ExprEval eval(String x, int y) {
            return ExprEval.of(y < 1 ? NullHandling.defaultStringValue() : StringUtils.repeat(x, y));
        }
    }

    public static class ReverseFunc
    extends UnivariateFunction {
        @Override
        public String name() {
            return "reverse";
        }

        @Override
        protected ExprEval eval(ExprEval param) {
            if (param.type() != ExprType.STRING) {
                throw new IAE("Function[%s] needs a string argument", this.name());
            }
            String arg = param.asString();
            return ExprEval.of(arg == null ? NullHandling.defaultStringValue() : new StringBuilder(arg).reverse().toString());
        }
    }

    public static class UpperFunc
    implements Function {
        @Override
        public String name() {
            return "upper";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toUpperCase(arg));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }
    }

    public static class LowerFunc
    implements Function {
        @Override
        public String name() {
            return "lower";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.toLowerCase(arg));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }
    }

    public static class ReplaceFunc
    implements Function {
        @Override
        public String name() {
            return "replace";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            String pattern = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            String replacement = NullHandling.nullToEmptyIfNeeded(args.get(2).eval(bindings).asString());
            if (arg == null) {
                return ExprEval.of(NullHandling.defaultStringValue());
            }
            return ExprEval.of(StringUtils.replace(arg, pattern, replacement));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class LeftFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "left";
        }

        @Override
        protected ExprEval eval(@Nullable String x, int y) {
            if (y < 0) {
                throw new IAE("Function[%s] needs a postive integer as second argument", this.name());
            }
            if (x == null) {
                return ExprEval.of(null);
            }
            return ExprEval.of(y < x.length() ? x.substring(0, y) : x);
        }
    }

    public static class RightFunc
    extends StringLongFunction {
        @Override
        public String name() {
            return "right";
        }

        @Override
        protected ExprEval eval(@Nullable String x, int y) {
            if (y < 0) {
                throw new IAE("Function[%s] needs a postive integer as second argument", this.name());
            }
            if (x == null) {
                return ExprEval.of(null);
            }
            int len = x.length();
            return ExprEval.of(y < len ? x.substring(len - y) : x);
        }
    }

    public static class SubstringFunc
    implements Function {
        @Override
        public String name() {
            return "substring";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            if (arg == null) {
                return ExprEval.of(null);
            }
            int index = args.get(1).eval(bindings).asInt();
            int length = args.get(2).eval(bindings).asInt();
            if (index < arg.length()) {
                if (length >= 0) {
                    return ExprEval.of(arg.substring(index, Math.min(index + length, arg.length())));
                }
                return ExprEval.of(arg.substring(index));
            }
            return ExprEval.of(NullHandling.defaultStringValue());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class StrposFunc
    implements Function {
        @Override
        public String name() {
            return "strpos";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String haystack = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            String needle = NullHandling.nullToEmptyIfNeeded(args.get(1).eval(bindings).asString());
            if (haystack == null || needle == null) {
                return ExprEval.of(null);
            }
            int fromIndex = args.size() >= 3 ? args.get(2).eval(bindings).asInt() : 0;
            return ExprEval.of(haystack.indexOf(needle, fromIndex));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() < 2 || args.size() > 3) {
                throw new IAE("Function[%s] needs 2 or 3 arguments", this.name());
            }
        }
    }

    public static class StringFormatFunc
    implements Function {
        @Override
        public String name() {
            return "format";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String formatString = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (formatString == null) {
                return ExprEval.of(null);
            }
            Object[] formatArgs = new Object[args.size() - 1];
            for (int i = 1; i < args.size(); ++i) {
                formatArgs[i - 1] = args.get(i).eval(bindings).value();
            }
            return ExprEval.of(StringUtils.nonStrictFormat(formatString, formatArgs));
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() < 1) {
                throw new IAE("Function[%s] needs 1 or more arguments", this.name());
            }
        }
    }

    public static class StrlenFunc
    implements Function {
        @Override
        public String name() {
            return "strlen";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            String arg = args.get(0).eval(bindings).asString();
            return arg == null ? ExprEval.ofLong(NullHandling.defaultLongValue()) : ExprEval.of(arg.length());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }
    }

    public static class ConcatFunc
    implements Function {
        @Override
        public String name() {
            return "concat";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.size() == 0) {
                return ExprEval.of(null);
            }
            String first = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (first == null) {
                return ExprEval.of(null);
            }
            StringBuilder builder = new StringBuilder(first);
            for (int i = 1; i < args.size(); ++i) {
                String s = NullHandling.nullToEmptyIfNeeded(args.get(i).eval(bindings).asString());
                if (s == null) {
                    return ExprEval.of(null);
                }
                builder.append(s);
            }
            return ExprEval.of(builder.toString());
        }

        @Override
        public void validateArguments(List<Expr> args) {
        }
    }

    public static class NvlFunc
    implements Function {
        @Override
        public String name() {
            return "nvl";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval eval = args.get(0).eval(bindings);
            return eval.value() == null ? args.get(1).eval(bindings) : eval;
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
        }
    }

    public static class UnixTimestampFunc
    extends TimestampFromEpochFunc {
        @Override
        public String name() {
            return "unix_timestamp";
        }

        @Override
        protected final ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis() / 1000L);
        }
    }

    public static class TimestampFromEpochFunc
    implements Function {
        @Override
        public String name() {
            return "timestamp";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            DateTime date;
            ExprEval value = args.get(0).eval(bindings);
            if (value.type() != ExprType.STRING) {
                throw new IAE("first argument should be string type but got %s type", new Object[]{value.type()});
            }
            DateTimes.UtcFormatter formatter = DateTimes.ISO_DATE_OPTIONAL_TIME;
            if (args.size() > 1) {
                ExprEval format = args.get(1).eval(bindings);
                if (format.type() != ExprType.STRING) {
                    throw new IAE("second argument should be string type but got %s type", new Object[]{format.type()});
                }
                formatter = DateTimes.wrapFormatter(DateTimeFormat.forPattern((String)format.asString()));
            }
            try {
                date = formatter.parse(value.asString());
            }
            catch (IllegalArgumentException e) {
                throw new IAE(e, "invalid value %s", value.asString());
            }
            return this.toValue(date);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1 && args.size() != 2) {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
        }

        protected ExprEval toValue(DateTime date) {
            return ExprEval.of(date.getMillis());
        }
    }

    public static class CastFunc
    extends BivariateFunction {
        @Override
        public String name() {
            return "cast";
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            ExprType castTo;
            if (NullHandling.sqlCompatible() && x.value() == null) {
                return ExprEval.of(null);
            }
            try {
                castTo = ExprType.valueOf(StringUtils.toUpperCase(y.asString()));
            }
            catch (IllegalArgumentException e) {
                throw new IAE("invalid type '%s'", y.asString());
            }
            return x.castTo(castTo);
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            if (args.get(1).isLiteral()) {
                ExprType castTo = ExprType.valueOf(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString()));
                switch (castTo) {
                    case LONG_ARRAY: 
                    case DOUBLE_ARRAY: 
                    case STRING_ARRAY: {
                        return Collections.emptySet();
                    }
                }
                return ImmutableSet.of((Object)args.get(0));
            }
            return Collections.emptySet();
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            if (args.get(1).isLiteral()) {
                ExprType castTo = ExprType.valueOf(StringUtils.toUpperCase(args.get(1).getLiteralValue().toString()));
                switch (castTo) {
                    case DOUBLE: 
                    case LONG: 
                    case STRING: {
                        return Collections.emptySet();
                    }
                }
                return ImmutableSet.of((Object)args.get(0));
            }
            return Collections.emptySet();
        }
    }

    public static class CaseSimpleFunc
    implements Function {
        @Override
        public String name() {
            return "case_simple";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            for (int i = 1; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!new BinEqExpr("==", args.get(0), args.get(i)).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() < 3) {
                throw new IAE("Function[%s] must have at least 3 arguments", this.name());
            }
        }
    }

    public static class CaseSearchedFunc
    implements Function {
        @Override
        public String name() {
            return "case_searched";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            for (int i = 0; i < args.size(); i += 2) {
                if (i == args.size() - 1) {
                    return args.get(i).eval(bindings);
                }
                if (!args.get(i).eval(bindings).asBoolean()) continue;
                return args.get(i + 1).eval(bindings);
            }
            return ExprEval.of(null);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() < 2) {
                throw new IAE("Function[%s] must have at least 2 arguments", this.name());
            }
        }
    }

    public static class ConditionFunc
    implements Function {
        @Override
        public String name() {
            return "if";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval x = args.get(0).eval(bindings);
            return x.asBoolean() ? args.get(1).eval(bindings) : args.get(2).eval(bindings);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 3) {
                throw new IAE("Function[%s] needs 3 arguments", this.name());
            }
        }
    }

    public static class Scalb
    extends BivariateFunction {
        @Override
        public String name() {
            return "scalb";
        }

        @Override
        protected ExprEval eval(ExprEval x, ExprEval y) {
            return ExprEval.of(Math.scalb(x.asDouble(), y.asInt()));
        }
    }

    public static class Pow
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "pow";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.pow(x, y));
        }
    }

    public static class NextAfter
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "nextAfter";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.nextAfter(x, y));
        }
    }

    public static abstract class ReduceFunc
    implements Function {
        private final DoubleBinaryOperator doubleReducer;
        private final LongBinaryOperator longReducer;
        private final BinaryOperator<String> stringReducer;

        ReduceFunc(DoubleBinaryOperator doubleReducer, LongBinaryOperator longReducer, BinaryOperator<String> stringReducer) {
            this.doubleReducer = doubleReducer;
            this.longReducer = longReducer;
            this.stringReducer = stringReducer;
        }

        @Override
        public void validateArguments(List<Expr> args) {
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            if (args.isEmpty()) {
                return ExprEval.of(null);
            }
            ExprAnalysis exprAnalysis = this.analyzeExprs(args, bindings);
            if (exprAnalysis.exprEvals.isEmpty()) {
                return ExprEval.of(null);
            }
            Stream exprEvalStream = exprAnalysis.exprEvals.stream();
            switch (exprAnalysis.comparisonType) {
                case DOUBLE: {
                    return ExprEval.of(exprEvalStream.mapToDouble(ExprEval::asDouble).reduce(this.doubleReducer).getAsDouble());
                }
                case LONG: {
                    return ExprEval.of(exprEvalStream.mapToLong(ExprEval::asLong).reduce(this.longReducer).getAsLong());
                }
            }
            return ExprEval.of(exprEvalStream.map(ExprEval::asString).reduce(this.stringReducer).get());
        }

        private ExprAnalysis analyzeExprs(List<Expr> exprs, Expr.ObjectBinding bindings) {
            EnumSet<ExprType> presentTypes = EnumSet.noneOf(ExprType.class);
            ArrayList exprEvals = new ArrayList();
            for (Expr expr : exprs) {
                ExprEval exprEval = expr.eval(bindings);
                ExprType exprType = exprEval.type();
                if (this.isValidType(exprType)) {
                    presentTypes.add(exprType);
                }
                if (exprEval.value() == null) continue;
                exprEvals.add(exprEval);
            }
            ExprType comparisonType = ReduceFunc.getComparisionType(presentTypes);
            return new ExprAnalysis(comparisonType, exprEvals);
        }

        private boolean isValidType(ExprType exprType) {
            switch (exprType) {
                case DOUBLE: 
                case LONG: 
                case STRING: {
                    return true;
                }
            }
            throw new IAE("Function[%s] does not accept %s types", new Object[]{this.name(), exprType});
        }

        private static ExprType getComparisionType(Set<ExprType> exprTypes) {
            if (exprTypes.contains((Object)ExprType.STRING)) {
                return ExprType.STRING;
            }
            if (exprTypes.contains((Object)ExprType.DOUBLE)) {
                return ExprType.DOUBLE;
            }
            return ExprType.LONG;
        }

        private static class ExprAnalysis {
            final ExprType comparisonType;
            final List<ExprEval<?>> exprEvals;

            ExprAnalysis(ExprType comparisonType, List<ExprEval<?>> exprEvals) {
                this.comparisonType = comparisonType;
                this.exprEvals = exprEvals;
            }
        }
    }

    public static class LeastFunc
    extends ReduceFunc {
        public static final String NAME = "least";

        public LeastFunc() {
            super(Math::min, Math::min, BinaryOperator.minBy(Comparator.naturalOrder()));
        }

        @Override
        public String name() {
            return NAME;
        }
    }

    public static class GreatestFunc
    extends ReduceFunc {
        public static final String NAME = "greatest";

        public GreatestFunc() {
            super(Math::max, Math::max, BinaryOperator.maxBy(Comparator.naturalOrder()));
        }

        @Override
        public String name() {
            return NAME;
        }
    }

    public static class Min
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "min";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.min(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.min(x, y));
        }
    }

    public static class Max
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "max";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(Math.max(x, y));
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.max(x, y));
        }
    }

    public static class Remainder
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "remainder";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.IEEEremainder(x, y));
        }
    }

    public static class Hypot
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "hypot";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.hypot(x, y));
        }
    }

    public static class CopySign
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "copySign";
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of(Math.copySign(x, y));
        }
    }

    public static class Atan2
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "atan2";
        }

        @Override
        protected ExprEval eval(double y, double x) {
            return ExprEval.of(Math.atan2(y, x));
        }
    }

    public static class Ulp
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "ulp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ulp(param));
        }
    }

    public static class ToRadians
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "toRadians";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toRadians(param));
        }
    }

    public static class ToDegrees
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "toDegrees";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.toDegrees(param));
        }
    }

    public static class Tanh
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "tanh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tanh(param));
        }
    }

    public static class Tan
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "tan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.tan(param));
        }
    }

    public static class Sqrt
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "sqrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sqrt(param));
        }
    }

    public static class Sinh
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "sinh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sinh(param));
        }
    }

    public static class Sin
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "sin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.sin(param));
        }
    }

    public static class Signum
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "signum";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.signum(param));
        }
    }

    public static class Round
    implements Function {
        private static final BigDecimal MAX_FINITE_VALUE = BigDecimal.valueOf(Double.MAX_VALUE);
        private static final BigDecimal MIN_FINITE_VALUE = BigDecimal.valueOf(-1.7976931348623157E308);

        @Override
        public String name() {
            return "round";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval value1 = args.get(0).eval(bindings);
            if (value1.type() != ExprType.LONG && value1.type() != ExprType.DOUBLE) {
                throw new IAE("The first argument to the function[%s] should be integer or double type but got the type: %s", new Object[]{this.name(), value1.type()});
            }
            if (args.size() == 1) {
                return this.eval(value1);
            }
            ExprEval value2 = args.get(1).eval(bindings);
            if (value2.type() != ExprType.LONG) {
                throw new IAE("The second argument to the function[%s] should be integer type but got the type: %s", new Object[]{this.name(), value2.type()});
            }
            return this.eval(value1, value2.asInt());
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1 && args.size() != 2) {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
        }

        private ExprEval eval(ExprEval param) {
            return this.eval(param, 0);
        }

        private ExprEval eval(ExprEval param, int scale) {
            if (param.type() == ExprType.LONG) {
                return ExprEval.of(BigDecimal.valueOf(param.asLong()).setScale(scale, RoundingMode.HALF_UP).longValue());
            }
            if (param.type() == ExprType.DOUBLE) {
                BigDecimal decimal = Round.safeGetFromDouble(param.asDouble());
                return ExprEval.of(decimal.setScale(scale, RoundingMode.HALF_UP).doubleValue());
            }
            return ExprEval.of(null);
        }

        private static BigDecimal safeGetFromDouble(double val) {
            if (Double.isNaN(val)) {
                return BigDecimal.ZERO;
            }
            if (val == Double.POSITIVE_INFINITY) {
                return MAX_FINITE_VALUE;
            }
            if (val == Double.NEGATIVE_INFINITY) {
                return MIN_FINITE_VALUE;
            }
            return BigDecimal.valueOf(val);
        }
    }

    public static class Rint
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "rint";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.rint(param));
        }
    }

    public static class NextUp
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "nextUp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.nextUp(param));
        }
    }

    public static class Log1p
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "log1p";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log1p(param));
        }
    }

    public static class Log10
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "log10";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log10(param));
        }
    }

    public static class Log
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "log";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.log(param));
        }
    }

    public static class GetExponent
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "getExponent";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.getExponent(param));
        }
    }

    public static class Floor
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "floor";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.floor(param));
        }
    }

    public static class Expm1
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "expm1";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.expm1(param));
        }
    }

    public static class Exp
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "exp";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.exp(param));
        }
    }

    public static class Div
    extends BivariateMathFunction {
        @Override
        public String name() {
            return "div";
        }

        @Override
        protected ExprEval eval(long x, long y) {
            return ExprEval.of(x / y);
        }

        @Override
        protected ExprEval eval(double x, double y) {
            return ExprEval.of((long)(x / y));
        }
    }

    public static class Cot
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "cot";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param) / Math.sin(param));
        }
    }

    public static class Cosh
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "cosh";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cosh(param));
        }
    }

    public static class Cos
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "cos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cos(param));
        }
    }

    public static class Ceil
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "ceil";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.ceil(param));
        }
    }

    public static class Cbrt
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "cbrt";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.cbrt(param));
        }
    }

    public static class Atan
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "atan";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.atan(param));
        }
    }

    public static class Asin
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "asin";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.asin(param));
        }
    }

    public static class Acos
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "acos";
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.acos(param));
        }
    }

    public static class Abs
    extends UnivariateMathFunction {
        @Override
        public String name() {
            return "abs";
        }

        @Override
        protected ExprEval eval(long param) {
            return ExprEval.of(Math.abs(param));
        }

        @Override
        protected ExprEval eval(double param) {
            return ExprEval.of(Math.abs(param));
        }
    }

    public static class Pi
    implements Function {
        private static final double PI = Math.PI;

        @Override
        public String name() {
            return "pi";
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            return ExprEval.of(Math.PI);
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() > 0) {
                throw new IAE("Function[%s] needs 0 argument", this.name());
            }
        }
    }

    public static class ParseLong
    implements Function {
        @Override
        public String name() {
            return "parse_long";
        }

        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1 && args.size() != 2) {
                throw new IAE("Function[%s] needs 1 or 2 arguments", this.name());
            }
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            long retVal;
            int radix = args.size() == 1 ? 10 : args.get(1).eval(bindings).asInt();
            String input = NullHandling.nullToEmptyIfNeeded(args.get(0).eval(bindings).asString());
            if (input == null) {
                return ExprEval.ofLong(null);
            }
            try {
                retVal = radix == 16 && (input.startsWith("0x") || input.startsWith("0X")) ? Long.parseLong(input.substring(2), radix) : Long.parseLong(input, radix);
            }
            catch (NumberFormatException e) {
                return ExprEval.ofLong(null);
            }
            return ExprEval.of(retVal);
        }
    }

    public static abstract class ArraysFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return Collections.emptySet();
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.copyOf(args);
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval arrayExpr1 = args.get(0).eval(bindings);
            ExprEval arrayExpr2 = args.get(1).eval(bindings);
            if (arrayExpr1.asArray() == null || arrayExpr2.asArray() == null) {
                return ExprEval.of(null);
            }
            return this.doApply(arrayExpr1, arrayExpr2);
        }

        abstract ExprEval doApply(ExprEval var1, ExprEval var2);
    }

    public static abstract class ArrayScalarFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 argument", this.name());
            }
        }

        @Override
        public Set<Expr> getScalarInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(1));
        }

        @Override
        public Set<Expr> getArrayInputs(List<Expr> args) {
            return ImmutableSet.of((Object)args.get(0));
        }

        @Override
        public boolean hasArrayInputs() {
            return true;
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            ExprEval arrayExpr = args.get(0).eval(bindings);
            ExprEval scalarExpr = args.get(1).eval(bindings);
            if (arrayExpr.asArray() == null) {
                return ExprEval.of(null);
            }
            return this.doApply(arrayExpr, scalarExpr);
        }

        abstract ExprEval doApply(ExprEval var1, ExprEval var2);
    }

    public static abstract class StringLongFunction
    extends BivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (x.type() != ExprType.STRING || y.type() != ExprType.LONG) {
                throw new IAE("Function[%s] needs a string as first argument and an integer as second argument", this.name());
            }
            return this.eval(x.asString(), y.asInt());
        }

        protected abstract ExprEval eval(@Nullable String var1, int var2);
    }

    public static abstract class BivariateMathFunction
    extends BivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval x, ExprEval y) {
            if (x.type() == ExprType.STRING || y.type() == ExprType.STRING) {
                return ExprEval.of(null);
            }
            if (x.type() == ExprType.LONG && y.type() == ExprType.LONG) {
                return this.eval(x.asLong(), y.asLong());
            }
            return this.eval(x.asDouble(), y.asDouble());
        }

        protected ExprEval eval(long x, long y) {
            return this.eval((double)x, (double)y);
        }

        protected ExprEval eval(double x, double y) {
            return this.eval((long)x, (long)y);
        }
    }

    public static abstract class UnivariateMathFunction
    extends UnivariateFunction {
        @Override
        protected final ExprEval eval(ExprEval param) {
            if (NullHandling.sqlCompatible() && param.isNumericNull()) {
                return ExprEval.of(null);
            }
            if (param.type() == ExprType.LONG) {
                return this.eval(param.asLong());
            }
            if (param.type() == ExprType.DOUBLE) {
                return this.eval(param.asDouble());
            }
            return ExprEval.of(null);
        }

        protected ExprEval eval(long param) {
            return this.eval((double)param);
        }

        protected ExprEval eval(double param) {
            return this.eval((long)param);
        }
    }

    public static abstract class BivariateFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 2) {
                throw new IAE("Function[%s] needs 2 arguments", this.name());
            }
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Expr expr1 = args.get(0);
            Expr expr2 = args.get(1);
            return this.eval(expr1.eval(bindings), expr2.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1, ExprEval var2);
    }

    public static abstract class UnivariateFunction
    implements Function {
        @Override
        public void validateArguments(List<Expr> args) {
            if (args.size() != 1) {
                throw new IAE("Function[%s] needs 1 argument", this.name());
            }
        }

        @Override
        public ExprEval apply(List<Expr> args, Expr.ObjectBinding bindings) {
            Expr expr = args.get(0);
            return this.eval(expr.eval(bindings));
        }

        protected abstract ExprEval eval(ExprEval var1);
    }
}

