/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.elasticsearch.client;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.InstanceProfileCredentialsProvider;
import com.facebook.airlift.concurrent.Threads;
import com.facebook.airlift.json.JsonCodec;
import com.facebook.airlift.json.JsonObjectMapperProvider;
import com.facebook.airlift.log.Logger;
import com.facebook.airlift.security.pem.PemReader;
import com.facebook.presto.elasticsearch.AwsSecurityConfig;
import com.facebook.presto.elasticsearch.ElasticsearchConfig;
import com.facebook.presto.elasticsearch.ElasticsearchErrorCode;
import com.facebook.presto.elasticsearch.PasswordConfig;
import com.facebook.presto.elasticsearch.client.AwsRequestSigner;
import com.facebook.presto.elasticsearch.client.CountResponse;
import com.facebook.presto.elasticsearch.client.ElasticsearchNode;
import com.facebook.presto.elasticsearch.client.IndexMetadata;
import com.facebook.presto.elasticsearch.client.NodesResponse;
import com.facebook.presto.elasticsearch.client.SearchShardsResponse;
import com.facebook.presto.elasticsearch.client.Shard;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.NullNode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.units.Duration;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollRequest;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.ResponseException;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;

public class ElasticsearchClient {
    private static final Logger LOG = Logger.get(ElasticsearchClient.class);
    private static final JsonCodec<SearchShardsResponse> SEARCH_SHARDS_RESPONSE_CODEC = JsonCodec.jsonCodec(SearchShardsResponse.class);
    private static final JsonCodec<NodesResponse> NODES_RESPONSE_CODEC = JsonCodec.jsonCodec(NodesResponse.class);
    private static final JsonCodec<CountResponse> COUNT_RESPONSE_CODEC = JsonCodec.jsonCodec(CountResponse.class);
    private static final ObjectMapper OBJECT_MAPPER = new JsonObjectMapperProvider().get();
    private static final Pattern ADDRESS_PATTERN = Pattern.compile("((?<cname>[^/]+)/)?(?<ip>.+):(?<port>\\d+)");
    private final RestHighLevelClient client;
    private final int scrollSize;
    private final Duration scrollTimeout;
    private final AtomicReference<Set<ElasticsearchNode>> nodes = new AtomicReference<ImmutableSet>(ImmutableSet.of());
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"NodeRefresher"));
    private final AtomicBoolean started = new AtomicBoolean();
    private final Duration refreshInterval;
    private final boolean tlsEnabled;
    private final boolean ignorePublishAddress;

    @Inject
    public ElasticsearchClient(ElasticsearchConfig config, Optional<AwsSecurityConfig> awsSecurityConfig, Optional<PasswordConfig> passwordConfig) {
        this.client = ElasticsearchClient.createClient(config, awsSecurityConfig, passwordConfig);
        this.ignorePublishAddress = config.isIgnorePublishAddress();
        this.scrollSize = config.getScrollSize();
        this.scrollTimeout = config.getScrollTimeout();
        this.refreshInterval = config.getNodeRefreshInterval();
        this.tlsEnabled = config.isTlsEnabled();
    }

    @PostConstruct
    public void initialize() {
        if (!this.started.getAndSet(true)) {
            this.refreshNodes();
            this.executor.scheduleWithFixedDelay(this::refreshNodes, this.refreshInterval.toMillis(), this.refreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    @PreDestroy
    public void close() throws IOException {
        this.executor.shutdownNow();
        this.client.close();
    }

    private void refreshNodes() {
        try {
            Set<ElasticsearchNode> nodes = this.fetchNodes();
            HttpHost[] hosts = (HttpHost[])nodes.stream().map(ElasticsearchNode::getAddress).filter(Optional::isPresent).map(Optional::get).map(address -> HttpHost.create((String)String.format("%s://%s", this.tlsEnabled ? "https" : "http", address))).toArray(HttpHost[]::new);
            if (hosts.length > 0 && !this.ignorePublishAddress) {
                this.client.getLowLevelClient().setHosts(hosts);
            }
            this.nodes.set(nodes);
        }
        catch (Throwable e) {
            LOG.error(e, "Error refreshing nodes");
        }
    }

    private static RestHighLevelClient createClient(ElasticsearchConfig config, Optional<AwsSecurityConfig> awsSecurityConfig, Optional<PasswordConfig> passwordConfig) {
        RestClientBuilder builder = RestClient.builder((HttpHost[])new HttpHost[]{new HttpHost(config.getHost(), config.getPort(), config.isTlsEnabled() ? "https" : "http")}).setMaxRetryTimeoutMillis((int)config.getMaxRetryTime().toMillis());
        builder.setHttpClientConfigCallback(ignored -> {
            RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(StrictMath.toIntExact(config.getConnectTimeout().toMillis())).setSocketTimeout(StrictMath.toIntExact(config.getRequestTimeout().toMillis())).build();
            IOReactorConfig reactorConfig = IOReactorConfig.custom().setIoThreadCount(config.getHttpThreadCount()).build();
            HttpAsyncClientBuilder clientBuilder = HttpAsyncClientBuilder.create().setDefaultRequestConfig(requestConfig).setDefaultIOReactorConfig(reactorConfig).setMaxConnPerRoute(config.getMaxHttpConnections()).setMaxConnTotal(config.getMaxHttpConnections());
            if (config.isTlsEnabled()) {
                ElasticsearchClient.buildSslContext(config.getKeystorePath(), config.getKeystorePassword(), config.getTrustStorePath(), config.getTruststorePassword()).ifPresent(arg_0 -> ((HttpAsyncClientBuilder)clientBuilder).setSSLContext(arg_0));
                if (config.isVerifyHostnames()) {
                    clientBuilder.setSSLHostnameVerifier((HostnameVerifier)NoopHostnameVerifier.INSTANCE);
                }
            }
            passwordConfig.ifPresent(securityConfig -> {
                BasicCredentialsProvider credentials = new BasicCredentialsProvider();
                credentials.setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(securityConfig.getUser(), securityConfig.getPassword()));
                clientBuilder.setDefaultCredentialsProvider((CredentialsProvider)credentials);
            });
            awsSecurityConfig.ifPresent(securityConfig -> clientBuilder.addInterceptorLast((HttpRequestInterceptor)new AwsRequestSigner(securityConfig.getRegion(), ElasticsearchClient.getAwsCredentialsProvider(securityConfig))));
            return clientBuilder;
        });
        return new RestHighLevelClient(builder);
    }

    private static AWSCredentialsProvider getAwsCredentialsProvider(AwsSecurityConfig config) {
        if (config.getAccessKey().isPresent() && config.getSecretKey().isPresent()) {
            return new AWSStaticCredentialsProvider((AWSCredentials)new BasicAWSCredentials(config.getAccessKey().get(), config.getSecretKey().get()));
        }
        if (config.isUseInstanceCredentials()) {
            return InstanceProfileCredentialsProvider.getInstance();
        }
        return DefaultAWSCredentialsProviderChain.getInstance();
    }

    private static Optional<SSLContext> buildSslContext(Optional<File> keyStorePath, Optional<String> keyStorePassword, Optional<File> trustStorePath, Optional<String> trustStorePassword) {
        if (!keyStorePath.isPresent() && !trustStorePath.isPresent()) {
            return Optional.empty();
        }
        try {
            KeyStore keyStore = null;
            KeyManager[] keyManagers = null;
            if (keyStorePath.isPresent()) {
                char[] keyManagerPassword;
                try {
                    keyStore = PemReader.loadKeyStore((File)keyStorePath.get(), (File)keyStorePath.get(), keyStorePassword);
                    keyManagerPassword = new char[]{};
                }
                catch (IOException | GeneralSecurityException ignored) {
                    keyManagerPassword = keyStorePassword.map(String::toCharArray).orElse(null);
                    keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
                    try (FileInputStream in = new FileInputStream(keyStorePath.get());){
                        keyStore.load(in, keyManagerPassword);
                    }
                }
                ElasticsearchClient.validateCertificates(keyStore);
                KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
                keyManagerFactory.init(keyStore, keyManagerPassword);
                keyManagers = keyManagerFactory.getKeyManagers();
            }
            KeyStore trustStore = keyStore;
            if (trustStorePath.isPresent()) {
                trustStore = ElasticsearchClient.loadTrustStore(trustStorePath.get(), trustStorePassword);
            }
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(trustStore);
            Object[] trustManagers = trustManagerFactory.getTrustManagers();
            if (trustManagers.length != 1 || !(trustManagers[0] instanceof X509TrustManager)) {
                throw new RuntimeException("Unexpected default trust managers:" + Arrays.toString(trustManagers));
            }
            X509TrustManager trustManager = (X509TrustManager)trustManagers[0];
            SSLContext result = SSLContext.getInstance("SSL");
            result.init(keyManagers, new TrustManager[]{trustManager}, null);
            return Optional.of(result);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_SSL_INITIALIZATION_FAILURE, (Throwable)e);
        }
    }

    private static KeyStore loadTrustStore(File trustStorePath, Optional<String> trustStorePassword) throws IOException, GeneralSecurityException {
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        try {
            List certificateChain = PemReader.readCertificateChain((File)trustStorePath);
            if (!certificateChain.isEmpty()) {
                trustStore.load(null, null);
                for (X509Certificate certificate : certificateChain) {
                    X500Principal principal = certificate.getSubjectX500Principal();
                    trustStore.setCertificateEntry(principal.getName(), certificate);
                }
                return trustStore;
            }
        }
        catch (IOException | GeneralSecurityException certificateChain) {
            // empty catch block
        }
        try (FileInputStream in = new FileInputStream(trustStorePath);){
            trustStore.load(in, trustStorePassword.map(String::toCharArray).orElse(null));
        }
        return trustStore;
    }

    private static void validateCertificates(KeyStore keyStore) throws GeneralSecurityException {
        for (String alias : Collections.list(keyStore.aliases())) {
            Certificate certificate;
            if (!keyStore.isKeyEntry(alias) || !((certificate = keyStore.getCertificate(alias)) instanceof X509Certificate)) continue;
            try {
                ((X509Certificate)certificate).checkValidity();
            }
            catch (CertificateExpiredException e) {
                throw new CertificateExpiredException("KeyStore certificate is expired: " + e.getMessage());
            }
            catch (CertificateNotYetValidException e) {
                throw new CertificateNotYetValidException("KeyStore certificate is not yet valid: " + e.getMessage());
            }
        }
    }

    private Set<ElasticsearchNode> fetchNodes() {
        NodesResponse nodesResponse = this.doRequest("/_nodes/http", arg_0 -> NODES_RESPONSE_CODEC.fromJson(arg_0));
        ImmutableSet.Builder result = ImmutableSet.builder();
        for (Map.Entry<String, NodesResponse.Node> entry : nodesResponse.getNodes().entrySet()) {
            String nodeId = entry.getKey();
            NodesResponse.Node node = entry.getValue();
            if (!node.getRoles().contains("data")) continue;
            Optional<String> address = node.getAddress().flatMap(ElasticsearchClient::extractAddress);
            result.add((Object)new ElasticsearchNode(nodeId, address));
        }
        return result.build();
    }

    public Set<ElasticsearchNode> getNodes() {
        return this.nodes.get();
    }

    public List<Shard> getSearchShards(String index) {
        Map nodeById = (Map)this.getNodes().stream().collect(ImmutableMap.toImmutableMap(ElasticsearchNode::getId, Function.identity()));
        SearchShardsResponse shardsResponse = this.doRequest(String.format("/%s/_search_shards", index), arg_0 -> SEARCH_SHARDS_RESPONSE_CODEC.fromJson(arg_0));
        ImmutableList.Builder shards = ImmutableList.builder();
        ImmutableList nodes = ImmutableList.copyOf(nodeById.values());
        for (List<SearchShardsResponse.Shard> shardGroup : shardsResponse.getShardGroups()) {
            ElasticsearchNode node;
            SearchShardsResponse.Shard chosen;
            Stream preferred = shardGroup.stream().sorted(this::shardPreference);
            Optional<SearchShardsResponse.Shard> candidate = preferred.filter(shard -> shard.getNode() != null && nodeById.containsKey(shard.getNode())).findFirst();
            if (candidate.isPresent()) {
                chosen = candidate.get();
                node = (ElasticsearchNode)nodeById.get(chosen.getNode());
            } else {
                chosen = (SearchShardsResponse.Shard)preferred.findFirst().get();
                node = (ElasticsearchNode)nodes.get(chosen.getShard() % nodes.size());
            }
            shards.add((Object)new Shard(chosen.getIndex(), chosen.getShard(), node.getAddress()));
        }
        return shards.build();
    }

    private int shardPreference(SearchShardsResponse.Shard left, SearchShardsResponse.Shard right) {
        if (left.isPrimary() == right.isPrimary()) {
            return 0;
        }
        return left.isPrimary() ? 1 : -1;
    }

    public List<String> getIndexes() {
        return (List)this.doRequest("/_cat/indices?h=index&format=json&s=index:asc", body -> {
            try {
                ImmutableList.Builder result = ImmutableList.builder();
                JsonNode root = OBJECT_MAPPER.readTree(body);
                for (int i = 0; i < root.size(); ++i) {
                    result.add((Object)root.get(i).get("index").asText());
                }
                return result.build();
            }
            catch (IOException e) {
                throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    public Map<String, List<String>> getAliases() {
        return (Map)this.doRequest("/_aliases", body -> {
            try {
                ImmutableMap.Builder result = ImmutableMap.builder();
                JsonNode root = OBJECT_MAPPER.readTree(body);
                Iterator elements = root.fields();
                while (elements.hasNext()) {
                    Map.Entry element = (Map.Entry)elements.next();
                    JsonNode aliases = ((JsonNode)element.getValue()).get("aliases");
                    Iterator aliasNames = aliases.fieldNames();
                    if (!aliasNames.hasNext()) continue;
                    result.put(element.getKey(), (Object)ImmutableList.copyOf((Iterator)aliasNames));
                }
                return result.build();
            }
            catch (IOException e) {
                throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    public IndexMetadata getIndexMetadata(String index) {
        String path = String.format("/%s/_mappings", index);
        return this.doRequest(path, body -> {
            try {
                JsonNode mappings = ((JsonNode)OBJECT_MAPPER.readTree(body).elements().next()).get("mappings");
                if (!mappings.has("properties")) {
                    mappings = (JsonNode)mappings.elements().next();
                }
                JsonNode metaNode = this.nullSafeNode(mappings, "_meta");
                return new IndexMetadata(this.parseType(mappings.get("properties"), this.nullSafeNode(metaNode, "presto")));
            }
            catch (IOException e) {
                throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    private IndexMetadata.ObjectType parseType(JsonNode properties, JsonNode metaProperties) {
        Iterator entries = properties.fields();
        ImmutableList.Builder result = ImmutableList.builder();
        block9: while (entries.hasNext()) {
            JsonNode metaNode;
            Map.Entry field = (Map.Entry)entries.next();
            String name = (String)field.getKey();
            JsonNode value = (JsonNode)field.getValue();
            String type = "object";
            if (value.has("type")) {
                type = value.get("type").asText();
            }
            boolean isArray = !(metaNode = this.nullSafeNode(metaProperties, name)).isNull() && metaNode.has("isArray") && metaNode.get("isArray").asBoolean();
            switch (type) {
                case "date": {
                    Object formats = ImmutableList.of();
                    if (value.has("format")) {
                        formats = Arrays.asList(value.get("format").asText().split("\\|\\|"));
                    }
                    result.add((Object)new IndexMetadata.Field(isArray, name, new IndexMetadata.DateTimeType((List<String>)formats)));
                    continue block9;
                }
                case "nested": 
                case "object": {
                    if (value.has("properties")) {
                        result.add((Object)new IndexMetadata.Field(isArray, name, this.parseType(value.get("properties"), metaNode)));
                        continue block9;
                    }
                    LOG.debug("Ignoring empty object field: %s", new Object[]{name});
                    continue block9;
                }
            }
            result.add((Object)new IndexMetadata.Field(isArray, name, new IndexMetadata.PrimitiveType(type)));
        }
        return new IndexMetadata.ObjectType((List<IndexMetadata.Field>)result.build());
    }

    private JsonNode nullSafeNode(JsonNode jsonNode, String name) {
        if (jsonNode == null || jsonNode.isNull() || jsonNode.get(name) == null) {
            return NullNode.getInstance();
        }
        return jsonNode.get(name);
    }

    public String executeQuery(String index, String query) {
        String body;
        Response response;
        String path = String.format("/%s/_search", index);
        try {
            response = this.client.getLowLevelClient().performRequest("GET", path, (Map)ImmutableMap.of(), (HttpEntity)new ByteArrayEntity(query.getBytes(StandardCharsets.UTF_8)), new Header[]{new BasicHeader("Content-Type", "application/json"), new BasicHeader("Accept-Encoding", "application/json")});
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        try {
            body = EntityUtils.toString((HttpEntity)response.getEntity());
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
        }
        return body;
    }

    public SearchResponse beginSearch(String index, int shard, QueryBuilder query, Optional<List<String>> fields, List<String> documentFields, Optional<String> sort) {
        SearchSourceBuilder sourceBuilder = SearchSourceBuilder.searchSource().query(query).size(this.scrollSize);
        sort.ifPresent(arg_0 -> ((SearchSourceBuilder)sourceBuilder).sort(arg_0));
        fields.ifPresent(values -> {
            if (values.isEmpty()) {
                sourceBuilder.fetchSource(false);
            } else {
                sourceBuilder.fetchSource(values.toArray(new String[0]), null);
            }
        });
        documentFields.forEach(arg_0 -> ((SearchSourceBuilder)sourceBuilder).docValueField(arg_0));
        SearchRequest request = new SearchRequest(new String[]{index}).searchType(SearchType.QUERY_THEN_FETCH).preference("_shards:" + shard).scroll(new TimeValue(this.scrollTimeout.toMillis())).source(sourceBuilder);
        try {
            return this.client.search(request, new Header[0]);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        catch (ElasticsearchStatusException e) {
            Throwable cause;
            Throwable[] suppressed = e.getSuppressed();
            if (suppressed.length > 0 && (cause = suppressed[0]) instanceof ResponseException) {
                HttpEntity entity = ((ResponseException)cause).getResponse().getEntity();
                try {
                    JsonNode reason = OBJECT_MAPPER.readTree(entity.getContent()).path("error").path("root_cause").path(0).path("reason");
                    if (!reason.isMissingNode()) {
                        throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, reason.asText(), (Throwable)e);
                    }
                }
                catch (IOException ex) {
                    e.addSuppressed((Throwable)ex);
                }
            }
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
    }

    public SearchResponse nextPage(String scrollId) {
        SearchScrollRequest request = new SearchScrollRequest(scrollId).scroll(new TimeValue(this.scrollTimeout.toMillis()));
        try {
            return this.client.searchScroll(request, new Header[0]);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
    }

    public long count(String index, int shard, QueryBuilder query) {
        Response response;
        SearchSourceBuilder sourceBuilder = SearchSourceBuilder.searchSource().query(query);
        LOG.debug("Count: %s:%s, query: %s", new Object[]{index, shard, sourceBuilder});
        try {
            response = this.client.getLowLevelClient().performRequest("GET", String.format("/%s/_count?preference=_shards:%s", index, shard), (Map)ImmutableMap.of(), (HttpEntity)new StringEntity(sourceBuilder.toString()), new Header[]{new BasicHeader("Content-Type", "application/json")});
        }
        catch (ResponseException e) {
            throw ElasticsearchClient.propagate(e);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        try {
            return ((CountResponse)COUNT_RESPONSE_CODEC.fromJson(EntityUtils.toByteArray((HttpEntity)response.getEntity()))).getCount();
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
        }
    }

    public void clearScroll(String scrollId) {
        ClearScrollRequest request = new ClearScrollRequest();
        request.addScrollId(scrollId);
        try {
            this.client.clearScroll(request, new Header[0]);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
    }

    private <T> T doRequest(String path, ResponseHandler<T> handler) {
        String body;
        Response response;
        Preconditions.checkArgument((boolean)path.startsWith("/"), (Object)"path must be an absolute path");
        try {
            response = this.client.getLowLevelClient().performRequest("GET", path, new Header[0]);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        try {
            body = EntityUtils.toString((HttpEntity)response.getEntity());
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
        }
        return handler.process(body);
    }

    private static PrestoException propagate(ResponseException exception) {
        HttpEntity entity = exception.getResponse().getEntity();
        if (entity != null && entity.getContentType() != null) {
            try {
                JsonNode reason = OBJECT_MAPPER.readTree(entity.getContent()).path("error").path("root_cause").path(0).path("reason");
                if (!reason.isMissingNode()) {
                    throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, reason.asText(), (Throwable)exception);
                }
            }
            catch (IOException e) {
                PrestoException result = new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, (Throwable)exception);
                result.addSuppressed((Throwable)e);
                throw result;
            }
        }
        throw new PrestoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, (Throwable)exception);
    }

    @VisibleForTesting
    static Optional<String> extractAddress(String address) {
        Matcher matcher = ADDRESS_PATTERN.matcher(address);
        if (!matcher.matches()) {
            return Optional.empty();
        }
        String cname = matcher.group("cname");
        String ip = matcher.group("ip");
        String port = matcher.group("port");
        if (cname != null) {
            return Optional.of(cname + ":" + port);
        }
        return Optional.of(ip + ":" + port);
    }

    private static interface ResponseHandler<T> {
        public T process(String var1);
    }
}

