/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.plugin.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.GeometryCursor;
import com.esri.core.geometry.MultiPath;
import com.esri.core.geometry.MultiPoint;
import com.esri.core.geometry.MultiVertexGeometry;
import com.esri.core.geometry.NonSimpleResult;
import com.esri.core.geometry.OperatorSimplifyOGC;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.Polygon;
import com.esri.core.geometry.Polyline;
import com.esri.core.geometry.SpatialReference;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.esri.core.geometry.ogc.OGCGeometryCollection;
import com.esri.core.geometry.ogc.OGCLineString;
import com.esri.core.geometry.ogc.OGCMultiPolygon;
import com.esri.core.geometry.ogc.OGCPoint;
import com.esri.core.geometry.ogc.OGCPolygon;
import com.facebook.presto.geospatial.GeometryType;
import com.facebook.presto.geospatial.GeometryUtils;
import com.facebook.presto.geospatial.serde.GeometrySerde;
import com.facebook.presto.geospatial.serde.GeometrySerializationType;
import com.facebook.presto.geospatial.serde.JtsGeometrySerde;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.SqlNullable;
import com.facebook.presto.spi.function.SqlType;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.locationtech.jts.linearref.LengthIndexedLine;
import org.locationtech.jts.simplify.TopologyPreservingSimplifier;

public final class GeoFunctions {
    private static final Joiner OR_JOINER = Joiner.on((String)" or ");
    private static final Slice EMPTY_POLYGON = GeometrySerde.serialize((OGCGeometry)new OGCPolygon(new Polygon(), null));
    private static final Slice EMPTY_MULTIPOINT = GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new MultiPoint(), null, (boolean)true));
    private static final double EARTH_RADIUS_KM = 6371.01;
    private static final Map<NonSimpleResult.Reason, String> NON_SIMPLE_REASONS = ImmutableMap.builder().put((Object)NonSimpleResult.Reason.DegenerateSegments, (Object)"Degenerate segments").put((Object)NonSimpleResult.Reason.Clustering, (Object)"Repeated points").put((Object)NonSimpleResult.Reason.Cracking, (Object)"Intersecting or overlapping segments").put((Object)NonSimpleResult.Reason.CrossOver, (Object)"Self-intersection").put((Object)NonSimpleResult.Reason.OGCPolylineSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCPolygonSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCDisconnectedInterior, (Object)"Disconnected interior").build();

    private GeoFunctions() {
    }

    @Description(value="Returns a Geometry type LineString object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_LineFromText")
    @SqlType(value="Geometry")
    public static Slice parseLine(@SqlType(value="varchar") Slice input) {
        OGCGeometry geometry = GeoFunctions.geometryFromText(input);
        GeoFunctions.validateType("ST_LineFromText", geometry, EnumSet.of(GeometryType.LINE_STRING));
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @Description(value="Returns a Geometry type Point object with the given coordinate values")
    @ScalarFunction(value="ST_Point")
    @SqlType(value="Geometry")
    public static Slice stPoint(@SqlType(value="double") double x, @SqlType(value="double") double y) {
        OGCGeometry geometry = OGCGeometry.createFromEsriGeometry((Geometry)new Point(x, y), null);
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @Description(value="Returns a Geometry type Polygon object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_Polygon")
    @SqlType(value="Geometry")
    public static Slice stPolygon(@SqlType(value="varchar") Slice input) {
        OGCGeometry geometry = GeoFunctions.geometryFromText(input);
        GeoFunctions.validateType("ST_Polygon", geometry, EnumSet.of(GeometryType.POLYGON));
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @Description(value="Returns the area of a polygon using Euclidean measurement on a 2D plane (based on spatial ref) in projected units")
    @ScalarFunction(value="ST_Area")
    @SqlType(value="double")
    public static double stArea(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Area", geometry, EnumSet.of(GeometryType.POLYGON, GeometryType.MULTI_POLYGON));
        return geometry.getEsriGeometry().calculateArea2D();
    }

    @Description(value="Returns a Geometry type object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_GeometryFromText")
    @SqlType(value="Geometry")
    public static Slice stGeometryFromText(@SqlType(value="varchar") Slice input) {
        return GeometrySerde.serialize((OGCGeometry)GeoFunctions.geometryFromText(input));
    }

    @Description(value="Returns the Well-Known Text (WKT) representation of the geometry")
    @ScalarFunction(value="ST_AsText")
    @SqlType(value="varchar")
    public static Slice stAsText(@SqlType(value="Geometry") Slice input) {
        return Slices.utf8Slice((String)GeometrySerde.deserialize((Slice)input).asText());
    }

    @SqlNullable
    @Description(value="Returns the geometry that represents all points whose distance from the specified geometry is less than or equal to the specified distance")
    @ScalarFunction(value="ST_Buffer")
    @SqlType(value="Geometry")
    public static Slice stBuffer(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (distance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        if (distance == 0.0) {
            return input;
        }
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.buffer(distance));
    }

    @Description(value="Returns the Point value that is the mathematical centroid of a Geometry")
    @ScalarFunction(value="ST_Centroid")
    @SqlType(value="Geometry")
    public static Slice stCentroid(@SqlType(value="Geometry") Slice input) {
        Point centroid;
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Centroid", geometry, EnumSet.of(GeometryType.POINT, new GeometryType[]{GeometryType.MULTI_POINT, GeometryType.LINE_STRING, GeometryType.MULTI_LINE_STRING, GeometryType.POLYGON, GeometryType.MULTI_POLYGON}));
        GeometryType geometryType = GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (geometryType == GeometryType.POINT) {
            return input;
        }
        int pointCount = ((MultiVertexGeometry)geometry.getEsriGeometry()).getPointCount();
        if (pointCount == 0) {
            return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Point(), (SpatialReference)geometry.getEsriSpatialReference()));
        }
        switch (geometryType) {
            case MULTI_POINT: {
                centroid = GeoFunctions.computePointsCentroid((MultiVertexGeometry)geometry.getEsriGeometry());
                break;
            }
            case LINE_STRING: 
            case MULTI_LINE_STRING: {
                centroid = GeoFunctions.computeLineCentroid((Polyline)geometry.getEsriGeometry());
                break;
            }
            case POLYGON: {
                centroid = GeoFunctions.computePolygonCentroid((Polygon)geometry.getEsriGeometry());
                break;
            }
            case MULTI_POLYGON: {
                centroid = GeoFunctions.computeMultiPolygonCentroid((OGCMultiPolygon)geometry);
                break;
            }
            default: {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Unexpected geometry type: " + geometryType);
            }
        }
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)centroid, (SpatialReference)geometry.getEsriSpatialReference()));
    }

    @Description(value="Returns the minimum convex geometry that encloses all input geometries")
    @ScalarFunction(value="ST_ConvexHull")
    @SqlType(value="Geometry")
    public static Slice stConvexHull(@SqlType(value="Geometry") Slice input) {
        MultiVertexGeometry multiVertex;
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_ConvexHull", geometry, EnumSet.of(GeometryType.POINT, new GeometryType[]{GeometryType.MULTI_POINT, GeometryType.LINE_STRING, GeometryType.MULTI_LINE_STRING, GeometryType.POLYGON, GeometryType.MULTI_POLYGON}));
        if (geometry.isEmpty()) {
            return input;
        }
        if (GeometryType.getForEsriGeometryType((String)geometry.geometryType()) == GeometryType.POINT) {
            return input;
        }
        OGCGeometry convexHull = geometry.convexHull();
        if (convexHull.isEmpty()) {
            return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)((MultiVertexGeometry)geometry.getEsriGeometry()).getPoint(0), null));
        }
        if (GeometryType.getForEsriGeometryType((String)convexHull.geometryType()) == GeometryType.MULTI_POLYGON && (multiVertex = (MultiVertexGeometry)convexHull.getEsriGeometry()).getPointCount() == 2) {
            OGCGeometry linestring = OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(multiVertex.getPoint(0), multiVertex.getPoint(1)), null);
            return GeometrySerde.serialize((OGCGeometry)linestring);
        }
        return GeometrySerde.serialize((OGCGeometry)convexHull);
    }

    @Description(value="Return the coordinate dimension of the Geometry")
    @ScalarFunction(value="ST_CoordDim")
    @SqlType(value="tinyint")
    public static long stCoordinateDimension(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserialize((Slice)input).coordinateDimension();
    }

    @Description(value="Returns the inherent dimension of this Geometry object, which must be less than or equal to the coordinate dimension")
    @ScalarFunction(value="ST_Dimension")
    @SqlType(value="tinyint")
    public static long stDimension(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserialize((Slice)input).dimension();
    }

    @SqlNullable
    @Description(value="Returns TRUE if the LineString or Multi-LineString's start and end points are coincident")
    @ScalarFunction(value="ST_IsClosed")
    @SqlType(value="boolean")
    public static Boolean stIsClosed(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsClosed", geometry, EnumSet.of(GeometryType.LINE_STRING, GeometryType.MULTI_LINE_STRING));
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        int pathCount = lines.getPathCount();
        for (int i = 0; i < pathCount; ++i) {
            Point start = lines.getPoint(lines.getPathStart(i));
            Point end = lines.getPoint(lines.getPathEnd(i) - 1);
            if (end.equals((Object)start)) continue;
            return false;
        }
        return true;
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is an empty geometrycollection, polygon, point etc")
    @ScalarFunction(value="ST_IsEmpty")
    @SqlType(value="boolean")
    public static Boolean stIsEmpty(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserialize((Slice)input).isEmpty();
    }

    @Description(value="Returns TRUE if this Geometry has no anomalous geometric points, such as self intersection or self tangency")
    @ScalarFunction(value="ST_IsSimple")
    @SqlType(value="boolean")
    public static boolean stIsSimple(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        return geometry.isEmpty() || geometry.isSimple();
    }

    @Description(value="Returns true if the input geometry is well formed")
    @ScalarFunction(value="ST_IsValid")
    @SqlType(value="boolean")
    public static boolean stIsValid(@SqlType(value="Geometry") Slice input) {
        Geometry geometry;
        GeometryCursor cursor = GeometrySerde.deserialize((Slice)input).getEsriGeometryCursor();
        do {
            if ((geometry = cursor.next()) != null) continue;
            return true;
        } while (OperatorSimplifyOGC.local().isSimpleOGC(geometry, null, true, null, null));
        return false;
    }

    @Description(value="Returns the reason for why the input geometry is not valid. Returns null if the input is valid.")
    @ScalarFunction(value="geometry_invalid_reason")
    @SqlType(value="varchar")
    @SqlNullable
    public static Slice invalidReason(@SqlType(value="Geometry") Slice input) {
        Geometry geometry;
        GeometryCursor cursor = GeometrySerde.deserialize((Slice)input).getEsriGeometryCursor();
        NonSimpleResult result = new NonSimpleResult();
        do {
            if ((geometry = cursor.next()) != null) continue;
            return null;
        } while (OperatorSimplifyOGC.local().isSimpleOGC(geometry, null, true, result, null));
        String reasonText = NON_SIMPLE_REASONS.getOrDefault(result.m_reason, result.m_reason.name());
        if (!(geometry instanceof MultiVertexGeometry)) {
            return Slices.utf8Slice((String)reasonText);
        }
        MultiVertexGeometry multiVertexGeometry = (MultiVertexGeometry)geometry;
        if (result.m_vertexIndex1 >= 0 && result.m_vertexIndex2 >= 0) {
            Point point1 = multiVertexGeometry.getPoint(result.m_vertexIndex1);
            Point point2 = multiVertexGeometry.getPoint(result.m_vertexIndex2);
            return Slices.utf8Slice((String)String.format("%s at or near (%s %s) and (%s %s)", reasonText, point1.getX(), point1.getY(), point2.getX(), point2.getY()));
        }
        if (result.m_vertexIndex1 >= 0) {
            Point point = multiVertexGeometry.getPoint(result.m_vertexIndex1);
            return Slices.utf8Slice((String)String.format("%s at or near (%s %s)", reasonText, point.getX(), point.getY()));
        }
        return Slices.utf8Slice((String)reasonText);
    }

    @Description(value="Returns the length of a LineString or Multi-LineString using Euclidean measurement on a 2D plane (based on spatial ref) in projected units")
    @ScalarFunction(value="ST_Length")
    @SqlType(value="double")
    public static double stLength(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Length", geometry, EnumSet.of(GeometryType.LINE_STRING, GeometryType.MULTI_LINE_STRING));
        return geometry.getEsriGeometry().calculateLength2D();
    }

    @SqlNullable
    @Description(value="Returns a float between 0 and 1 representing the location of the closest point on the LineString to the given Point, as a fraction of total 2d line length.")
    @ScalarFunction(value="line_locate_point")
    @SqlType(value="double")
    public static Double lineLocatePoint(@SqlType(value="Geometry") Slice lineSlice, @SqlType(value="Geometry") Slice pointSlice) {
        org.locationtech.jts.geom.Geometry line = JtsGeometrySerde.deserialize((Slice)lineSlice);
        org.locationtech.jts.geom.Geometry point = JtsGeometrySerde.deserialize((Slice)pointSlice);
        if (line.isEmpty() || point.isEmpty()) {
            return null;
        }
        GeometryType lineType = GeometryType.getForJtsGeometryType((String)line.getGeometryType());
        if (lineType != GeometryType.LINE_STRING && lineType != GeometryType.MULTI_LINE_STRING) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("First argument to line_locate_point must be a LineString or a MultiLineString. Got: %s", line.getGeometryType()));
        }
        GeometryType pointType = GeometryType.getForJtsGeometryType((String)point.getGeometryType());
        if (pointType != GeometryType.POINT) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Second argument to line_locate_point must be a Point. Got: %s", point.getGeometryType()));
        }
        return new LengthIndexedLine(line).indexOf(point.getCoordinate()) / line.getLength();
    }

    @SqlNullable
    @Description(value="Returns X maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMax")
    @SqlType(value="double")
    public static Double stXMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope == null) {
            return null;
        }
        return envelope.getXMax();
    }

    @SqlNullable
    @Description(value="Returns Y maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMax")
    @SqlType(value="double")
    public static Double stYMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope == null) {
            return null;
        }
        return envelope.getYMax();
    }

    @SqlNullable
    @Description(value="Returns X minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMin")
    @SqlType(value="double")
    public static Double stXMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope == null) {
            return null;
        }
        return envelope.getXMin();
    }

    @SqlNullable
    @Description(value="Returns Y minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMin")
    @SqlType(value="double")
    public static Double stYMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope == null) {
            return null;
        }
        return envelope.getYMin();
    }

    @SqlNullable
    @Description(value="Returns the cardinality of the collection of interior rings of a polygon")
    @ScalarFunction(value="ST_NumInteriorRing")
    @SqlType(value="bigint")
    public static Long stNumInteriorRings(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_NumInteriorRing", geometry, EnumSet.of(GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPolygon)geometry).numInteriorRing();
    }

    @Description(value="Returns the cardinality of the geometry collection")
    @ScalarFunction(value="ST_NumGeometries")
    @SqlType(value="integer")
    public static long stNumGeometries(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return 0L;
        }
        GeometryType type = GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!type.isMultitype()) {
            return 1L;
        }
        return ((OGCGeometryCollection)geometry).numGeometries();
    }

    @SqlNullable
    @Description(value="Returns the geometry element at the specified index (indices started with 1)")
    @ScalarFunction(value="ST_GeometryN")
    @SqlType(value="Geometry")
    public static Slice stGeometryN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        GeometryType type = GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!type.isMultitype()) {
            if (index == 1L) {
                return input;
            }
            return null;
        }
        OGCGeometryCollection geometryCollection = (OGCGeometryCollection)geometry;
        if (index < 1L || index > (long)geometryCollection.numGeometries()) {
            return null;
        }
        OGCGeometry ogcGeometry = geometryCollection.geometryN((int)index - 1);
        return GeometrySerde.serialize((OGCGeometry)ogcGeometry);
    }

    @Description(value="Returns the number of points in a Geometry")
    @ScalarFunction(value="ST_NumPoints")
    @SqlType(value="bigint")
    public static long stNumPoints(@SqlType(value="Geometry") Slice input) {
        return GeometryUtils.getPointCount((OGCGeometry)GeometrySerde.deserialize((Slice)input));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if the line is closed and simple")
    @ScalarFunction(value="ST_IsRing")
    @SqlType(value="boolean")
    public static Boolean stIsRing(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsRing", geometry, EnumSet.of(GeometryType.LINE_STRING));
        OGCLineString line = (OGCLineString)geometry;
        return line.isClosed() && line.isSimple();
    }

    @SqlNullable
    @Description(value="Returns the first point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_StartPoint")
    @SqlType(value="Geometry")
    public static Slice stStartPoint(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_StartPoint", geometry, EnumSet.of(GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        SpatialReference reference = geometry.getEsriSpatialReference();
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)lines.getPoint(0), (SpatialReference)reference));
    }

    @Description(value="Returns a \"simplified\" version of the given geometry")
    @ScalarFunction(value="simplify_geometry")
    @SqlType(value="Geometry")
    public static Slice simplifyGeometry(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distanceTolerance) {
        if (Double.isNaN(distanceTolerance)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is NaN");
        }
        if (distanceTolerance < 0.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is negative");
        }
        if (distanceTolerance == 0.0) {
            return input;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)TopologyPreservingSimplifier.simplify((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input), (double)distanceTolerance));
    }

    @SqlNullable
    @Description(value="Returns the last point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_EndPoint")
    @SqlType(value="Geometry")
    public static Slice stEndPoint(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_EndPoint", geometry, EnumSet.of(GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        SpatialReference reference = geometry.getEsriSpatialReference();
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)lines.getPoint(lines.getPointCount() - 1), (SpatialReference)reference));
    }

    @SqlNullable
    @Description(value="Return the X coordinate of the point")
    @ScalarFunction(value="ST_X")
    @SqlType(value="double")
    public static Double stX(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_X", geometry, EnumSet.of(GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPoint)geometry).X();
    }

    @SqlNullable
    @Description(value="Return the Y coordinate of the point")
    @ScalarFunction(value="ST_Y")
    @SqlType(value="double")
    public static Double stY(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Y", geometry, EnumSet.of(GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPoint)geometry).Y();
    }

    @Description(value="Returns the closure of the combinatorial boundary of this Geometry")
    @ScalarFunction(value="ST_Boundary")
    @SqlType(value="Geometry")
    public static Slice stBoundary(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty() && GeometryType.getForEsriGeometryType((String)geometry.geometryType()) == GeometryType.LINE_STRING) {
            return EMPTY_MULTIPOINT;
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.boundary());
    }

    @Description(value="Returns the bounding rectangular polygon of a Geometry")
    @ScalarFunction(value="ST_Envelope")
    @SqlType(value="Geometry")
    public static Slice stEnvelope(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope == null) {
            return EMPTY_POLYGON;
        }
        return GeometrySerde.serialize((Envelope)envelope);
    }

    @Description(value="Returns the Geometry value that represents the point set difference of two geometries")
    @ScalarFunction(value="ST_Difference")
    @SqlType(value="Geometry")
    public static Slice stDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.difference(rightGeometry));
    }

    @Description(value="Returns the 2-dimensional cartesian minimum distance (based on spatial ref) between two geometries in projected units")
    @ScalarFunction(value="ST_Distance")
    @SqlType(value="double")
    public static double stDistance(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.distance(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns a line string representing the exterior ring of the POLYGON")
    @ScalarFunction(value="ST_ExteriorRing")
    @SqlType(value="Geometry")
    public static Slice stExteriorRing(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_ExteriorRing", geometry, EnumSet.of(GeometryType.POLYGON, GeometryType.MULTI_POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)((OGCPolygon)geometry).exteriorRing());
    }

    @Description(value="Returns the Geometry value that represents the point set intersection of two Geometries")
    @ScalarFunction(value="ST_Intersection")
    @SqlType(value="Geometry")
    public static Slice stIntersection(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (GeometrySerde.deserializeType((Slice)left) == GeometrySerializationType.ENVELOPE && GeometrySerde.deserializeType((Slice)right) == GeometrySerializationType.ENVELOPE) {
            Envelope rightEnvelope;
            Envelope leftEnvelope = GeometrySerde.deserializeEnvelope((Slice)left);
            if (!leftEnvelope.intersect(rightEnvelope = GeometrySerde.deserializeEnvelope((Slice)right))) {
                return EMPTY_POLYGON;
            }
            Envelope intersection = leftEnvelope;
            if (intersection.getXMin() == intersection.getXMax()) {
                if (intersection.getYMin() == intersection.getYMax()) {
                    return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Point(intersection.getXMin(), intersection.getXMax()), null));
                }
                return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new Point(intersection.getXMin(), intersection.getYMin()), new Point(intersection.getXMin(), intersection.getYMax())), null));
            }
            if (intersection.getYMin() == intersection.getYMax()) {
                return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new Point(intersection.getXMin(), intersection.getYMin()), new Point(intersection.getXMax(), intersection.getYMin())), null));
            }
            return GeometrySerde.serialize((Envelope)intersection);
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.intersection(rightGeometry));
    }

    @Description(value="Returns the Geometry value that represents the point set symmetric difference of two Geometries")
    @ScalarFunction(value="ST_SymDifference")
    @SqlType(value="Geometry")
    public static Slice stSymmetricDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.symDifference(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if no points of right lie in the exterior of left, and at least one point of the interior of left lies in the interior of right")
    @ScalarFunction(value="ST_Contains")
    @SqlType(value="boolean")
    public static Boolean stContains(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.contains(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the supplied geometries have some, but not all, interior points in common")
    @ScalarFunction(value="ST_Crosses")
    @SqlType(value="boolean")
    public static Boolean stCrosses(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.crosses(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries do not spatially intersect - if they do not share any space together")
    @ScalarFunction(value="ST_Disjoint")
    @SqlType(value="boolean")
    public static Boolean stDisjoint(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return true;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.disjoint(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the given geometries represent the same geometry")
    @ScalarFunction(value="ST_Equals")
    @SqlType(value="boolean")
    public static Boolean stEquals(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.equals(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries spatially intersect in 2D - (share any portion of space) and FALSE if they don't (they are Disjoint)")
    @ScalarFunction(value="ST_Intersects")
    @SqlType(value="boolean")
    public static Boolean stIntersects(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.intersects(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries share space, are of the same dimension, but are not completely contained by each other")
    @ScalarFunction(value="ST_Overlaps")
    @SqlType(value="boolean")
    public static Boolean stOverlaps(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.overlaps(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is spatially related to another Geometry")
    @ScalarFunction(value="ST_Relate")
    @SqlType(value="boolean")
    public static Boolean stRelate(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right, @SqlType(value="varchar") Slice relation) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.relate(rightGeometry, relation.toStringUtf8());
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometries have at least one point in common, but their interiors do not intersect")
    @ScalarFunction(value="ST_Touches")
    @SqlType(value="boolean")
    public static Boolean stTouches(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.touches(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometry A is completely inside geometry B")
    @ScalarFunction(value="ST_Within")
    @SqlType(value="boolean")
    public static Boolean stWithin(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(right, left, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.within(rightGeometry);
    }

    @Description(value="Returns the type of the geometry")
    @ScalarFunction(value="ST_GeometryType")
    @SqlType(value="varchar")
    public static Slice stGeometryType(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.getGeometryType((Slice)input).standardName();
    }

    @ScalarFunction
    @Description(value="Calculates the great-circle distance between two points on the Earth's surface in kilometers")
    @SqlType(value="double")
    public static double greatCircleDistance(@SqlType(value="double") double latitude1, @SqlType(value="double") double longitude1, @SqlType(value="double") double latitude2, @SqlType(value="double") double longitude2) {
        GeoFunctions.checkLatitude(latitude1);
        GeoFunctions.checkLongitude(longitude1);
        GeoFunctions.checkLatitude(latitude2);
        GeoFunctions.checkLongitude(longitude2);
        double radianLatitude1 = Math.toRadians(latitude1);
        double radianLatitude2 = Math.toRadians(latitude2);
        double sin1 = Math.sin(radianLatitude1);
        double cos1 = Math.cos(radianLatitude1);
        double sin2 = Math.sin(radianLatitude2);
        double cos2 = Math.cos(radianLatitude2);
        double deltaLongitude = Math.toRadians(longitude1) - Math.toRadians(longitude2);
        double cosDeltaLongitude = Math.cos(deltaLongitude);
        double t1 = cos2 * Math.sin(deltaLongitude);
        double t2 = cos1 * sin2 - sin1 * cos2 * cosDeltaLongitude;
        double t3 = sin1 * sin2 + cos1 * cos2 * cosDeltaLongitude;
        return Math.atan2(Math.sqrt(t1 * t1 + t2 * t2), t3) * 6371.01;
    }

    private static void checkLatitude(double latitude) {
        if (Double.isNaN(latitude) || Double.isInfinite(latitude) || latitude < -90.0 || latitude > 90.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Latitude must be between -90 and 90");
        }
    }

    private static void checkLongitude(double longitude) {
        if (Double.isNaN(longitude) || Double.isInfinite(longitude) || longitude < -180.0 || longitude > 180.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Longitude must be between -180 and 180");
        }
    }

    private static OGCGeometry geometryFromText(Slice input) {
        OGCGeometry geometry;
        try {
            geometry = OGCGeometry.fromText((String)input.toStringUtf8());
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid WKT: " + input.toStringUtf8(), (Throwable)e);
        }
        geometry.setSpatialReference(null);
        return geometry;
    }

    private static void validateType(String function, OGCGeometry geometry, Set<GeometryType> validTypes) {
        GeometryType type = GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!validTypes.contains(type)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%s only applies to %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    private static void verifySameSpatialReference(OGCGeometry leftGeometry, OGCGeometry rightGeometry) {
        Preconditions.checkArgument((boolean)Objects.equals(leftGeometry.getEsriSpatialReference(), rightGeometry.getEsriSpatialReference()), (Object)"Input geometries must have the same spatial reference");
    }

    private static Point computePointsCentroid(MultiVertexGeometry multiVertex) {
        double xSum = 0.0;
        double ySum = 0.0;
        for (int i = 0; i < multiVertex.getPointCount(); ++i) {
            Point point = multiVertex.getPoint(i);
            xSum += point.getX();
            ySum += point.getY();
        }
        return new Point(xSum / (double)multiVertex.getPointCount(), ySum / (double)multiVertex.getPointCount());
    }

    private static Point computeLineCentroid(Polyline polyline) {
        double xSum = 0.0;
        double ySum = 0.0;
        double weightSum = 0.0;
        for (int i = 0; i < polyline.getPathCount(); ++i) {
            Point startPoint = polyline.getPoint(polyline.getPathStart(i));
            Point endPoint = polyline.getPoint(polyline.getPathEnd(i) - 1);
            double dx = endPoint.getX() - startPoint.getX();
            double dy = endPoint.getY() - startPoint.getY();
            double length = Math.sqrt(dx * dx + dy * dy);
            weightSum += length;
            xSum += (startPoint.getX() + endPoint.getX()) * length / 2.0;
            ySum += (startPoint.getY() + endPoint.getY()) * length / 2.0;
        }
        return new Point(xSum / weightSum, ySum / weightSum);
    }

    private static Point computePolygonCentroid(Polygon polygon) {
        int pathCount = polygon.getPathCount();
        if (pathCount == 1) {
            return GeoFunctions.getPolygonSansHolesCentroid(polygon);
        }
        double xSum = 0.0;
        double ySum = 0.0;
        double areaSum = 0.0;
        for (int i = 0; i < pathCount; ++i) {
            int startIndex = polygon.getPathStart(i);
            int endIndex = polygon.getPathEnd(i);
            Polygon sansHoles = GeoFunctions.getSubPolygon(polygon, startIndex, endIndex);
            Point centroid = GeoFunctions.getPolygonSansHolesCentroid(sansHoles);
            double area = sansHoles.calculateArea2D();
            xSum += centroid.getX() * area;
            ySum += centroid.getY() * area;
            areaSum += area;
        }
        return new Point(xSum / areaSum, ySum / areaSum);
    }

    private static Polygon getSubPolygon(Polygon polygon, int startIndex, int endIndex) {
        Polyline boundary = new Polyline();
        boundary.startPath(polygon.getPoint(startIndex));
        for (int i = startIndex + 1; i < endIndex; ++i) {
            Point current = polygon.getPoint(i);
            boundary.lineTo(current);
        }
        Polygon newPolygon = new Polygon();
        newPolygon.add((MultiPath)boundary, false);
        return newPolygon;
    }

    private static Point getPolygonSansHolesCentroid(Polygon polygon) {
        int pointCount = polygon.getPointCount();
        double xSum = 0.0;
        double ySum = 0.0;
        double signedArea = 0.0;
        for (int i = 0; i < pointCount; ++i) {
            Point current = polygon.getPoint(i);
            Point next = polygon.getPoint((i + 1) % polygon.getPointCount());
            double ladder = current.getX() * next.getY() - next.getX() * current.getY();
            xSum += (current.getX() + next.getX()) * ladder;
            ySum += (current.getY() + next.getY()) * ladder;
            signedArea += ladder / 2.0;
        }
        return new Point(xSum / (signedArea * 6.0), ySum / (signedArea * 6.0));
    }

    private static Point computeMultiPolygonCentroid(OGCMultiPolygon multiPolygon) {
        double xSum = 0.0;
        double ySum = 0.0;
        double weightSum = 0.0;
        for (int i = 0; i < multiPolygon.numGeometries(); ++i) {
            Point centroid = GeoFunctions.computePolygonCentroid((Polygon)multiPolygon.geometryN(i).getEsriGeometry());
            Polygon polygon = (Polygon)multiPolygon.geometryN(i).getEsriGeometry();
            double weight = polygon.calculateArea2D();
            weightSum += weight;
            xSum += centroid.getX() * weight;
            ySum += centroid.getY() * weight;
        }
        return new Point(xSum / weightSum, ySum / weightSum);
    }

    private static boolean envelopes(Slice left, Slice right, EnvelopesPredicate predicate) {
        Envelope leftEnvelope = GeometrySerde.deserializeEnvelope((Slice)left);
        Envelope rightEnvelope = GeometrySerde.deserializeEnvelope((Slice)right);
        if (leftEnvelope == null || rightEnvelope == null) {
            return false;
        }
        return predicate.apply(leftEnvelope, rightEnvelope);
    }

    private static interface EnvelopesPredicate {
        public boolean apply(Envelope var1, Envelope var2);
    }
}

